#ifndef _EXPRESSION_H
#define _EXPRESSION_H

#include <math.h>
#include <qdict.h>
#include <qpaintd.h>
#include <qstring.h>


typedef struct {
  uchar numParameters;

  union {
    double (*oneParam)( double );
    double (*twoParams)( double, double );
    double (*threeParams)( double, double, double );
  };
} funcCall;


extern QDict<double>*   dictConstants;
extern QDict<funcCall>* dictFunctions;

const int err_Ok                      =  0;
const int err_TooManyClosingBrackets  = -1;
const int err_ClosingBracketMissing   = -2;
const int err_funcCallUnknown         = -3;
const int err_wrongNumberOfParameters = -4;
const int err_constantUnknown         = -5;
const int err_notANumber              = -6;
const int err_expressionEmpty         = -7;


class KExpression;
class KNode;
class KNodePlus;
class KNodeMinus;
class KNodeMult;
class KNodeDiv;
class KNodePow;
class KNodeX;


int parse( QString expression, KNode*& exp );


class KNode
{
public:
  KNode* left, * right;
  KNode();
  virtual inline double getValue( double x ) const;
  virtual ~KNode();
}; // class KNode


class KNodePlus : public KNode
{
public:
  virtual inline double getValue( double x ) const
    { return left->getValue( x ) + right->getValue( x ); };
}; // class KNodePlus

class KNodeMinus : public KNode
{
public:
  virtual inline double getValue( double x ) const
    { return left->getValue( x ) - right->getValue( x ); };
}; // class KNodeMinus

class KNodeMult : public KNode
{
public:
  virtual inline double getValue( double x ) const
    { return left->getValue( x ) * right->getValue( x ); };
}; // class KNodeMinus

class KNodeDiv : public KNode
{
public:
  virtual inline double getValue( double x ) const
    { return left->getValue( x ) / right->getValue( x ); };
}; // class KNodeMinus

class KNodePow : public KNode
{
public:
  virtual inline double getValue( double x ) const
    { return pow( left->getValue( x ), right->getValue( x ) ); };
}; // class KNodePow

class KNodeValue : public KNode
{
public:
  KNodeValue( double val = 0.0 ) : KNode() { value = val; };
  virtual inline double getValue( double ) const
    { return value; };

protected:
  double value;
}; // class KNodeMinus

class KNodeX : public KNode
{
public:
  virtual inline double getValue( double x ) const
    { return x; };
}; // class KNodeX

class KNodeFuncCall : public KNode
{
public:
  virtual double getValue( double x ) const;

  funcCall  function;
  KNode* param[3];
}; // class KNodeFuncCall


class KExpression
{
public:
  KExpression();
  KExpression( QString exp );
 ~KExpression();

  int parse( QString exp );

  inline double getValue( double x ) const
    { return expression->getValue( x ); };

  void plot( QPaintDevice* pd, QPainter* p,
	     QPen pen, double minX, double minY, double maxX, double maxY );

protected:
  KNode* expression;
}; // class KRootExpression

#endif // _EXPRESSION_H
