/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __BLOCK_HPP__
#define __BLOCK_HPP__

#include <string.h>

#include "globals.hpp"

#define MAXBLOCKSIZE	32

class SCL_API CBlockKey
{
	public:
		virtual Word GetKeySize() = 0;
};

class SCL_API CBlockObject
{
	public:
		virtual void Encrypt(const CBlockKey &) = 0;
		virtual void Decrypt(const CBlockKey &) = 0;
		virtual Word GetBlockSize() = 0;
		virtual Byte *GetData() = 0;
		virtual void SetData(const Byte *, Word) = 0;
};

class SCL_API CECBCryptographer
{
	public:
		CECBCryptographer(CBlockObject &, CBlockKey &);
		virtual ~CECBCryptographer();
	
	public:
		Word EncryptData(Byte *, Word, bool = true, bool = true);
		Word DecryptData(Byte *, Word, bool = true);
		Word EncryptData(const Byte *, Byte *, Word, bool = true, bool = true);
		Word DecryptData(const Byte *, Byte *, Word, bool = true);

	public:
		const Byte *GetLastBlock() const { return bLastBlock; };
	
	private:
		CBlockObject &cBlockObject;
		CBlockKey &cBlockKey;
		Byte bLastBlock[MAXBLOCKSIZE];
};

class SCL_API CECBCTSCryptographer
{
	public:
		CECBCTSCryptographer(CBlockObject &, CBlockKey &);
		virtual ~CECBCTSCryptographer();
	
	public:
		Word EncryptData(Byte *pbData, Word wLength) { return EncryptData(pbData, pbData, wLength); };
		Word DecryptData(Byte *pbData, Word wLength) { return DecryptData(pbData, pbData, wLength); };
		Word EncryptData(const Byte *, Byte *, Word);
		Word DecryptData(const Byte *, Byte *, Word);

	private:
		CBlockObject &cBlockObject;
		CBlockKey &cBlockKey;
		Byte bLastBlock[MAXBLOCKSIZE];
};

class SCL_API CCBCCryptographer
{
	public:
		CCBCCryptographer(CBlockObject &, CBlockKey &, const Byte *);
		virtual ~CCBCCryptographer();
	
	public:
		Word EncryptData(Byte *, Word, bool = true, bool = true);
		Word DecryptData(Byte *, Word, bool = true);
		Word EncryptData(const Byte *, Byte *, Word, bool = true, bool = true);
		Word DecryptData(const Byte *, Byte *, Word, bool = true);

	public:
		const Byte *GetLastBlock() const { return bLastBlock; };
		const Byte *GetIV() const { return bIV; };
		void SetIV(const Byte *);
	
	private:
		CBlockObject &cBlockObject;
		CBlockKey &cBlockKey;
		Byte bBuffer[MAXBLOCKSIZE];
		Byte bLastBlock[MAXBLOCKSIZE];
		Byte bFeedBack[MAXBLOCKSIZE * 2];
		Byte bIV[MAXBLOCKSIZE];
		Word wBlockSize;
};

class SCL_API CCBCCTSCryptographer
{
	public:
		CCBCCTSCryptographer(CBlockObject &, CBlockKey &, const Byte *);
		virtual ~CCBCCTSCryptographer();
	
	public:
		Word EncryptData(Byte *pbData, Word wLength) { return EncryptData(pbData, pbData, wLength); };
		Word DecryptData(Byte *pbData, Word wLength) { return DecryptData(pbData, pbData, wLength); };
		Word EncryptData(const Byte *, Byte *, Word);
		Word DecryptData(const Byte *, Byte *, Word);

	public:
		const Byte *GetLastBlock() const { return bLastBlock; };
		const Byte *GetIV() const { return bIV; };
		void SetIV(const Byte *);
	
	private:
		CBlockObject &cBlockObject;
		CBlockKey &cBlockKey;
		Byte bBuffer[MAXBLOCKSIZE];
		Byte bLastBlock[MAXBLOCKSIZE];
		Byte bFeedBack[MAXBLOCKSIZE * 2];
		Byte bIV[MAXBLOCKSIZE];
		Word wBlockSize;
};

class SCL_API CFBCryptographer
{
	public:
		CFBCryptographer(CBlockObject &, CBlockKey &, const Byte *, Word = 0);
		virtual ~CFBCryptographer();
	
	public:
		virtual Word EncryptData(const Byte *, Byte *, Word) = 0;
		virtual Word DecryptData(const Byte *, Byte *, Word) = 0;

	public:
		const Byte *GetIV() const { return bIV; };
		void SetIV(const Byte *);
	
	protected:
		CBlockObject &cBlockObject;
		CBlockKey &cBlockKey;
		Byte bIV[MAXBLOCKSIZE];
		Byte bRegister[MAXBLOCKSIZE];
		Byte bBuffer[MAXBLOCKSIZE];
		Word wFeedBackSize;
		Word wBlockSize;
		Word wFeedBackPointer;
};

class SCL_API CCFBCryptographer : public CFBCryptographer
{
	public:
		CCFBCryptographer(CBlockObject &, CBlockKey &, const Byte *, Word = 0);
	
	public:
		Word EncryptData(Byte *pbData, Word wLength) { return EncryptData(pbData, pbData, wLength); };
		Word DecryptData(Byte *pbData, Word wLength) { return DecryptData(pbData, pbData, wLength); };
		virtual Word EncryptData(const Byte *, Byte *, Word);
		virtual Word DecryptData(const Byte *, Byte *, Word);
};

class SCL_API COFBCryptographer : public CFBCryptographer
{
	public:
		COFBCryptographer(CBlockObject &, CBlockKey &, const Byte *, Word = 0);
	
	public:
		Word EncryptData(Byte *pbData, Word wLength) { return EncryptData(pbData, pbData, wLength); };
		Word DecryptData(Byte *pbData, Word wLength) { return DecryptData(pbData, pbData, wLength); };
		virtual Word EncryptData(const Byte *, Byte *, Word);
		virtual Word DecryptData(const Byte *, Byte *, Word);
};

class SCL_API COFBCCryptographer : public CFBCryptographer
{
	public:
		COFBCCryptographer(CBlockObject &, CBlockKey &, const Byte *, Word = 0);
	
	public:
		Word EncryptData(Byte *pbData, Word wLength) { return EncryptData(pbData, pbData, wLength); };
		Word DecryptData(Byte *pbData, Word wLength) { return DecryptData(pbData, pbData, wLength); };
		virtual Word EncryptData(const Byte *, Byte *, Word);
		virtual Word DecryptData(const Byte *, Byte *, Word);
};

#endif // __BLOCK_HPP__
