/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __TIMERS_HPP__
#define __TIMERS_HPP__

#include "config.h"
#include "globals.hpp"

#ifdef WIN32

#include <sys/timeb.h>
typedef struct timeb timetype;
#define gettime(s) (void)ftime(s)
#define tickdiff(s, t) (((s).time - (t).time) * DWORDCONST(1000) + (s).millitm - (t).millitm)
#define getdwtime(s) ((s).time * DWORDCONST(1000) + (s).millitm)
#define MINTICK 0

#else

#include <time.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TIMEB_H
#include <sys/timeb.h>
#endif
#include <sys/types.h>
#include <sys/times.h>

#ifdef HAVE_GETHRTIME
typedef hrtime_t timetype;
#define gettime(s) (*(s) = gethrtime())
#define tickdiff(s, t) ((s) - (t))
#define MINTICK 0
#elif HAVE_CLOCK_GETTIME
#define CHOICE_CLOCK_GETTIME 1
typedef struct timespec timetype;
#define gettime(s) (void)clock_gettime(CLOCK_REALTIME, s)
#define tickdiff(s, t) (((s).tv_sec - (t).tv_sec) * DWORDCONST(1000000000) + (s).tv_nsec - (t).tv_nsec)
#define getdwtime(s) ((s).tv_sec * DWORDCONST(1000000000) + (s).tv_nsec)
#elif HAVE_GETTIMEOFDAY
typedef struct timeval timetype;
#define gettime(s) (void)gettimeofday(s, (struct timezone *)0)
#define tickdiff(s, t) (((s).tv_sec - (t).tv_sec) * DWORDCONST(1000000) + (s).tv_usec - (t).tv_usec)
#define getdwtime(s) ((s).tv_sec * DWORDCONST(1000000) + (s).tv_usec)
#define MINTICK 1
#elif HAVE_GETITIMER
#include <signal.h>
typedef struct itimerval timetype;
#define gettime(s) (void)getitimer(ITIMER_REAL, s)
#define starttimer(s) (s).it_value.tv_sec = 1000000000; 	\
							 (s).it_interval.tv_sec = 1000000000;	\
							 (s).it_interval.tv_usec = 0;				\
							 setitimer(ITIMER_REAL, &(s), NULL)
#define tickdiff(s, t) (((t).it_value.tv_sec - (s).it_value.tv_sec) * DWORDCONST(1000000) + (t).it_value.tv_usec - (s).it_value.tv_usec)
#define getdwtime(s) ((s).it_value.tv_sec * DWORDCONST(1000000) + (s).it_value.tv_usec)
#define MINTICK 1
#elif HAVE_FTIME
#include <sys/timeb.h>
typedef struct timeb timetype;
#define gettime(s) (void)ftime(s)
#define tickdiff(s, t) (((s).time - (t).time) * DWORDCONST(1000) + (s).millitm - (t).millitm)
#define getdwtime(s) ((s).time * DWORDCONST(1000) + (s).millitm)
#define MINTICK 0
#else
#error "No clock available"
#endif

#endif // WIN32

#endif // __TIMERS_HPP__
