/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "elgamal.hpp"

CElGamalKey::CElGamalKey() : fgHoldParams(false), fgHoldKey(false)
{
}

CElGamalKey::CElGamalKey(Word wPrimeSize) : fgHoldParams(false)
{
	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_ELGAMAL_PRIMESIZE);
	CElGamalKey::wPrimeSize = wPrimeSize;
	GenerateKeys();
}

CElGamalKey::CElGamalKey(const CProbablePrime &cPrime,
								 const CProbablePrime &cGenerator,
								 const CBigNumber &cPublic,
								 const CBigNumber &cPrivate)
{
	wPrimeSize = cPrime.GetWords() * BITSINWORD;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_ELGAMAL_PRIMESIZE);

	CElGamalKey::cPrime = cPrime;
	CElGamalKey::cGenerator = cGenerator;
	CElGamalKey::cPublic = cPublic;
	CElGamalKey::cPrivate = cPrivate;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = false;
}

CElGamalKey::CElGamalKey(const CProbablePrime &cPrime,
								 const CProbablePrime &cGenerator,
								 const CBigNumber &cPublic)
{
	wPrimeSize = cPrime.GetWords() * BITSINWORD;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_ELGAMAL_PRIMESIZE);

	CElGamalKey::cPrime = cPrime;
	CElGamalKey::cGenerator = cGenerator;
	CElGamalKey::cPublic = cPublic;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = true;
}

CElGamalKey::CElGamalKey(const CElGamalKey &cElGamalKey)
{
	wPrimeSize = cElGamalKey.wPrimeSize;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_ELGAMAL_PRIMESIZE);

	cPrime = cElGamalKey.cPrime;
	cGenerator = cElGamalKey.cGenerator;
	cPublic = cElGamalKey.cPublic;
	cPrivate = cElGamalKey.cPrivate;

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = false;
}

void CElGamalKey::GenerateKeys()
{
	if (false == fgHoldParams) {
		cGenerator = 5;

		cPrime.SetRandom(wPrimeSize, true);
		Word i = cPrime % 10;
		while ((i != 3) && (i != 7)) {
			cPrime += 2;
			i = cPrime % 10;
		}
		do {
			cPrime += 10;
		} while (!cPrime.IsPrime());

		fgHoldParams = true;
	}

	cPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);
	cPrivate.Mod(cPrime - 1);

	cPublic = CBigNumber::ModExp(cGenerator, cPrivate, cPrime);

	fgHoldKey = true;
	fgEncryptOnly = false;
}

void CElGamalKey::WritePrivateKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cVersion((Word)ELGAMAL_PRIVATE_KEY_VERSION);
	CDEREncodedBigNumber cPrime(CElGamalKey::cPrime);
	CDEREncodedBigNumber cGenerator(CElGamalKey::cGenerator);
	CDEREncodedBigNumber cPublic(CElGamalKey::cPublic);
	CDEREncodedBigNumber cPrivate(CElGamalKey::cPrivate);

	CDEREncodedSequence cSequence;

	cSequence.AddPrimitive(cVersion);
	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cGenerator);
	cSequence.AddPrimitive(cPublic);
	cSequence.AddPrimitive(cPrivate);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_ELGAMAL_PRIVATE_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_ELGAMAL_PRIVATE_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CElGamalKey::WritePublicKey(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cPrime(CElGamalKey::cPrime);
	CDEREncodedBigNumber cGenerator(CElGamalKey::cGenerator);
	CDEREncodedBigNumber cPublic(CElGamalKey::cPublic);

	CDEREncodedSequence cSequence;

	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cGenerator);
	cSequence.AddPrimitive(cPublic);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_ELGAMAL_PUBLIC_KEY "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_ELGAMAL_PUBLIC_KEY "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CElGamalKey::WriteParams(int iOut, bool fgBase64)
{
	CDEREncodedBigNumber cPrime(CElGamalKey::cPrime);
	CDEREncodedBigNumber cGenerator(CElGamalKey::cGenerator);

	CDEREncodedSequence cSequence;
	cSequence.AddPrimitive(cPrime);
	cSequence.AddPrimitive(cGenerator);

	if (true == fgBase64) {
		write_string(iOut, BEGIN_ELGAMAL_PARAMS "\n");
		cSequence.WriteBase64(iOut, true);
		write_string(iOut, "\n" END_ELGAMAL_PARAMS "\n");
	} else {
		cSequence.Write(iOut);
	}
}

void CElGamalKey::WriteParams(char *pszFile, bool fgBase64)
{
	int iFile = 1;
	if (NULL != pszFile)
		if (-1 == (iFile = open(pszFile, O_RDWR | O_CREAT, 0644)))
		  throw OPEN_ERROR;
	WriteParams(iFile, fgBase64);
	if (NULL != pszFile)
		close(iFile);
}

void CElGamalKey::ReadPrivateKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_ELGAMAL_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_ELGAMAL_PRIVATE_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cVersion(cSequence);
	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);
	CDEREncodedBigNumber cPublic(cSequence);
	CDEREncodedBigNumber cPrivate(cSequence);

	CElGamalKey::cPrime = cPrime;
	CElGamalKey::cGenerator = cGenerator;
	CElGamalKey::cPublic = cPublic;
	CElGamalKey::cPrivate = cPrivate;

	wPrimeSize = cPrime.GetWords() * BITSINWORD;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_ELGAMAL_PRIMESIZE);

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = false;
}

void CElGamalKey::ReadPublicKey(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_ELGAMAL_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_ELGAMAL_PUBLIC_KEY "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);
	CDEREncodedBigNumber cPublic(cSequence);

	CElGamalKey::cPrime = cPrime;
	CElGamalKey::cGenerator = cGenerator;
	CElGamalKey::cPublic = cPublic;

	wPrimeSize = cPrime.GetWords() * BITSINWORD;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_ELGAMAL_PRIMESIZE);

	fgHoldParams = true;
	fgHoldKey = true;
	fgEncryptOnly = true;
}

void CElGamalKey::ReadParams(int iIn, bool fgBase64)
{
	CDEREncodedSequence cSequence;
	if (true == fgBase64) {
		if (false == match_string(iIn, BEGIN_ELGAMAL_PARAMS "\n"))
			throw(KEYFILE_ERROR);
		cSequence.ReadBase64(iIn);
		if (false == match_string(iIn, "\n" END_ELGAMAL_PARAMS "\n"))
			throw(KEYFILE_ERROR);
	} else {
		cSequence.Read(iIn);
	}

	CDEREncodedBigNumber cPrime(cSequence);
	CDEREncodedBigNumber cGenerator(cSequence);

	wPrimeSize = cPrime.GetWords() * BITSINWORD;

	if ((wPrimeSize < 512) || (wPrimeSize > 3072))
		throw(BAD_ELGAMAL_PRIMESIZE);

	CElGamalKey::cPrime = cPrime;
	CElGamalKey::cGenerator = cGenerator;

	fgHoldParams = true;
}

void CElGamalKey::ReadParams(char *pszFile, bool fgBase64)
{
	int iFile = 0;
	if (NULL != pszFile)
		if (-1 == (iFile = open(pszFile, O_RDONLY)))
		  throw OPEN_ERROR;
	ReadParams(iFile, fgBase64);
	if (NULL != pszFile)
		close(iFile);
}

Word CElGamalKey::Check()
{
	if (!fgHoldKey || fgEncryptOnly)
		throw(BAD_ELGAMAL_OPERATION);

	Word wResult = ELGAMAL_OK;

	if (!cPrime.IsPrime())
		wResult |= ELGAMAL_PNOTPRIME;
	if (cPublic != CBigNumber::ModExp(cGenerator, cPrivate, cPrime))
		wResult |= ELGAMAL_BADPUBLIC;

	return wResult;
}

void CElGamalKey::Dump()
{
	printf("prime = "); cPrime.Dump();
	printf("generator = "); cGenerator.Dump();
	printf("public = "); cPublic.Dump();
	printf("private = "); cPrivate.Dump();
}

CElGamalBlock::CElGamalBlock(const CElGamalKey &cElGamalKey, void *pvData, Word wData)
{
	cKey = cElGamalKey;
	c = CBigNumber(pvData, wData);
}

CElGamalBlock::CElGamalBlock(const CElGamalKey &cElGamalKey, const CBigNumber &e)
{
	cKey = cElGamalKey;
	c = e;
}

CElGamalBlock::CElGamalBlock(const CElGamalKey &cElGamalKey, const CBigNumber &e, const CBigNumber &f)
{
	cKey = cElGamalKey;
	c = e;										// We will decrypt the ciphertext initially in e and f
	d = f;
}

CElGamalBlock::CElGamalBlock(const CElGamalKey &cElGamalKey)
{
	cKey = cElGamalKey;
}

void CElGamalBlock::Encrypt()
{
	if (!cKey.HoldKeyFlag())
		throw(BAD_ELGAMAL_OPERATION);

	CBigNumber k;

	k.SetRandom(cKey.GetPrime().GetWords() * BITSINWORD, false);
	k.Mod(cKey.GetPrime() - 1);

	c = (c * CBigNumber::ModExp(cKey.GetPublic(), k, cKey.GetPrime())) % cKey.GetPrime();
	d = CBigNumber::ModExp(cKey.GetGenerator(), k, cKey.GetPrime());
}

void CElGamalBlock::Decrypt()
{
	if (!cKey.HoldKeyFlag() || cKey.EncryptOnlyFlag())
		throw(BAD_ELGAMAL_OPERATION);

	c = (c * CBigNumber::ModExp(d, cKey.GetPrime() - 1 - cKey.GetPrivate(), cKey.GetPrime())) % cKey.GetPrime();
}

void CElGamalBlock::Dump()
{
	printf("c = "); c.Dump();
	printf("d = "); d.Dump();
}

void CElGamalBlock::Write(int iOut)
{
	CDEREncodedBigNumber cData1(c);
	CDEREncodedBigNumber cData2(d);
	cData1.Write(iOut);
	cData2.Write(iOut);
}

void CElGamalBlock::Read(int iIn)
{
	CDEREncodedBigNumber cData1;
	CDEREncodedBigNumber cData2;

	cData1.Read(iIn);
	cData2.Read(iIn);

	c = cData1;
	d = cData2;
}

void CElGamalBlock::SetData(Byte *pbData, Word wDataLength)
{
	c = CBigNumber(pbData, wDataLength);
}

Byte *CElGamalBlock::GetData()
{
	return (Byte *)(c.GetData() + 1);
}

Word CElGamalBlock::GetDataSize()
{
	return c.GetData()[0];
}
