/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "obn.hpp"
#include "pgprp.hpp"

SCL_API CPRNG cPRNG;							// Generator of Pseudo Random Numbers
													// for the generation of random big numbers
// Default constructor. Allocates zero initialized memory for CBigNumber.
CBigNumber::CBigNumber() : wWords(0), fgSign(false)
{
	wAllocated = BLOCK;
	pwBigNumber = (Word *)calloc(BLOCK, sizeof(Word));
	if (NULL == pwBigNumber)
		throw(MEMORYALLOCATION_ERROR);
}

CBigNumber::CBigNumber(Word wValue)
{
	CBigNumber::wWords = 1;
	CBigNumber::wAllocated = BLOCK;
	CBigNumber::fgSign = false;
	
	pwBigNumber = (Word *)calloc(BLOCK, sizeof(Word));
	if (NULL == pwBigNumber)
		throw(MEMORYALLOCATION_ERROR);
	pwBigNumber[0] = wValue;
}

void CBigNumber::SetWords(Word wWords)
{
	Word wOld = wAllocated;
	
	CBigNumber::wWords = wWords;
	if (wWords > wAllocated) {
		wAllocated = (wWords / BLOCK + 1) * BLOCK;
		pwBigNumber = (Word *)realloc(pwBigNumber, wAllocated * sizeof(Word));
		if (NULL == pwBigNumber)
			throw(MEMORYALLOCATION_ERROR);
		for (register Word w = wOld; w < wAllocated; w++)
			pwBigNumber[w] = 0;
	}
}

void CBigNumber::SetRandom(Word wBits, bool fgOdd)
{
	Word i = (wBits + BITSINWORD - 1) / BITSINWORD;
	wWords = i;
	if (wWords > wAllocated) {
		wAllocated = (wWords / BLOCK + 1) * BLOCK;
		pwBigNumber = (Word *)realloc(pwBigNumber, wAllocated * sizeof(Word));
		if (NULL == pwBigNumber)
			throw(MEMORYALLOCATION_ERROR);
	}
	cPRNG.GetRandomData(pwBigNumber, wWords * BYTESINWORD);
	if (fgOdd)
		SetBit(0);
	if (wBits % BITSINWORD != 0)
		pwBigNumber[wWords - 1] &= (MaxWORD >> (BITSINWORD - (wBits % BITSINWORD)));
	SetBit(wBits - 1);
}

CBigNumber::CBigNumber(char *pszHexString) : pwBigNumber(NULL), wWords(0), wAllocated(0), fgSign(false)
{
	char szHex[] = "0123456789ABCDEF";
	
	if ('-' == pszHexString[0]) {
		fgSign = true;
		pszHexString += 1;
	}
	
	int iStringLength = strlen(pszHexString);
	
	wWords = (iStringLength + CHARSINWORD - 1) / CHARSINWORD;
	wAllocated = (wWords / BLOCK + 1) * BLOCK;
	
	pwBigNumber = (Word *)calloc(wAllocated, sizeof(Word));
	
	for (int i = 0; i < iStringLength; i++)
		pwBigNumber[i / CHARSINWORD] |= (Word)(strchr(szHex, toupper(pszHexString[iStringLength - i - 1])) - szHex) << 4 * (i % CHARSINWORD);
	
	StripLeadingZeroes();
}

CBigNumber::CBigNumber(void *pvData, Word wDataLength) : pwBigNumber(NULL), wWords(0), wAllocated(0), fgSign(false)
{
	wWords = (wDataLength * BITSINBYTE + BITSINWORD - 1) / BITSINWORD + 1;
	wAllocated = (wWords / BLOCK + 1) * BLOCK;
	
	pwBigNumber = (Word *)calloc(wAllocated, sizeof(Word));
	
	pwBigNumber[0] = wDataLength;
	for (Word i = 0; i < wDataLength; i++)
#ifdef WORDS_BIGENDIAN
		pwBigNumber[i / BYTESINWORD + 1] |= (Word)(((char *)pvData)[i] << BITSINBYTE * (BYTESINWORD - 1 - i % BYTESINWORD));
#else
		pwBigNumber[i / BYTESINWORD + 1] |= (Word)(((Byte *)pvData)[i] << (BITSINBYTE * (i % BYTESINWORD)));
#endif // WORDS_BIGENDIAN
	
	StripLeadingZeroes();
}

CBigNumber::CBigNumber(Word *pwData, Word wDataLength) : pwBigNumber(NULL), wWords(wDataLength), wAllocated(0), fgSign(false)
{
	wAllocated = (wWords / BLOCK + 1) * BLOCK;
	
	pwBigNumber = (Word *)calloc(wAllocated, sizeof(Word));
	
	memcpy(pwBigNumber, pwData, wWords * sizeof(Word));
	
	StripLeadingZeroes();
}

CBigNumber::CBigNumber(const CBigNumber &cNumber)
{
	fgSign = cNumber.fgSign;
	wAllocated = cNumber.wAllocated;
	wWords = cNumber.wWords;
	if (NULL != cNumber.pwBigNumber) {
		pwBigNumber = (Word *)calloc(wAllocated, sizeof(Word));
		memcpy(pwBigNumber, cNumber.pwBigNumber, wWords * sizeof(Word));
	}
}

CBigNumber::~CBigNumber()
{
	if (NULL != pwBigNumber)
		free(pwBigNumber);
	pwBigNumber = NULL;
}

void CBigNumber::Neg()
{
	fgSign = !fgSign;
}

Word CBigNumber::GetSum()
{
	Word wResult = 0;
	for (Word i = 0; i < wWords; i++)
		wResult += pwBigNumber[i];
	return wResult;
}

void CBigNumber::SetCheckSum()
{
	Word wCheck = GetSum();
	SetWords(wWords + 1);
	pwBigNumber[wWords - 1] = wCheck;
}

bool CBigNumber::CheckSum()
{
	if (wWords >= 1) {
		wWords -= 1;
		Word wCheck = GetSum();
		if (pwBigNumber[wWords] == wCheck)
			return true;
	}
	return false;
}

CBigNumber CBigNumber::operator = (const CBigNumber &cNumber)
{
	if (this != &cNumber) {
		if (NULL != pwBigNumber)
			free(pwBigNumber);
		fgSign = cNumber.fgSign;
		wAllocated = cNumber.wAllocated;
		wWords = cNumber.wWords;
		if (NULL != cNumber.pwBigNumber) {
			pwBigNumber = (Word *)calloc(wAllocated, sizeof(Word));
			memcpy(pwBigNumber, cNumber.pwBigNumber, wWords * sizeof(Word));
		}
	}
	return *this;
}

CBigNumber CBigNumber::operator + (const CBigNumber &cAddend) const
{
	CBigNumber cSum;
	if (fgSign == cAddend.fgSign) {
		cSum = *this;
		cSum.Add(cAddend);
	} else {
		cSum = *this;
		cSum.Sub(cAddend);
	}
	return cSum;
}

CBigNumber & CBigNumber::operator += (const CBigNumber &cAddend)
{
	if (fgSign == cAddend.fgSign)
		Add(cAddend);
	else
		Sub(cAddend);
	return *this;
}

CBigNumber & CBigNumber::operator -= (const CBigNumber &cSubtrahend)
{
	if (fgSign == cSubtrahend.fgSign)
		Sub(cSubtrahend);
	else
		Add(cSubtrahend);
	return *this;
}

void CBigNumber::Add(const CBigNumber &cAddend)
{
	SetWords(Max(wWords, cAddend.wWords));

	register Word i;
	register Word j;
	register Word c = 0;
	
	Word *q = cAddend.pwBigNumber;
	Word *r = pwBigNumber;
	
	for (i = 0; i < cAddend.wWords; i++) {
		j = HiWORD((Dword)q[i] + (Dword)r[i] + (Dword)c);
		r[i] += (q[i] + c);
		c = j;
	}
	for (; i < wWords; i++) {
		j = HiWORD((Dword)r[i] + (Dword)c);
		r[i] += c;
		c = j;
	}
	if (c != 0) {
		SetWords(wWords + 1);
		pwBigNumber[i] = c;
	}
	
	StripLeadingZeroes();	
}

CBigNumber CBigNumber::operator - (const CBigNumber &cSubtrahend) const
{
	CBigNumber cDifference;
	if (fgSign == cSubtrahend.fgSign) {
		cDifference = *this;
		cDifference.Sub(cSubtrahend);
	} else {
		cDifference = *this;
		cDifference.Add(cSubtrahend);
	}
	return cDifference;
}

// Calculated the result of abs(cMinuend) /* this */ - abs(cSubtrahend)
// cMinuend is expected to be >= cMinuend
void CBigNumber::NormalSub(const CBigNumber &cSubtrahend)
{
	register Word d;
	register Word i;
	register Word b = 0;						// borrow

	register Word *q = cSubtrahend.pwBigNumber;
	register Word *r = pwBigNumber;	
	for (i = 0; i < cSubtrahend.wWords; i++) {
		d = r[i] < q[i] + b;
		r[i] -= (q[i] + b);
		b = d;
	}
	for (; i < wWords && b; i++) {
		d = r[i] < b;
		r[i] -= b;
		b = d;
	}
	
	StripLeadingZeroes();
}

void CBigNumber::Sub(const CBigNumber &cSubtrahend)
{
	register Word d;
	register Word i;
	register Word b = 0;						// borrow

	Word *q;
	Word *r;
	
	if (Cmp(*this, cSubtrahend) == -1) {
		SetWords(cSubtrahend.wWords);
		  
		q = cSubtrahend.pwBigNumber;
		r = pwBigNumber;	
		for (i = 0; i < wWords; i++) {
			d = q[i] < r[i] + b;
			r[i] = q[i] - r[i] - b;
			b = d;
		}
		for (; i < cSubtrahend.wWords; i++) {
			d = q[i] < b;
			r[i] = q[i] - b;
			b = d;
		}
		
		fgSign = !fgSign;
	} else {
		q = cSubtrahend.pwBigNumber;
		r = pwBigNumber;	
		for (i = 0; i < cSubtrahend.wWords; i++) {
			d = r[i] < q[i] + b;
			r[i] -= (q[i] + b);
			b = d;
		}
		for (; i < wWords; i++) {
			d = r[i] < b;
			r[i] -= b;
			b = d;
		}
	}
	
	StripLeadingZeroes();
}

CBigNumber CBigNumber::operator * (Word wMultiplier) const
{
	CBigNumber cProduct;
	if (IsZero() || (0 == wMultiplier))
		return cProduct;

	cProduct.SetWords(wWords);

	Dword b = 0;
	Word w;
	for (w = 0; w < wWords; w++) {
		b = pwBigNumber[w] * (Dword)wMultiplier + HiWORD(b);
		cProduct.pwBigNumber[w] = LoWORD(b);
	}
	if (0 != HiWORD(b)) {
		cProduct.SetWords(wWords + 1);
		cProduct.pwBigNumber[w] = HiWORD(b);
	}

	return cProduct;
}

CBigNumber CBigNumber::operator * (const CBigNumber &cMultiplier) const
{
	CBigNumber cProduct;
	if (IsZero() || cMultiplier.IsZero())
		return cProduct;
	cProduct.SetWords(wWords + cMultiplier.wWords);
	
	Word *p = pwBigNumber;
	Word *q = cMultiplier.pwBigNumber;
	Word *r = cProduct.pwBigNumber;
	
	Word i, j, k;
	register Dword t;
	register Dword m;
	for (i = 0; i < wWords; i++) {
		m = p[i];
// This is the critical loop		
		for (j = i, k = 0; j < i + cMultiplier.wWords; j++) {
			t = m * q[j - i] + r[j] + k;
			r[j] = LoWORD(t);
			k = HiWORD(t);
		}
		r[j] = k;
	}
	
	if (fgSign != cMultiplier.fgSign)
		cProduct.fgSign = true;
	
	cProduct.StripLeadingZeroes();
	return cProduct;
}

CBigNumber CBigNumber::operator / (const CBigNumber &cDivisor) const
{
	CBigNumber cQuotient = *this;
	if (wWords < cDivisor.wWords)
		return CBigNumber((Word)0);
	cQuotient.Div(cDivisor);
	return cQuotient;
}

CBigNumber CBigNumber::operator / (Word wDivisor) const
{
	CBigNumber cQuotient;
	if (!IsZero())
		cQuotient.Div(wDivisor);
	
	return cQuotient;
}

CBigNumber CBigNumber::operator % (const CBigNumber &cDivisor) const
{
	CBigNumber cRemainder = *this;
	if (wWords >= cDivisor.wWords)
		cRemainder.Mod(cDivisor);
	return cRemainder;
}

Word CBigNumber::operator % (Word wDivisor) const
{
	if (IsZero())
		return 0;
	return Mod(wDivisor);
}

CBigNumber CBigNumber::operator << (Word wAmount) const
{
	CBigNumber cResult = *this;
	cResult.Shl(wAmount);
	return cResult;
}

CBigNumber & CBigNumber::operator <<= (Word wAmount)
{
	Shl(wAmount);
	return *this;
}

void CBigNumber::Shl(Word wAmount)
{
	if (0 == wWords)
		return;
	Word wBits = wAmount % BITSINWORD;
	Word wWord = wAmount / BITSINWORD;
	Word i, q = BITSINWORD - wBits;
	
	if ((pwBigNumber[wWords - 1] >> q) != 0)
		SetWords(wWords + 1);
	for (i = wWords - 1; i > 0; i--) {
		pwBigNumber[i] <<= wBits;
		pwBigNumber[i] |= (pwBigNumber[i - 1] >> q);
	}
	pwBigNumber[0] <<= wBits;
	if (wWord != 0) {
		SetWords(wWords + wWord);
		for (i = wWords - 1; i - wWord < wWords; i--)
			pwBigNumber[i] = pwBigNumber[i - wWord];
		for (; i < wWords; i--)
			pwBigNumber[i] = 0;
	}
}

void CBigNumber::Shl()
{
	if (0 == wWords)
		return;
	
	if ((pwBigNumber[wWords - 1] & (1 << (BITSINWORD - 1))) != 0)
		SetWords(wWords + 1);
	Word *p = pwBigNumber;	
	for (register Word i = wWords - 1; i > 0; i--) {
		p[i] <<= 1;
		p[i] |= (p[i - 1] >> (BITSINWORD - 1));
	}
	p[0] <<= 1;
}

void CBigNumber::Shr(Word wAmount)
{
	if (0 == wWords)
		return;

	Word wBits = wAmount % BITSINWORD;
	Word wWord = wAmount / BITSINWORD;
	Word i, t = BITSINWORD - wBits;

	wWords -= wWord;
	for (i = 0; i < wWords; i++)
		pwBigNumber[i] = pwBigNumber[i + wWord];
	for (i = 0; i < wWords - 1; i++)
		pwBigNumber[i] = (pwBigNumber[i] >> wBits) | (pwBigNumber[i + 1] << t);
	pwBigNumber[wWords - 1] >>= wBits;
	StripLeadingZeroes();
}

void CBigNumber::Shr()
{
	if (0 == wWords)
		return;

	for (Word i = 0; i < wWords - 1; i++)
		pwBigNumber[i] = (pwBigNumber[i] >> 1) | (pwBigNumber[i + 1] << (BITSINWORD - 1));
	pwBigNumber[wWords - 1] >>= 1;

	if (0 == pwBigNumber[wWords - 1])
		wWords -= 1;
}

CBigNumber & CBigNumber::operator += (Word wValue)
{
	if (wWords == 0)
		SetWords(1);

	Word c = pwBigNumber[0] + (Dword)wValue > MaxWORD ? 1 : 0;
	pwBigNumber[0] += wValue;
	
	Word i = 1;
	while (0 != c) {
		if (i == wWords)
			SetWords(wWords + 1);
		pwBigNumber[i] += c;
		c = pwBigNumber[i] + (Dword)c > MaxWORD ? 1 : 0;
		i += 1;
	}
	return *this;
}

CBigNumber & CBigNumber::operator -= (Word wValue)
{
	if (pwBigNumber[0] >= wValue) {
		pwBigNumber[0] -= wValue;
	} else {
		pwBigNumber[0] = (Word)(pwBigNumber[0] + BASE - wValue);
		Word i = 1;
		while (0 == pwBigNumber[i])
			pwBigNumber[i++] -= 1;
		pwBigNumber[i] -= 1;
	}
	StripLeadingZeroes();
	return *this;
}

CBigNumber CBigNumber::operator >> (Word wAmount) const
{
	CBigNumber cResult = *this;
	cResult.Shr(wAmount);
	return cResult;
}

CBigNumber & CBigNumber::operator >>= (Word wAmount)
{
	Shr(wAmount);
	return *this;
}

bool CBigNumber::operator < (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return true;
	if (!fgSign && cRightValue.fgSign)
		return false;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return iCmp == 1;
	
	return iCmp == -1;
}

bool CBigNumber::operator <= (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return true;
	if (!fgSign && cRightValue.fgSign)
		return false;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return (iCmp == 1) || (iCmp == 0);
	
	return (iCmp == -1) || (iCmp == 0);
}

bool CBigNumber::operator > (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return false;
	if (!fgSign && cRightValue.fgSign)
		return true;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return iCmp == -1;
	
	return iCmp == 1;
}

bool CBigNumber::operator >= (const CBigNumber &cRightValue) const
{
	if (fgSign && !cRightValue.fgSign)
		return false;
	if (!fgSign && cRightValue.fgSign)
		return true;
	
	int iCmp = Cmp(*this, cRightValue);
	
	if (fgSign && cRightValue.fgSign)
		return (iCmp == -1) || (iCmp == 0);
	
	return (iCmp == 1) || (iCmp == 0);
}

bool CBigNumber::operator == (const CBigNumber &cRightValue) const
{
	if (fgSign != cRightValue.fgSign)
		return false;
	
	return 0 == Cmp(*this, cRightValue);
}

// Returns 0 if abs(cLeftValue) == abs(cRightValue)
// Returns 1 if abs(cLeftValue) > abs(cRightValue)
// Returns -1 if abs(cLeftValue) < abs(cRightValue)
int CBigNumber::Cmp(const CBigNumber &cLeftValue, const CBigNumber &cRightValue) const
{
	if (cLeftValue.wWords > cRightValue.wWords)
		return 1;
	if (cLeftValue.wWords < cRightValue.wWords)
		return -1;
	if (0 == cLeftValue.wWords)
		return 0;
	
	Word *p = cLeftValue.pwBigNumber;
	Word *q = cRightValue.pwBigNumber;
	
	register Word i;
	register Word j = cLeftValue.wWords;
	for (i = j - 1; i < j; i--) {
		if (p[i] > q[i])
			return 1;
		if (p[i] < q[i])
			return -1;
	}
	return 0;
}

bool CBigNumber::operator != (const CBigNumber &cRightValue) const
{
	return !(*this == cRightValue);
}

bool CBigNumber::operator < (Word wRightValue)
{
	if (IsZero() && (0 != wRightValue))
		return true;
	if (fgSign)									// Negative number is smaller than
		return true;							// any nonnegative word
	if (wWords != 1)
		return false;
	return pwBigNumber[0] < wRightValue;
}

bool CBigNumber::operator <= (Word wRightValue)
{
	if (IsZero())
		return true;
	if (fgSign)									// Negative number is smaller than
		return true;							// any nonnegative word
	if (wWords != 1)
		return false;
	return pwBigNumber[0] <= wRightValue;
}

bool CBigNumber::operator > (Word wRightValue)
{
	if (IsZero() && (0 != wRightValue))
		return false;
	if (fgSign)
		return false;
	if (wWords != 1)
		return false;
	return pwBigNumber[0] > wRightValue;
}

bool CBigNumber::operator >= (Word wRightValue)
{
	if (IsZero() && (0 == wRightValue))
		return true;
	if (fgSign)
		return false;
	if (wWords != 1)
		return false;
	return pwBigNumber[0] >= wRightValue;
}

bool CBigNumber::operator == (Word wRightValue)
{
	if (IsZero() && (0 == wRightValue))
		return true;
	if (wWords != 1)
		return false;
	if (fgSign)
		return false;
	return pwBigNumber[0] == wRightValue;
}

bool CBigNumber::operator != (Word wRightValue)
{
	return !(*this == wRightValue);
}

/*
CBigNumber CBigNumber::ModExp(const CBigNumber &b, const CBigNumber &c, const CBigNumber &d)
{
	CBigNumber cResult(1);
	
	if (c.IsZero())
		return cResult;
	
	CBigNumber cTemp = b;

	if (c.pwBigNumber[0] & 1)
		cResult = b;

	for (Word z = 0; z < c.wWords; z++) {
		Word u = c.pwBigNumber[z];
		Word v = 0;
		Word w = BITSINWORD - 1;
		if (0 == z) {
			u >>= 1;
			v = 1;
		}
		if (c.wWords - 1 == z) {
			for (; (w != 0) && (0 == ((u >> (w - 1)) & 1)); w--);
			w += 1;
		}
		for (Word y = v; y <= w; y++) {
			cTemp.Sqr();
			cTemp.Mod(d);
			if (u & 1) {
				cResult = cResult * cTemp;
				cResult.Mod(d);
			}
			u >>= 1;
		}
	}

	return cResult;
}
*/

CBigNumber CBigNumber::ModExp(const CBigNumber &b, const CBigNumber &c, const CBigNumber &d)
{
	CBigNumber cResult(1);
	
	CBigNumber cCache[3];
	
	cCache[0] = b;
	cCache[2] = b;
	cCache[0].Mod(d);
	cCache[1] = cCache[0];
	cCache[1].Sqr();
	cCache[1].Mod(d);
	cCache[2] = cCache[1] * cCache[0];
	cCache[2].Mod(d);
	
	for (Word z = c.wWords - 1; z < c.wWords; z--) {
		Word w = 32;
		Word u = c.pwBigNumber[z];
		if (z == c.wWords - 1) {
			for (Word i = 0; (i < 32) && (0 == ((u >> (31 - i)) & 1)); i++, w--);
			w = ((w + 1) / 2) * 2;
			u <<= (32 - w);
		}
		for (Word y = 0; y < w; y += 2) {
			Word g = ((u >> 30) & 3);
			
			cResult.ModExpWord(4, d);
			if (0 != g)
    			cResult = (cResult * cCache[g - 1]) % d;
			u <<= 2;
		}
	}

	return cResult;
}

// This function is designed for c <= 4 && c >= 1
void CBigNumber::ModExpWord(Word c, const CBigNumber &cModulus)
{
	if (c == 3) {
		CBigNumber cCopy = *this;
		Mod(cModulus);
		Sqr();
		Mod(cModulus);
		*this = *this * cCopy;
	}
	Mod(cModulus);
	if (c == 2) {
		Sqr();
		Mod(cModulus);
	}
	if (c == 4) {
		Sqr();
		Mod(cModulus);
		Sqr();
		Mod(cModulus);
	}
}

CBigNumber CBigNumber::LCM(const CBigNumber &a, const CBigNumber &b)
{
	return a / GCD(a, b) * b;
}

CBigNumber CBigNumber::ModInv(const CBigNumber &a, const CBigNumber &m)
// modular inverse
// returns i in range 1..m-1 such that i * a = 1 mod m
// a must be in range 1..m-1
{
	CBigNumber j(1), i, b = m, c = a, x, y;
	
	while (!c.IsZero()) {
		x = b / c;
		y = b - x * c;
		b = c;
		c = y;
		y = j;
		j = i - j * x;
		i = y;
	} 
	if (i.IsNegative())
		i += m;
	return i;
}

void CBigNumber::StripLeadingZeroes()
{
	register Word i;
	for (i = wWords - 1; i < wWords && 0L == pwBigNumber[i]; i--);
	wWords = i + 1;
}

void CBigNumber::Dump() const
{
	if (fgSign)
		printf("-");
	for (Word i = 0; i < wWords; i++)
		printf("%08x", pwBigNumber[wWords - i - 1]);
	printf("\n");
}

void CBigNumber::Sqr()
{
	if (IsZero())
		return;

	CBigNumber w;

	w.SetWords(2 * wWords);					// If x != 0 then 2 * x.wWords >= 0 for each x
	
	Dword b, c, k;
	Word e = w.wWords;
	
	for (Word i = 0; i < wWords; i++) {
		k = w.pwBigNumber[2 * i] + pwBigNumber[i] * (Dword)pwBigNumber[i];
		if (2 * i == e)
			k += BASE;
		w.pwBigNumber[2 * i] = LoWORD(k);
		c = HiWORD(k);
		for (Word j = i + 1; j < wWords; j++) {
			b = pwBigNumber[j] * (Dword)pwBigNumber[i];
			k = 2 * b + (w.pwBigNumber[i + j] + c + (i + j == e ? BASE : 0));
			c = HiWORD(k);
			if (k < b)
				c += BASE;
			w.pwBigNumber[i + j] = LoWORD(k);
		}
		w.pwBigNumber[i + wWords] = LoWORD(c);
		e = HiWORD(c) ? i + wWords : w.wWords;
	}
	
	w.StripLeadingZeroes();
	
	*this = w;
}

void CBigNumber::Div(CBigNumber y)
{
	if (y.IsZero())
		throw(DIVISIONBYZERO_ERROR);

	if (1 == y.wWords) {
		Div(y.pwBigNumber[0]);
		return;
	}
	if (IsZero()) {
		wWords = 0;
		return;
	}
	CBigNumber cQuotient;
	
	while (y.pwBigNumber[y.wWords - 1] < (BASE / 2)) {				// Normalization
		Shl();																	// Multiply the devident and divisor
		y.Shl();																	// by two until the oldest digit of y
	}																				// is smaller than BASE / 2

	cQuotient.fgSign = (fgSign != y.fgSign);

	fgSign = y.fgSign = false;
	
	Dword c, p, m;
	Word a, b, i, j;
	Word z = wWords - y.wWords, t = y.wWords - 1, n = wWords - 1;

	cQuotient.SetWords(z + 1);												// Step 1
	y.SetWords(y.wWords + z);
	
	register Word *q = cQuotient.pwBigNumber;
	register Word *v = y.pwBigNumber;
	  
	for (i = y.wWords - 1; i - z < y.wWords; i--)					// Step 2
		v[i] = v[i - z];
	for (; i < y.wWords; i--)
		v[i] = 0;
	
	while (*this >= y) {
		q[wWords - 1 - t]++;
		NormalSub(y);
	}																				// Step 2
	
	for (i = n - 1; i >= t; i--) {										// Step 3
		for (j = 0; j + 1 < y.wWords; j++)
			v[j] = v[j + 1];
		if (0 != y.wWords)
			y.wWords -= 1;
		
		Word h = i - t;
		if (v[i] == pwBigNumber[i + 1])									// Step 3.1
			q[h] = MaxWORD; 
		else
			q[h] = (Word)(((Dword)pwBigNumber[i + 1] * BASE + pwBigNumber[i]) / v[i]);

		c = *(Dword *)(pwBigNumber + i);
		p = (Dword)q[h] * v[i - 1];
		m = (Dword)q[h] * v[i] + HiWORD(p);
		if (((m > c) || ((m == c) && (LoWORD(p) > pwBigNumber[i - 1])))) {
			q[h]--;
			p = (Dword)q[h] * v[i - 1];
			m = (Dword)q[h] * v[i] + HiWORD(p);
			if (((m > c) || ((m == c) && (LoWORD(p) > pwBigNumber[i - 1]))))
				q[h]--;
		}
		c = 0;
		b = 0;
		for (j = 0; j < y.wWords; j++) {
			c = v[j] * (Dword)q[h] + HiWORD(c);
			a = pwBigNumber[j] < LoWORD(c) + b;
			pwBigNumber[j] -= (LoWORD(c) + b);
			b = a;
		}
		for ( ; j < wWords; j++) {
			c >>= 32;
			a = pwBigNumber[j] < LoWORD(c) + b;
			pwBigNumber[j] -= (LoWORD(c) + b);
			b = a;
		}
		if (b) {									// borrow ?
			c = 0;
			for (j = 0; j < y.wWords; j++) {
				c = (Dword)pwBigNumber[j] + v[j] + HiWORD(c);
				pwBigNumber[j] = LoWORD(c);
			}
			for ( ; j < wWords; j++) {
				c = (Dword)pwBigNumber[j] + HiWORD(c);
				pwBigNumber[j] = LoWORD(c);
			}
			q[h]--;
		}
		StripLeadingZeroes();
	}
	
	*this = cQuotient;
	StripLeadingZeroes();
}

void CBigNumber::Div(Word wDivisor)
{
	if (0 == wDivisor)
		throw(DIVISIONBYZERO_ERROR);
	
	CBigNumber cQuotient;
	
	cQuotient.SetWords(wWords);
	
	Dword b = 0;
	
	for (Word i = wWords - 1; i < wWords; i--) {
		b = (b << 32) | pwBigNumber[i];
		cQuotient.pwBigNumber[i] = (Word)(b / (Dword)wDivisor);
		b -= (b / (Dword)wDivisor) * (Dword)wDivisor;
	}
	
	cQuotient.StripLeadingZeroes();
	*this = cQuotient;
}

void CBigNumber::Mod(CBigNumber y)
{
	if (y.IsZero())
		throw(DIVISIONBYZERO_ERROR);

	if (wWords < y.wWords)
		return;
	
	CBigNumber cQuotient;
	
	if (1 == y.wWords) {
		pwBigNumber[0] = Mod(y.pwBigNumber[0]);
		wWords = 1;
		StripLeadingZeroes();
		return;
	}

	Word l = 0;
	while ((y.pwBigNumber[y.wWords - 1] << l) < (BASE / 2))		// Normalization
		l += 1;
	if (0 != l) {																// Multiply the devident and divisor
		Shl(l);																	// by two until the oldest digit of y
		y.Shl(l);																// is smaller than BASE / 2
	}

	bool fgSign = CBigNumber::fgSign;

	fgSign = y.fgSign = false;
	
	Dword c, p, m;
	Word i, j, a, b;
	Word z = wWords - y.wWords, t = y.wWords - 1, n = wWords - 1;

	cQuotient.SetWords(z + 1);												// Step 1
	y.SetWords(y.wWords + z);
	
	Word *q = cQuotient.pwBigNumber;
	Word *v = y.pwBigNumber;

	for (i = y.wWords - 1; i - z < y.wWords; i--)					// Step 2
		v[i] = v[i - z];
	for (; i < y.wWords; i--)
		v[i] = 0;

	while (*this >= y) {
		q[wWords - 1 - t]++;
		NormalSub(y);
	}																				// Step 2
	
	for (i = n - 1; i >= t; i--) {										// Step 3
		for (j = 0; j + 1 < y.wWords; j++)
			v[j] = v[j + 1];
		if (0 != y.wWords)
			y.wWords -= 1;
		
		Word h = i - t;
		if (v[i] == pwBigNumber[i + 1])									// Step 3.1
			q[h] = MaxWORD; 
		else
			q[h] = (Word)(((Dword)pwBigNumber[i + 1] * BASE + pwBigNumber[i]) / v[i]);

		c = *(Dword *)(pwBigNumber + i);
		p = (Dword)q[h] * v[i - 1];
		m = (Dword)q[h] * v[i] + HiWORD(p);
		if (((m > c) || ((m == c) && (LoWORD(p) > pwBigNumber[i - 1])))) {
			q[h]--;
			p = (Dword)q[h] * v[i - 1];
			m = (Dword)q[h] * v[i] + HiWORD(p);
			if (((m > c) || ((m == c) && (LoWORD(p) > pwBigNumber[i - 1]))))
				q[h]--;
		}
		c = 0;
		b = 0;
		for (j = 0; j < y.wWords; j++) {
			c = v[j] * (Dword)q[h] + HiWORD(c);
			a = pwBigNumber[j] < LoWORD(c) + b;
			pwBigNumber[j] -= (LoWORD(c) + b);
			b = a;
		}
		for ( ; j < wWords; j++) {
			c >>= 32;
			a = pwBigNumber[j] < LoWORD(c) + b;
			pwBigNumber[j] -= (LoWORD(c) + b);
			b = a;
		}
		if (b) {									// borrow ?
			c = 0;
			for (j = 0; j < y.wWords; j++) {
				c = (Dword)pwBigNumber[j] + v[j] + HiWORD(c);
				pwBigNumber[j] = LoWORD(c);
			}
			for ( ; j < wWords; j++) {
				c = (Dword)pwBigNumber[j] + HiWORD(c);
				pwBigNumber[j] = LoWORD(c);
			}
			q[h]--;
		}
		StripLeadingZeroes();
	}
	
	CBigNumber::fgSign = fgSign;

	if (l)
		Shr(l);																	// Normalization
}

Word CBigNumber::Mod(Word wDivisor) const
{
	if (0 == wDivisor)
		throw(DIVISIONBYZERO_ERROR);

	Dword b = 0;
	for (Word i = wWords - 1; i < wWords; i--) {
		b = b << BITSINWORD | pwBigNumber[i];
		b %= wDivisor;
	}
	
	return (Word)b;
}

int CBigNumber::Jacobi(CBigNumber a, CBigNumber b)
{
	Word i, j;
	int iResult = 1;
	
	a.Mod(b);
	while (!a.IsZero()) {
		for (i = 0; 0 == a.GetBit(i); i++);
		a.Shr(i);
		j = b % 8;
		if (i % 2 == 1 && (j == 3 || j == 5))
			iResult = -iResult;
		if (a % 4 == 3 && b % 4 == 3)
			iResult = -iResult;
		a.Swap(b);
		a.Mod(b);
	}
	return b.IsOne() ? iResult : 0;
}

void CBigNumber::Swap(CBigNumber &cNumber)
{
	bool fgSign = cNumber.fgSign;
	Word wAllocated = cNumber.wAllocated;
	Word wWords = cNumber.wWords;
	Word *pwBigNumber = cNumber.pwBigNumber;

	cNumber.fgSign = CBigNumber::fgSign;
	cNumber.wAllocated = CBigNumber::wAllocated;
	cNumber.wWords = CBigNumber::wWords;
	cNumber.pwBigNumber = CBigNumber::pwBigNumber;

	CBigNumber::fgSign = fgSign;
	CBigNumber::wAllocated = wAllocated;
	CBigNumber::wWords = wWords;
	CBigNumber::pwBigNumber = pwBigNumber;
}

// Returns the floor of the base 2 logarithm for the given CBigNumber
Word CBigNumber::Lg() const
{
	Word wResult = (wWords - 1) * BITSINWORD;
	Dword dwTemp = pwBigNumber[wWords - 1];
	Dword dwTmp = 1;
	while (dwTmp <= dwTemp) {
		wResult += 1;
		dwTmp <<= 1;
	}
	return wResult - 1;
}

// Euclidean algorithm for computing the Gratest Common Divisor
CBigNumber CBigNumber::GCD(const CBigNumber &a, const CBigNumber &b)
{
	CBigNumber c = a;
	CBigNumber d = b;
	while (true) {
		if (d.IsZero())
			return c;
		c.Mod(d);
		if (c.IsZero())
			return d;
		d.Mod(c);
	}
}

// Extended Euclidean algorithm. Returns the Greatest Common Divisor (d)
// of a and b ans two integers (x and y) satisfying ax + by = d
CBigNumber CBigNumber::GCD(CBigNumber a, CBigNumber b, CBigNumber &x, CBigNumber &y)
{
	CBigNumber d;
	if (b.IsZero()) {
		d = a;
		x = 1;
		y = (Word)0;
		return d;
	}
	CBigNumber e(1), c((Word)0), g((Word)0), f(1), q, r;
	while (b.IsPositive() && !b.IsZero()) {
		q = a / b;
		r = a - q * b;
		x = e - q * c;
		y = g - q * f;
		a = b;
		b = r;
		e = c;
		c = x;
		g = f;
		f = y;
	}
	d = a;
	x = e;
	y = g;
	return d;
}
