/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "haval.hpp"

#define PASSES 5
#define FPTLEN 256
#define VERSION 1

#define f1(x6, x5, x4, x3, x2, x1, x0) ((x1) & ((x0) ^ (x4)) ^ (x2) & (x5) ^ (x3) & (x6) ^ (x0))
#define f2(x6, x5, x4, x3, x2, x1, x0) ((x2) & ((x1) & ~(x3) ^ (x4) & (x5) ^ (x6) ^ (x0)) ^ (x4) & ((x1) ^ (x5)) ^ (x3) & (x5) ^ (x0))
#define f3(x6, x5, x4, x3, x2, x1, x0) ((x3) & ((x1) & (x2) ^ (x6) ^ (x0)) ^ (x1) & (x4) ^ (x2) & (x5) ^ (x0))
#define f4(x6, x5, x4, x3, x2, x1, x0) ((x4) & ((x5) & ~(x2) ^ (x3) & ~(x6) ^ (x1) ^ (x6) ^ (x0)) ^ (x3) & ((x1) & (x2) ^ (x5) ^ (x6)) ^ (x2) & (x6) ^ (x0))
#define f5(x6, x5, x4, x3, x2, x1, x0) ((x0) & ((x1) & (x2) & (x3) ^ ~(x5)) ^ (x1) & (x4) ^ (x2) & (x5) ^ (x3) & (x6))

#if PASSES == 3
#define phi1(x6, x5, x4, x3, x2, x1, x0) f1(x1, x0, x3, x5, x6, x2, x4)
#define phi2(x6, x5, x4, x3, x2, x1, x0) f2(x4, x2, x1, x0, x5, x3, x6)
#define phi3(x6, x5, x4, x3, x2, x1, x0) f3(x6, x1, x2, x3, x4, x5, x0)
#endif

#if PASSES == 4
#define phi1(x6, x5, x4, x3, x2, x1, x0) f1(x2, x6, x1, x4, x5, x3, x0)
#define phi2(x6, x5, x4, x3, x2, x1, x0) f2(x3, x5, x2, x0, x1, x6, x4)
#define phi3(x6, x5, x4, x3, x2, x1, x0) f3(x1, x4, x3, x6, x0, x2, x5)
#define phi4(x6, x5, x4, x3, x2, x1, x0) f4(x6, x4, x0, x5, x2, x1, x3)
#endif

#if PASSES == 5
#define phi1(x6, x5, x4, x3, x2, x1, x0) f1(x3, x4, x1, x0, x5, x2, x6)
#define phi2(x6, x5, x4, x3, x2, x1, x0) f2(x6, x2, x1, x0, x3, x4, x5)
#define phi3(x6, x5, x4, x3, x2, x1, x0) f3(x2, x6, x0, x4, x3, x1, x5)
#define phi4(x6, x5, x4, x3, x2, x1, x0) f4(x1, x5, x3, x2, x0, x4, x6)
#define phi5(x6, x5, x4, x3, x2, x1, x0) f5(x2, x5, x0, x6, x4, x3, x1)
#endif

#define ror(x, n) (((x) >> (n)) | ((x) << (32 - (n))))

#define FF1(x7, x6, x5, x4, x3, x2, x1, x0, w) wTemp = phi1(x6, x5, x4, x3, x2, x1, x0); (x7) = ror(wTemp, 7) + ror((x7), 11) + (w);
#define FF2(x7, x6, x5, x4, x3, x2, x1, x0, w, c) wTemp = phi2(x6, x5, x4, x3, x2, x1, x0); (x7) = ror(wTemp, 7) + ror((x7), 11) + (w) + (c);
#define FF3(x7, x6, x5, x4, x3, x2, x1, x0, w, c) wTemp = phi3(x6, x5, x4, x3, x2, x1, x0); (x7) = ror(wTemp, 7) + ror((x7), 11) + (w) + (c);
#define FF4(x7, x6, x5, x4, x3, x2, x1, x0, w, c) wTemp = phi4(x6, x5, x4, x3, x2, x1, x0); (x7) = ror(wTemp, 7) + ror((x7), 11) + (w) + (c);
#define FF5(x7, x6, x5, x4, x3, x2, x1, x0, w, c) wTemp = phi5(x6, x5, x4, x3, x2, x1, x0); (x7) = ror(wTemp, 7) + ror((x7), 11) + (w) + (c);

CHavalDigest::CHavalDigest()
{
	Initialize();
}

CHavalDigest::CHavalDigest(const Byte *pbData, Word wDataLength)
{
	Initialize();
	HashData(pbData, wDataLength);
}

CHavalDigest::CHavalDigest(const char *pszString)
{
	Initialize();
	HashData((const Byte *)pszString, strlen(pszString));
}

const Byte *CHavalDigest::GetFingerPrint() const
{
	return (const Byte *)wFingerPrint;
}

void CHavalDigest::Initialize()
{
	dwBits = 0;

	wFingerPrint[0] = 0x243F6A88;
	wFingerPrint[1] = 0x85A308D3;
	wFingerPrint[2] = 0x13198A2E;
	wFingerPrint[3] = 0x03707344;
	wFingerPrint[4] = 0xA4093822;
	wFingerPrint[5] = 0x299F31D0;
	wFingerPrint[6] = 0x082EFA98;
	wFingerPrint[7] = 0xEC4E6C89;
}

void CHavalDigest::HashData(const Byte *pbData, Word wDataLength)
{
	dwBits = (Dword)wDataLength << 3;	// Number of bits in the original text
	
	int wFullBlocks = wDataLength / 128;
	int wLastBlock = wDataLength % 128;
	int i;
	
	for (i = 0; i < wFullBlocks; i++) {
		memcpy(wBlock, (Byte *)pbData + i * 128, 128);
		HashBlock();
	}
	memcpy(wBlock, (Byte *)pbData + i * 128, wLastBlock);
	((Byte *)wBlock)[wLastBlock] = 0x80;
	memset((Byte *)wBlock + wLastBlock + 1, 0, 128 - wLastBlock - 1);
	
// If there are not 10 free bytes at the end of wBlock then hash it and clear
//	wBlock after that
	if (wLastBlock >= 118) {
		HashBlock();
		memset(wBlock, 0, 118);
	}
	
	*((Byte *)wBlock + 118) = ((FPTLEN & 3) << 6 | (PASSES & 7) << 3 | VERSION & 7);
	*((Byte *)wBlock + 119) = (FPTLEN >> 2 & 0x0FF);
	*(Word *)((Byte *)wBlock + 120) = (Word)(dwBits);
	*(Word *)((Byte *)wBlock + 124) = (Word)(dwBits >> 32);
	
	HashBlock();
	
	HavalFold();
}

void CHavalDigest::HashBlock()
{
	Word t0 = wFingerPrint[0],
		  t1 = wFingerPrint[1],
		  t2 = wFingerPrint[2],
		  t3 = wFingerPrint[3],
		  t4 = wFingerPrint[4],
		  t5 = wFingerPrint[5],
		  t6 = wFingerPrint[6],
		  t7 = wFingerPrint[7],
		  *w = wBlock;

// Pass 1	
	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w     ));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 1 ));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 2 ));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 3 ));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 4 ));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 5 ));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 6 ));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 7 ));

	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 8 ));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 9 ));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 10));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 11));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 12));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 13));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 14));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 15));

	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 16));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 17));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 18));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 19));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 20));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 21));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 22));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 23));

	FF1(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 24));
	FF1(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 25));
	FF1(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 26));
	FF1(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 27));
	FF1(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 28));
	FF1(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 29));
	FF1(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 30));
	FF1(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 31));

// Pass 2
	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 5 ), 0x452821E6);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 14), 0x38D01377);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 26), 0xBE5466CF);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 18), 0x34E90C6C);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 11), 0xC0AC29B7);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 28), 0xC97C50DD);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 7 ), 0x3F84D5B5);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 16), 0xB5470917);

	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w     ), 0x9216D5D9);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 23), 0x8979FB1B);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 20), 0xD1310BA6);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 22), 0x98DFB5AC);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 1 ), 0x2FFD72DB);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 10), 0xD01ADFB7);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 4 ), 0xB8E1AFED);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 8 ), 0x6A267E96);

	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 30), 0xBA7C9045);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  3), 0xF12C7F99);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 21), 0x24A19947);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w +  9), 0xB3916CF7);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 17), 0x0801F2E2);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 24), 0x858EFC16);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 29), 0x636920D8);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  6), 0x71574E69);

	FF2(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 19), 0xA458FEA3);
	FF2(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 12), 0xF4933D7E);
	FF2(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 15), 0x0D95748F);
	FF2(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 13), 0x728EB658);
	FF2(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  2), 0x718BCD58);
	FF2(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 25), 0x82154AEE);
	FF2(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 31), 0x7B54A41D);
	FF2(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 27), 0xC25A59B5);

// Pass 3
	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 19), 0x9C30D539);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  9), 0x2AF26013);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w +  4), 0xC5D1B023);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 20), 0x286085F0);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 28), 0xCA417918);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 17), 0xB8DB38EF);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w +  8), 0x8E79DCB0);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 22), 0x603A180E);

	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 29), 0x6C9E0E8B);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 14), 0xB01E8A3E);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 25), 0xD71577C1);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 12), 0xBD314B27);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 24), 0x78AF2FDA);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 30), 0x55605C60);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 16), 0xE65525F3);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 26), 0xAA55AB94);

	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 31), 0x57489862);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 15), 0x63E81440);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w +  7), 0x55CA396A);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w +  3), 0x2AAB10B6);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  1), 0xB4CC5C34);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w     ), 0x1141E8CE);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 18), 0xA15486AF);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 27), 0x7C72E993);

	FF3(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 13), 0xB3EE1411);
	FF3(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  6), 0x636FBC2A);
	FF3(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 21), 0x2BA9C55D);
	FF3(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 10), 0x741831F6);
	FF3(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 23), 0xCE5C3E16);
	FF3(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 11), 0x9B87931E);
	FF3(t1, t0, t7, t6, t5, t4, t3, t2, *(w +  5), 0xAFD6BA33);
	FF3(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  2), 0x6C24CF5C);

#if PASSES >= 4
// Pass 4
	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 24), 0x7A325381);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  4), 0x28958677);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w     ), 0x3B8F4898);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 14), 0x6B4BB9AF);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  2), 0xC4BFE81B);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  7), 0x66282193);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 28), 0x61D809CC);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 23), 0xFB21A991);

	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 26), 0x487CAC60);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  6), 0x5DEC8032);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 30), 0xEF845D5D);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 20), 0xE98575B1);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 18), 0xDC262302);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 25), 0xEB651B88);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 19), 0x23893E81);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  3), 0xD396ACC5);

	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 22), 0x0F6D6FF3);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 11), 0x83F44239);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 31), 0x2E0B4482);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 21), 0xA4842004);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  8), 0x69C8F04A);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 27), 0x9E1F9B5E);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 12), 0x21C66842);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w +  9), 0xF6E96C9A);

	FF4(t7, t6, t5, t4, t3, t2, t1, t0, *(w +  1), 0x670C9C61);
	FF4(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 29), 0xABD388F0);
	FF4(t5, t4, t3, t2, t1, t0, t7, t6, *(w +  5), 0x6A51A0D2);
	FF4(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 15), 0xD8542F68);
	FF4(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 17), 0x960FA728);
	FF4(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 10), 0xAB5133A3);
	FF4(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 16), 0x6EEF0B6C);
	FF4(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 13), 0x137A3BE4);
#endif
	
#if PASSES == 5
// Pass 5	
	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 27), 0xBA3BF050);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  3), 0x7EFB2A98);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 21), 0xA1F1651D);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 26), 0x39AF0176);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 17), 0x66CA593E);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w + 11), 0x82430E88);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 20), 0x8CEE8619);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 29), 0x456F9FB4);

	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w + 19), 0x7D84A5C3);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w     ), 0x3B8B5EBE);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 12), 0xE06F75D8);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w +  7), 0x85C12073);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 13), 0x401A449F);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  8), 0x56C16AA6);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 31), 0x4ED3AA62);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 10), 0x363F7706);

	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w +  5), 0x1BFEDF72);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w +  9), 0x429B023D);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 14), 0x37D0D724);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 30), 0xD00A1248);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w + 18), 0xDB0FEAD3);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  6), 0x49F1C09B);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 28), 0x075372C9);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 24), 0x80991B7B);

	FF5(t7, t6, t5, t4, t3, t2, t1, t0, *(w +  2), 0x25D479D8);
	FF5(t6, t5, t4, t3, t2, t1, t0, t7, *(w + 23), 0xF6E8DEF7);
	FF5(t5, t4, t3, t2, t1, t0, t7, t6, *(w + 16), 0xE3FE501A);
	FF5(t4, t3, t2, t1, t0, t7, t6, t5, *(w + 22), 0xB6794C3B);
	FF5(t3, t2, t1, t0, t7, t6, t5, t4, *(w +  4), 0x976CE0BD);
	FF5(t2, t1, t0, t7, t6, t5, t4, t3, *(w +  1), 0x04C006BA);
	FF5(t1, t0, t7, t6, t5, t4, t3, t2, *(w + 25), 0xC1A94FB6);
	FF5(t0, t7, t6, t5, t4, t3, t2, t1, *(w + 15), 0x409F60C4);
#endif	
	
	wFingerPrint[0] += t0;
	wFingerPrint[1] += t1;
	wFingerPrint[2] += t2;
	wFingerPrint[3] += t3;
	wFingerPrint[4] += t4;
	wFingerPrint[5] += t5;
	wFingerPrint[6] += t6;
	wFingerPrint[7] += t7;
}

void CHavalDigest::HavalFold()
{
#if FPTLEN == 128
	wTemp = (wFingerPrint[7] & 0x000000FF) |
			  (wFingerPrint[6] & 0xFF000000) |
	  		  (wFingerPrint[5] & 0x00FF0000) |
	        (wFingerPrint[4] & 0x0000FF00);
	wFingerPrint[0] += ror(wTemp, 8);

	wTemp = (wFingerPrint[7] & 0x0000FF00) |
           (wFingerPrint[6] & 0x000000FF) |
           (wFingerPrint[5] & 0xFF000000) |
           (wFingerPrint[4] & 0x00FF0000);
	wFingerPrint[1] += ror(wTemp, 16);

	wTemp = (wFingerPrint[7] & 0x00FF0000) |
	  		  (wFingerPrint[6] & 0x0000FF00) |
	  		  (wFingerPrint[5] & 0x000000FF) |
	  		  (wFingerPrint[4] & 0xFF000000);
	wFingerPrint[2] += ror(wTemp, 24);

	wTemp = (wFingerPrint[7] & 0xFF000000) |
	  		  (wFingerPrint[6] & 0x00FF0000) |
	  		  (wFingerPrint[5] & 0x0000FF00) |
	  		  (wFingerPrint[4] & 0x000000FF);
	wFingerPrint[3] += wTemp;
#elif FPTLEN == 160
	wTemp = (wFingerPrint[7] & (Word)0x3F) |
	  		  (wFingerPrint[6] & ((Word)0x7F << 25)) |
	  		  (wFingerPrint[5] & ((Word)0x3F << 19));
	wFingerPrint[0] += ror(wTemp, 19);

	wTemp = (wFingerPrint[7] & ((Word)0x3F <<  6)) |
	  		  (wFingerPrint[6] &  (Word)0x3F) |
	  		  (wFingerPrint[5] & ((Word)0x7F << 25));
	wFingerPrint[1] += ror(wTemp, 25);

	wTemp = (wFingerPrint[7] & ((Word)0x7F << 12)) |
	  		  (wFingerPrint[6] & ((Word)0x3F <<  6)) |
	  		  (wFingerPrint[5] &  (Word)0x3F);
	wFingerPrint[2] += wTemp;

	wTemp = (wFingerPrint[7] & ((Word)0x3F << 19)) |
	  		  (wFingerPrint[6] & ((Word)0x7F << 12)) |
	  		  (wFingerPrint[5] & ((Word)0x3F <<  6));
	wFingerPrint[3] += wTemp >> 6;

	wTemp = (wFingerPrint[7] & ((Word)0x7F << 25)) |
	  		  (wFingerPrint[6] & ((Word)0x3F << 19)) |
	 		  (wFingerPrint[5] & ((Word)0x7F << 12));
	wFingerPrint[4] += wTemp >> 12;
#elif FPTLEN == 192
	wTemp = (wFingerPrint[7] & (Word)0x1F) |
	  		  (wFingerPrint[6] & ((Word)0x3F << 26));
	wFingerPrint[0] += ror(wTemp, 26);

	wTemp = (wFingerPrint[7] & ((Word)0x1F <<  5)) |
	  		  (wFingerPrint[6] &  (Word)0x1F);
	wFingerPrint[1] += wTemp;

	wTemp = (wFingerPrint[7] & ((Word)0x3F << 10)) |
	  		  (wFingerPrint[6] & ((Word)0x1F <<  5));
	wFingerPrint[2] += wTemp >> 5;

	wTemp = (wFingerPrint[7] & ((Word)0x1F << 16)) |
	  		  (wFingerPrint[6] & ((Word)0x3F << 10));
	wFingerPrint[3] += wTemp >> 10;

	wTemp = (wFingerPrint[7] & ((Word)0x1F << 21)) |
	  		  (wFingerPrint[6] & ((Word)0x1F << 16));
	wFingerPrint[4] += wTemp >> 16;

	wTemp = (wFingerPrint[7] & ((Word)0x3F << 26)) |
	  		  (wFingerPrint[6] & ((Word)0x1F << 21));
	wFingerPrint[5] += wTemp >> 21;
#elif FPTLEN == 224
	wFingerPrint[0] += (wFingerPrint[7] >> 27) & 0x1F;
	wFingerPrint[1] += (wFingerPrint[7] >> 22) & 0x1F;
	wFingerPrint[2] += (wFingerPrint[7] >> 18) & 0x0F;
	wFingerPrint[3] += (wFingerPrint[7] >> 13) & 0x1F;
	wFingerPrint[4] += (wFingerPrint[7] >>  9) & 0x0F;
	wFingerPrint[5] += (wFingerPrint[7] >>  4) & 0x1F;
	wFingerPrint[6] +=  wFingerPrint[7]        & 0x0F;
#endif
}

void CHavalDigest::SetFingerPrint(const Byte *bBytes)
{
	memcpy(wFingerPrint, bBytes, GetFingerPrintSize());
}

void CHavalDigest::SetBlock(const Byte *bBytes)
{
	memcpy(wBlock, bBytes, GetBlockSize());
}
