/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ripemd160.hpp"

#define rol(p, q) (((p) << (q)) | ((p) >> (32 - (q))))

#define F(x, y, z) (x ^ y ^ z)
#define G(x, y, z) (z ^ (x & (y ^ z)))
#define H(x, y, z) (z ^ (x | ~y))
#define I(x, y, z) (y ^ (z & (x ^ y)))
#define J(x, y, z) (x ^ (y | ~z))

#define FF(a, b, c, d, e, X, s, y) ((a) = (e) + rol((a) + ((F((b), (c), (d)) + (X) + (y))), (s))); ((c) = rol((c), 10))
#define GG(a, b, c, d, e, X, s, y) ((a) = (e) + rol((a) + ((G((b), (c), (d)) + (X) + (y))), (s))); ((c) = rol((c), 10))
#define HH(a, b, c, d, e, X, s, y) ((a) = (e) + rol((a) + ((H((b), (c), (d)) + (X) + (y))), (s))); ((c) = rol((c), 10))
#define II(a, b, c, d, e, X, s, y) ((a) = (e) + rol((a) + ((I((b), (c), (d)) + (X) + (y))), (s))); ((c) = rol((c), 10))
#define JJ(a, b, c, d, e, X, s, y) ((a) = (e) + rol((a) + ((J((b), (c), (d)) + (X) + (y))), (s))); ((c) = rol((c), 10))

CRIPEMD160Digest::CRIPEMD160Digest()
{
	Initialize();
}

CRIPEMD160Digest::CRIPEMD160Digest(const Byte *pbData, Word wDataLength)
{
	Initialize();
	HashData(pbData, wDataLength);
}

CRIPEMD160Digest::CRIPEMD160Digest(const char *pszString)
{
	Initialize();
	HashData((const Byte *)pszString, strlen(pszString));
}

const Byte *CRIPEMD160Digest::GetFingerPrint() const
{
	return (const Byte *)wFingerPrint;
}

void CRIPEMD160Digest::Initialize()
{
	dwBits = 0;
	
	wFingerPrint[0] = 0x67452301;
	wFingerPrint[1] = 0xefcdab89;
	wFingerPrint[2] = 0x98badcfe;
	wFingerPrint[3] = 0x10325476;
	wFingerPrint[4] = 0xc3d2e1f0;
}

void CRIPEMD160Digest::HashData(const Byte *pbData, Word wDataLength)
{
	dwBits = (Dword)wDataLength << 3;	// Number of bits in the original text
	
	Word wFullBlocks = wDataLength / 64;
	Word wLastBlock = wDataLength % 64;
	Word i;
	
	for (i = 0; i < wFullBlocks; i++) {
		memcpy(wBlock, (Byte *)pbData + i * 64, 64);
		HashBlock();
	}
	memcpy(wBlock, (Byte *)pbData + i * 64, wLastBlock);
	((Byte *)wBlock)[wLastBlock] = 0x80;
	memset((Byte *)wBlock + wLastBlock + 1, 0, 64 - wLastBlock - 1);
	
// If there are not 8 free bytes at the end of wBlock then hash it and clear
//	wBlock after that
	if (wLastBlock >= 56) {
		HashBlock();
		memset(wBlock, 0, 56);
	}
	
	*(Word *)((Byte *)wBlock + 56) = (Word)(dwBits);
	*(Word *)((Byte *)wBlock + 60) = (Word)(dwBits >> 32);
	
	HashBlock();
}

void CRIPEMD160Digest::HashBlock()
{
	Word Al = wFingerPrint[0],
		  Bl = wFingerPrint[1],
		  Cl = wFingerPrint[2],
		  Dl = wFingerPrint[3],
		  El = wFingerPrint[4],
		  Ar = wFingerPrint[0],
		  Br = wFingerPrint[1],
		  Cr = wFingerPrint[2],
		  Dr = wFingerPrint[3],
		  Er = wFingerPrint[4],
		  *X = wBlock, t;

#include "ripemd160rounds.inc"

	t = wFingerPrint[0];
	wFingerPrint[0] = wFingerPrint[1] + Cl + Dr;
	wFingerPrint[1] = wFingerPrint[2] + Dl + Er;
	wFingerPrint[2] = wFingerPrint[3] + El + Ar;
	wFingerPrint[3] = wFingerPrint[4] + Al + Br;
	wFingerPrint[4] = t + Bl + Cr;
}

void CRIPEMD160Digest::SetFingerPrint(const Byte *bBytes)
{
	memcpy(wFingerPrint, bBytes, GetFingerPrintSize());
}

void CRIPEMD160Digest::SetBlock(const Byte *bBytes)
{
	memcpy(wBlock, bBytes, GetBlockSize());
}
