/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "sha1.hpp"

#define rol(p, q) (((p) << (q)) | ((p) >> (32 - (q))))

#define F(x, y, z) (z ^ (x & (y ^ z)))
#define G(x, y, z) (x ^ y ^ z)
#define H(x, y, z) ((x & y) | (x & z) | (y & z))
#define I(x, y, z) (x ^ y ^ z)

#define FF(a, b, c, d, e, X, K) ((e) = rol((a), 5) + F((b), (c), (d)) + (e) + (X) + (K)); ((b) = rol((b), 30))
#define GG(a, b, c, d, e, X, K) ((e) = rol((a), 5) + G((b), (c), (d)) + (e) + (X) + (K)); ((b) = rol((b), 30))
#define HH(a, b, c, d, e, X, K) ((e) = rol((a), 5) + H((b), (c), (d)) + (e) + (X) + (K)); ((b) = rol((b), 30))
#define II(a, b, c, d, e, X, K) ((e) = rol((a), 5) + I((b), (c), (d)) + (e) + (X) + (K)); ((b) = rol((b), 30))

CSHA1Digest::CSHA1Digest()
{
	Initialize();
}

CSHA1Digest::CSHA1Digest(const Byte *pbData, Word wDataLength)
{
	Initialize();
	HashData(pbData, wDataLength);
}

CSHA1Digest::CSHA1Digest(const char *pszString)
{
	Initialize();
	HashData((const Byte *)pszString, strlen(pszString));
}

const Byte *CSHA1Digest::GetFingerPrint() const
{
	return (const Byte *)wFingerPrint;
}

void CSHA1Digest::Initialize()
{
	dwBits = 0;
	
	wFingerPrint[0] = 0x67452301;
	wFingerPrint[1] = 0xefcdab89;
	wFingerPrint[2] = 0x98badcfe;
	wFingerPrint[3] = 0x10325476;
	wFingerPrint[4] = 0xc3d2e1f0;
}

void CSHA1Digest::HashData(const Byte *pbData, Word wDataLength)
{
	dwBits = (Dword)wDataLength << 3;	// Number of bits in the original text
	
	Word wFullBlocks = wDataLength / 64;
	Word wLastBlock = wDataLength % 64;
	Word i;
	
	for (i = 0; i < wFullBlocks; i++) {
		memcpy(wBlock, (Byte *)pbData + i * 64, 64);
		for (int j = 0; j < 16; j++)
			wBlock[j] = REVERSEWORD(wBlock[j]);
		HashBlock();
	}
	memcpy(wBlock, (Byte *)pbData + i * 64, wLastBlock);
	((Byte *)wBlock)[wLastBlock] = 0x80;
	memset((Byte *)wBlock + wLastBlock + 1, 0, 64 - wLastBlock - 1);
	for (int j = 0; j < 16; j++)
		wBlock[j] = REVERSEWORD(wBlock[j]);

// If there are not 8 free bytes at the end of wBlock then hash it and clear
//	wBlock after that
	if (wLastBlock >= 56) {
		HashBlock();
		memset(wBlock, 0, 56);
	}

	*(Word *)((Byte *)wBlock + 56) = (Word)(dwBits >> 32);	
	*(Word *)((Byte *)wBlock + 60) = (Word)(dwBits);

	HashBlock();
	for (i = 0; i < 5; i++)
		wFingerPrint[i] = REVERSEWORD(wFingerPrint[i]);
}

void CSHA1Digest::HashBlock()
{
	Word a = wFingerPrint[0],
		  b = wFingerPrint[1],
		  c = wFingerPrint[2],
		  d = wFingerPrint[3],
		  e = wFingerPrint[4],
		  *W = wTemp, t;

	for (t = 0; t < 16; t++)
		W[t] = wBlock[t];
	for ( ; t < 80; t++) {
		W[t] = W[t - 3] ^ W[t - 8] ^ W[t - 14] ^ W[t - 16];
		W[t] = rol(W[t], 1);
	}
#include "sha1rounds.inc"
	wFingerPrint[0] += a;
	wFingerPrint[1] += b;
	wFingerPrint[2] += c;
	wFingerPrint[3] += d;
	wFingerPrint[4] += e;
}

void CSHA1Digest::SetFingerPrint(const Byte *bBytes)
{
	memcpy(wFingerPrint, bBytes, GetFingerPrintSize());
}

void CSHA1Digest::SetBlock(const Byte *bBytes)
{
	memcpy(wBlock, bBytes, GetBlockSize());
}
