/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "dh.hpp"

CDHKeyExchange::CDHKeyExchange(Word wPrimeSize)
{
	CDHKeyExchange::wPrimeSize = wPrimeSize;
	
	cGenerator = 5;

	cPrime.SetRandom(wPrimeSize, true);
	Word i = cPrime % 10;
	while ((i != 3) && (i != 7)) {
		cPrime += 2;
		i = cPrime % 10;
	}
	do {
		cPrime += 10;
	} while (!cPrime.IsPrime());

	cPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);

	MakeKeys();
}

CDHKeyExchange::CDHKeyExchange(const CProbablePrime &cPrime, const CBigNumber &cGenerator, const CBigNumber &cPrivate)
{
	CDHKeyExchange::cPrime = cPrime;
	CDHKeyExchange::cGenerator = cGenerator;
	CDHKeyExchange::cPrivate = cPrivate;
	
	MakeKeys();
}

CDHKeyExchange::CDHKeyExchange(const CProbablePrime &cPrime, const CBigNumber &cGenerator)
{
	CDHKeyExchange::cPrime = cPrime;
	CDHKeyExchange::cGenerator = cGenerator;

	cPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);
	
	MakeKeys();
}

CDHKeyExchange::CDHKeyExchange(const CDHKeyExchange &cDHKeyExchange)
{
	wPrimeSize = cDHKeyExchange.wPrimeSize;
	cPrime = cDHKeyExchange.cPrime;
	cGenerator = cDHKeyExchange.cGenerator;
	cPrivate = cDHKeyExchange.cPrivate;
	cPublic = cDHKeyExchange.cPublic;
	cOtherPublic = cDHKeyExchange.cOtherPublic;
	cKey = cDHKeyExchange.cKey;
}

void CDHKeyExchange::MakeKeys()
{
	cPublic = CBigNumber::ModExp(cGenerator, cPrivate, cPrime);
}

void CDHKeyExchange::Agree()
{
	cKey = CBigNumber::ModExp(cOtherPublic, cPrivate, cPrime);
}

void CDHKeyExchange::Dump()
{
	printf("p = "); cPrime.Dump();
	printf("g = "); cGenerator.Dump();
	printf("pri = "); cPrivate.Dump();
	printf("pub = "); cPublic.Dump();
	printf("other pub = "); cOtherPublic.Dump();
	printf("key = "); cKey.Dump();
}

void CDHKeyExchange::WritePublic(int iOut)
{
	CDEREncodedBigNumber cAll(cPublic);
	cAll.Write(iOut);
}

void CDHKeyExchange::ReadOtherPublic(int iIn)
{
	CDEREncodedBigNumber cAll;

	cAll.Read(iIn);

	cOtherPublic = cAll;
}

Byte *CDHKeyExchange::GetKey()
{
	return (Byte *)cKey.GetData();
}

Word CDHKeyExchange::GetKeySize()
{
	return cKey.GetWords() * BYTESINWORD;
}

CDH2KeyExchange::CDH2KeyExchange(Word wPrimeSize) : CDHKeyExchange(wPrimeSize)
{
	cTempPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);
	MakeKeys();
}

CDH2KeyExchange::CDH2KeyExchange(const CProbablePrime &cPrime, const CBigNumber &cGenerator, const CBigNumber &cPrivate, const CBigNumber &cTempPrivate) : CDHKeyExchange(cPrime, cGenerator, cPrivate)
{
	CDH2KeyExchange::cTempPrivate = cTempPrivate;
	
	MakeKeys();
}

CDH2KeyExchange::CDH2KeyExchange(const CProbablePrime &cPrime, const CBigNumber &cGenerator) : CDHKeyExchange(cPrime, cGenerator)
{
	cTempPrivate.SetRandom(cPrime.GetWords() * BITSINWORD, false);
	
	MakeKeys();
}

CDH2KeyExchange::CDH2KeyExchange(const CDH2KeyExchange &cDH2KeyExchange) : CDHKeyExchange((CDHKeyExchange &)cDH2KeyExchange)
{
	cTempPrivate = cDH2KeyExchange.cTempPrivate;
	cTempPublic = cDH2KeyExchange.cTempPublic;
	cOtherTempPublic = cDH2KeyExchange.cOtherTempPublic;
}

void CDH2KeyExchange::MakeKeys()
{
	cTempPublic = CBigNumber::ModExp(cGenerator, cTempPrivate, cPrime);
}

void CDH2KeyExchange::Dump()
{
	CDHKeyExchange::Dump();
	printf("temp pri = "); cTempPrivate.Dump();
	printf("temp pub = "); cTempPublic.Dump();
	printf("other temp pub = "); cOtherTempPublic.Dump();
}

void CDH2KeyExchange::SetOtherPublic(const CBigNumber &cPublic, const CBigNumber &cTempPublic)
{
	CDHKeyExchange::SetOtherPublic(cPublic);
	cOtherTempPublic = cTempPublic;
}

void CDH2KeyExchange::Agree()
{
	CDHKeyExchange::Agree();
	CBigNumber cSecondKey = CBigNumber::ModExp(cOtherTempPublic, cTempPrivate, cPrime);
	cKey.Shl(cSecondKey.GetWords() * BITSINWORD);
													// Make room for the second key
	cKey += cSecondKey;						// Concatenate it
}

void CDH2KeyExchange::WritePublic(int iOut)
{
	CDHKeyExchange::WritePublic(iOut);

	CDEREncodedBigNumber cAll(cTempPublic);
	cAll.Write(iOut);
}

void CDH2KeyExchange::ReadOtherPublic(int iIn)
{
	CDHKeyExchange::ReadOtherPublic(iIn);
	
	CDEREncodedBigNumber cAll;

	cAll.Read(iIn);

	cOtherTempPublic = cAll;
}
