/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "3way.hpp"

#define r(a, b, c)		\
{								\
	Word q = a;				\
	Word r = b;				\
	MIRROR(c, 32, a);		\
	MIRROR(q, 32, c);		\
	MIRROR(r, 32, b);		\
	q = r = 0;				\
}

#define f(a, b, c)						 							\
{																			\
	Word x, y, z;														\
	x = (a      ) ^ (a >> 16) ^ (b << 16) ^ (b >> 16) ^	\
	  	 (c << 16) ^ (b >> 24) ^ (c <<  8) ^ (c >>  8) ^	\
		 (a << 24) ^ (c >> 16) ^ (a << 16) ^ (c >> 24) ^	\
	  	 (a <<  8);														\
	y = (b      ) ^ (b >> 16) ^ (c << 16) ^ (c >> 16) ^	\
	  	 (a << 16) ^ (c >> 24) ^ (a <<  8) ^ (a >>  8) ^	\
	  	 (b << 24) ^ (a >> 16) ^ (b << 16) ^ (a >> 24) ^	\
	  	 (b <<  8);														\
	z = (c      ) ^ (c >> 16) ^ (a << 16) ^ (a >> 16) ^	\
	  	 (b << 16) ^ (a >> 24) ^ (b <<  8) ^ (b >>  8) ^	\
	  	 (c << 24) ^ (b >> 16) ^ (c << 16) ^ (b >> 24) ^	\
	  	 (c <<  8);														\
	a = x;																\
	b = y;																\
	c = z;																\
	x = y = z = 0;														\
}

C3WayKey::C3WayKey()
{
}

C3WayKey::C3WayKey(const C3WayKey &c3WayKey)
{
	wMaster[0] = c3WayKey.wMaster[0];
	wMaster[1] = c3WayKey.wMaster[1];
	wMaster[2] = c3WayKey.wMaster[2];
	MakeKeys();
}

C3WayKey::C3WayKey(const Byte *pbData, Word wLength)
{
	memcpy((Byte *)wMaster, pbData, wLength);
	memset((Byte *)wMaster + wLength, 0, sizeof(wMaster) - wLength);
	wMaster[0] = REVERSEWORD(wMaster[0]);
	wMaster[1] = REVERSEWORD(wMaster[1]);
	wMaster[2] = REVERSEWORD(wMaster[2]);
	MakeKeys();
}

C3WayKey::C3WayKey(Word a, Word b, Word c)
{
	wMaster[0] = a;
	wMaster[1] = b;
	wMaster[2] = c;
	MakeKeys();
	a = b = c = 0;								// Cleanup
}

C3WayKey::~C3WayKey()
{
	wMaster[0] = 0;
	wMaster[1] = 0;
	wMaster[2] = 0;							// Cleanup
}

void C3WayKey::MakeKeys()
{
	int i = 0;
	wKey[i] = 0x0b0b;
	for (i++; i <= THREEWAYROUNDS; i++) {
		wKey[i] = wKey[i - 1] << 1;
		if (wKey[i] & 0x10000)
			wKey[i] ^= 0x11011;
	}
	wKey[i] = 0xb1b1;
	for (i++; i <= (THREEWAYROUNDS * 2 + 1); i++) {
		wKey[i] = wKey[i - 1] << 1;
		if (wKey[i] & 0x10000)
			wKey[i] ^= 0x11011;
	}
	
	wMaster[3] = wMaster[0];
	wMaster[4] = wMaster[1];
	wMaster[5] = wMaster[2];

	f(wMaster[3], wMaster[4], wMaster[5]);
	r(wMaster[3], wMaster[4], wMaster[5]);
}

C3WayBlock::C3WayBlock()
{
	wData[0] = 0;								// This is here for definiteness
	wData[1] = 0;
	wData[2] = 0;
}

C3WayBlock::C3WayBlock(const C3WayBlock &c3WayBlock)
{
	wData[0] = c3WayBlock.wData[0];
	wData[1] = c3WayBlock.wData[1];
	wData[2] = c3WayBlock.wData[2];
}

C3WayBlock::C3WayBlock(Word a, Word b, Word c)
{
	wData[0] = a;
	wData[1] = b;
	wData[2] = c;
}

C3WayBlock::C3WayBlock(const Byte *pbData, Word wLength)
{
	SetData(pbData, wLength);
}

C3WayBlock::~C3WayBlock()
{
	wData[0] = 0;								// Cleanup
	wData[1] = 0;
	wData[2] = 0;
}

void C3WayBlock::Encrypt(const C3WayKey &c3WayKey)
{
	Crypt(c3WayKey, false);
}

void C3WayBlock::Decrypt(const C3WayKey &c3WayKey)
{
	r(wData[0], wData[1], wData[2]);
	Crypt(c3WayKey, true);
	r(wData[0], wData[1], wData[2]);
}

void C3WayBlock::Crypt(const C3WayKey &c3WayKey, bool fgDirection)
{
	Word a, b, c;
	
	int i;
	int k = fgDirection ? THREEWAYROUNDS + 1 : 0;
	int l = fgDirection ? 3 : 0;
	for (i = 0; i < THREEWAYROUNDS; i++) {
		wData[0] ^= c3WayKey.GetMaster(0 + l) ^ (c3WayKey.GetKey(k + i) << 16);
		wData[1] ^= c3WayKey.GetMaster(1 + l);
		wData[2] ^= c3WayKey.GetMaster(2 + l) ^ c3WayKey.GetKey(k + i);
		
		f(wData[0], wData[1], wData[2]);
		wData[0] = (wData[0] >> 10) ^ (wData[0] << 22);
		wData[2] = (wData[2] << 1) ^ (wData[2] >> 31);
		a = wData[0] ^ (wData[1] | ~wData[2]);
		b = wData[1] ^ (wData[2] | ~wData[0]);
		c = wData[2] ^ (wData[0] | ~wData[1]);
		wData[0] = a;
		wData[1] = b;
		wData[2] = c;
		wData[0] = (wData[0] << 1) ^ (wData[0] >> 31);
		wData[2] = (wData[2] >> 10) ^ (wData[2] << 22);
	}
	wData[0] ^= c3WayKey.GetMaster(0 + l) ^ (c3WayKey.GetKey(k + i) << 16);
	wData[1] ^= c3WayKey.GetMaster(1 + l);
	wData[2] ^= c3WayKey.GetMaster(2 + l) ^ c3WayKey.GetKey(k + i);
	
	f(wData[0], wData[1], wData[2]);
	
	a = b = c = 0;								// Cleanup
}

void C3WayBlock::SetData(Word a, Word b, Word c)
{
	wData[0] = a;
	wData[1] = b;
	wData[2] = c;
}

Word C3WayBlock::GetData(Word i)
{
	return wData[i];
}

void C3WayBlock::SetData(const Byte *pbData, Word wLength)
{
	memcpy((Byte *)wData, pbData, wLength);
	memset((Byte *)wData, 0, sizeof(wData) - wLength);
	wData[0] = REVERSEWORD(wData[0]);
	wData[1] = REVERSEWORD(wData[1]);
	wData[2] = REVERSEWORD(wData[2]);
}

Byte *C3WayBlock::GetData()
{
	wResult[0] = REVERSEWORD(wData[0]);
	wResult[1] = REVERSEWORD(wData[1]);
	wResult[2] = REVERSEWORD(wData[2]);
	return (Byte *)wResult;
}
