/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "diamond.hpp"
#include "crc32.hpp"

#define f(x, y, s, r) y[ 0] = s[r][ 0][x[ 0]]; \
							 y[ 1] = s[r][ 1][x[ 1]]; \
							 y[ 2] = s[r][ 2][x[ 2]]; \
							 y[ 3] = s[r][ 3][x[ 3]]; \
							 y[ 4] = s[r][ 4][x[ 4]]; \
							 y[ 5] = s[r][ 5][x[ 5]]; \
							 y[ 6] = s[r][ 6][x[ 6]]; \
							 y[ 7] = s[r][ 7][x[ 7]]; \
							 y[ 8] = s[r][ 8][x[ 8]]; \
							 y[ 9] = s[r][ 9][x[ 9]]; \
							 y[10] = s[r][10][x[10]]; \
							 y[11] = s[r][11][x[11]]; \
							 y[12] = s[r][12][x[12]]; \
							 y[13] = s[r][13][x[13]]; \
							 y[14] = s[r][14][x[14]]; \
							 y[15] = s[r][15][x[15]]

#define h(x, y, a, b, c, d, e, f, g, h) y[a] = (x[a] &   1) | (x[b] &   2) | (x[c] &   4) | (x[d] &   8) | \
															  (x[e] &  16) | (x[f] &  32) | (x[g] &  64) | (x[h] & 128)

#define p(x, y) h(x, y,  0,  1,  2,  3,  4,  5,  6,  7); \
					 h(x, y,  1,  2,  3,  4,  5,  6,  7,  8); \
					 h(x, y,  2,  3,  4,  5,  6,  7,  8,  9); \
					 h(x, y,  3,  4,  5,  6,  7,  8,  9, 10); \
					 h(x, y,  4,  5,  6,  7,  8,  9, 10, 11); \
					 h(x, y,  5,  6,  7,  8,  9, 10, 11, 12); \
					 h(x, y,  6,  7,  8,  9, 10, 11, 12, 13); \
					 h(x, y,  7,  8,  9, 10, 11, 12, 13, 14); \
					 h(x, y,  8,  9, 10, 11, 12, 13, 14, 15); \
					 h(x, y,  9, 10, 11, 12, 13, 14, 15,  0); \
					 h(x, y, 10, 11, 12, 13, 14, 15,  0,  1); \
					 h(x, y, 11, 12, 13, 14, 15,  0,  1,  2); \
					 h(x, y, 12, 13, 14, 15,  0,  1,  2,  3); \
					 h(x, y, 13, 14, 15,  0,  1,  2,  3,  4); \
					 h(x, y, 14, 15,  0,  1,  2,  3,  4,  5); \
					 h(x, y, 15,  0,  1,  2,  3,  4,  5,  6);

#define q(x, y) h(x, y,  0, 15, 14, 13, 12, 11, 10,  9); \
					 h(x, y,  1,  0, 15, 14, 13, 12, 11, 10); \
					 h(x, y,  2,  1,  0, 15, 14, 13, 12, 11); \
					 h(x, y,  3,  2,  1,  0, 15, 14, 13, 12); \
					 h(x, y,  4,  3,  2,  1,  0, 15, 14, 13); \
					 h(x, y,  5,  4,  3,  2,  1,  0, 15, 14); \
					 h(x, y,  6,  5,  4,  3,  2,  1,  0, 15); \
					 h(x, y,  7,  6,  5,  4,  3,  2,  1,  0); \
					 h(x, y,  8,  7,  6,  5,  4,  3,  2,  1); \
					 h(x, y,  9,  8,  7,  6,  5,  4,  3,  2); \
					 h(x, y, 10,  9,  8,  7,  6,  5,  4,  3); \
					 h(x, y, 11, 10,  9,  8,  7,  6,  5,  4); \
					 h(x, y, 12, 11, 10,  9,  8,  7,  6,  5); \
					 h(x, y, 13, 12, 11, 10,  9,  8,  7,  6); \
					 h(x, y, 14, 13, 12, 11, 10,  9,  8,  7); \
					 h(x, y, 15, 14, 13, 12, 11, 10,  9,  8);

CDiamondKey::CDiamondKey()
{
}

CDiamondKey::CDiamondKey(const CDiamondKey &cDiamondKey)
{
	for (int i = 0; i < 16; i++)
		bMaster[i] = cDiamondKey.bMaster[i];
	MakeKeys();
}

CDiamondKey::CDiamondKey(const Byte *pbMasterKey, Word wLength)
{
	for (Word i = 0; i < wLength; i++)
		bMaster[i] = pbMasterKey[i];
	MakeKeys();
}

CDiamondKey::~CDiamondKey()
{
	for (int i = 0; i < 16; i++)			// Cleanup
		bMaster[i] = 0;
}

void CDiamondKey::MakeKeys()
{
	Word wAccumulator = 0xffffffffL;
	Word wMask, r = 0;
	
	memset(bS, 0, DIAMONDROUNDS * 16 * 256);
	int i, j, k, iBox = -256;
	for (i = 0; i < DIAMONDROUNDS; i++)
		for (j = 0; j < 16; j++, iBox += 256)
			for (k = 255; k > 0; k--) {
				wMask = 1;
				while (wMask < (Word)k)		// k varies from 255 to 0
					wMask = (wMask << 1) | 1;
				int p, q = 0;
				do {
					wAccumulator = crc32(wAccumulator, (iBox == -256) ? bMaster[r] : *(bS[0][0] + iBox + bMaster[r]));
					if (++r == 16) {
						r = 0;
						wAccumulator = crc32(wAccumulator, 16);
						wAccumulator = crc32(wAccumulator, 0);
					}
					p = wAccumulator & wMask;
					if ((++q > 97) && (p > k))
						p -= k;					// Do not loop forever
				} while (p > k);
// p is the position among the empty components of the
// S box that the number k should be put
				int v = 0;
				while (p >= 0)
					if (0 == bS[i][j][v++])
						p--;
				bS[i][j][v - 1] = k;
			}
	for (i = 0; i < DIAMONDROUNDS; i++)
		for (j = 0; j < 16; j++)
			for (k = 0; k < 256; k++)
				bT[i][j][bS[i][j][k]] = k;
}

CDiamondBlock::CDiamondBlock()
{
	for (int i = 0; i < 16; i++)			// This is here for definiteness
		bData[i] = 0;
}

CDiamondBlock::CDiamondBlock(const CDiamondBlock &cDiamondBlock)
{
	for (int i = 0; i < 16; i++)
		bData[i] = cDiamondBlock.bData[i];
}

CDiamondBlock::CDiamondBlock(Byte *pbData)
{
	for (int i = 0; i < 16; i++)
		bData[i] = pbData[i];
}

CDiamondBlock::CDiamondBlock(const Byte *pbData, Word wLength)
{
	memcpy(bData, pbData, wLength);
	memset(bData + wLength, 0, sizeof(bData) - wLength);
}

CDiamondBlock::~CDiamondBlock()
{
	for (int i = 0; i < 16; i++)			// Cleanup
		bData[i] = 0;
}

void CDiamondBlock::Encrypt(const CDiamondKey &cDiamondKey)
{
	Byte y[16];
	Byte z[16];
	Byte *x = bData;
	const DKEYS &s = cDiamondKey.GetSBox();
	
   f(x, y, s, 0);
	for (int i = 1; i < (DIAMONDROUNDS - 1); i++) {
		p(y, z);
		f(z, y, s, i);
	}
	p(y, z);
	f(z, x, s, (DIAMONDROUNDS - 1));
}

void CDiamondBlock::Decrypt(const CDiamondKey &cDiamondKey)
{
	Byte y[16];
	Byte z[16];
	Byte *x = bData;
	const DKEYS &s = cDiamondKey.GetTBox();
	
   f(x, y, s, (DIAMONDROUNDS - 1));
	for (int i = (DIAMONDROUNDS - 2); i >= 1; i--) {
		q(y, z);
		f(z, y, s, i);
	}
	q(y, z);
	f(z, x, s, 0);
}

void CDiamondBlock::SetData(const Byte *pbData, Word wLength)
{
	memcpy(bData, pbData, wLength);
	memset(bData + wLength, 0, sizeof(bData) - wLength);
}

Byte CDiamondBlock::GetData(Word i)
{
	return bData[i];
}

Byte *CDiamondBlock::GetData()
{
	return bData;
}
