/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "rc5.hpp"

#define rol(p, q) (((p) << (q)) | ((p) >> (32 - (q))))
#define ror(p, q) (((p) >> (q)) | ((p) << (32 - (q))))

CRC5Key::CRC5Key()
{
}

CRC5Key::CRC5Key(const CRC5Key &cRC5Key)
{
	for (int i = 0; i < 16; i++)
		bMaster[i] = cRC5Key.bMaster[i];
	MakeKeys();
}

CRC5Key::CRC5Key(const Byte *pbMasterKey, Word wKeySize)
{
	for (Word i = 0; i < wKeySize; i++)
		bMaster[i] = pbMasterKey[i];
	MakeKeys();
}

CRC5Key::~CRC5Key()
{
	int i;
	for (i = 0; i < 16; i++)				// Cleanup
		bMaster[i] = 0;
	for (i = 0; i < 25; i++)
		wKey[i] = 0;
}

void CRC5Key::MakeKeys()
{
	Word i, wHelp[4];
	
	for (i = 0; i < 4; i++)
		wHelp[i] = MakeWord(bMaster[i * 4 + 3],
								  bMaster[i * 4 + 2],
								  bMaster[i * 4 + 1],
								  bMaster[i * 4 + 0]);
	
	wKey[0] = 0xb7e15163;					// Pw
	for (i = 1; i < 26; i++)
		wKey[i] = wKey[i - 1] + 0x9e3779b9;
													// Qw
	Word s, j = 0, A = 0, B = 0;
	i = 0;
	for (s = 0; s < 78; s++) {
		A = wKey[i] = rol(wKey[i] + A + B, 3);
		i = (i + 1) % 26;
		B = wHelp[j] = rol(wHelp[j] + A + B, A + B);
		j = (j + 1) % 4;
	}
	for (i = 0; i < 4; i++)					// Cleanup
		wHelp[i] = 0;
}

CRC5Block::CRC5Block()
{
	dwData = 0;									// This is here for definiteness
}

CRC5Block::CRC5Block(const CRC5Block &cRC5Block)
{
	dwData = cRC5Block.dwData;
}

CRC5Block::CRC5Block(Dword dwValue)
{
	dwData = dwValue;
}

CRC5Block::CRC5Block(const Byte *pbData, Word wLength)
{
	dwData = 0;
	memcpy((Byte *)&dwData, pbData, wLength);
}

CRC5Block::~CRC5Block()
{
	dwData = 0;									// Cleanup
}

void CRC5Block::Encrypt(const CRC5Key &cRC5Key)
{
	Word A = LoWORD(dwData) + cRC5Key.GetKey(0);
	Word B = HiWORD(dwData) + cRC5Key.GetKey(1);
	
	for (int i = 0; i < 12; i++) {
		A = rol(A ^ B, B) + cRC5Key.GetKey(i * 2 + 2);
		B = rol(A ^ B, A) + cRC5Key.GetKey(i * 2 + 3);
	}
	dwData = ((Dword)B << 32) | A;
	A = 0;										// Cleanup
	B = 0;
}

void CRC5Block::Decrypt(const CRC5Key &cRC5Key)
{
	Word A = LoWORD(dwData);
	Word B = HiWORD(dwData);
	
	for (int i = 12; i >= 1; i--) {
		B = ror(B - cRC5Key.GetKey(i * 2 + 1), A) ^ A;
		A = ror(A - cRC5Key.GetKey(i * 2), B) ^ B;
	}
	A -= cRC5Key.GetKey(0);
	B -= cRC5Key.GetKey(1);
	dwData = ((Dword)B << 32) | A;
	A = 0;										// Cleanup
	B = 0;
}

void CRC5Block::SetValue(Dword dwValue)
{
	dwData = dwValue;
}

Dword CRC5Block::GetValue()
{
	return dwData;
}

void CRC5Block::SetData(const Byte *pbData, Word wLength)
{
	dwData = 0;
	memcpy((Byte *)&dwData, pbData, wLength);
}

Byte *CRC5Block::GetData()
{
	return (Byte *)&dwData;
}
