/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>

#include "rc6.hpp"

#define rol(p, q) (((p) << (q)) | ((p) >> (32 - (q))))
#define ror(p, q) (((p) >> (q)) | ((p) << (32 - (q))))

CRC6Key::CRC6Key()
{
}

CRC6Key::CRC6Key(const CRC6Key &cRC6Key)
{
	for (int i = 0; i < 16; i++)
		bMaster[i] = cRC6Key.bMaster[i];
	MakeKeys();
}

CRC6Key::CRC6Key(const Byte *pbMasterKey, Word wKeySize)
{
	for (Word i = 0; i < wKeySize; i++)
		bMaster[i] = pbMasterKey[i];
	MakeKeys();
}

CRC6Key::~CRC6Key()
{
	int i;
	for (i = 0; i < 16; i++)				// Cleanup
		bMaster[i] = 0;
	for (i = 0; i < 44; i++)
		wKey[i] = 0;
}

void CRC6Key::MakeKeys()
{
	Word i, wHelp[4];
	
	for (i = 0; i < 4; i++)
		wHelp[i] = MakeWord(bMaster[i * 4 + 3],
								  bMaster[i * 4 + 2],
								  bMaster[i * 4 + 1],
								  bMaster[i * 4 + 0]);
	
	wKey[0] = 0xb7e15163;					// Pw
	for (i = 1; i < 44; i++)
		wKey[i] = wKey[i - 1] + 0x9e3779b9;
													// Qw
	Word s, j = 0, A = 0, B = 0;
	i = 0;
	for (s = 0; s < 132; s++) {
		A = wKey[i] = rol(wKey[i] + A + B, 3);
		i = (i + 1) % 44;
		B = wHelp[j] = rol(wHelp[j] + A + B, A + B);
		j = (j + 1) % 4;
	}
	for (i = 0; i < 4; i++)					// Cleanup
		wHelp[i] = 0;
}

CRC6Block::CRC6Block()
{
	dwData[0] = 0;								// This is here for definiteness
	dwData[1] = 0;
}

CRC6Block::CRC6Block(const CRC6Block &cRC6Block)
{
	dwData[0] = cRC6Block.dwData[0];
	dwData[1] = cRC6Block.dwData[1];
}

CRC6Block::CRC6Block(Dword dwValueLeft, Dword dwValueRight)
{
	dwData[0] = dwValueLeft;
	dwData[1] = dwValueRight;
}

CRC6Block::CRC6Block(const Byte *pbData, Word wLength)
{
	SetData(pbData, wLength);
}

CRC6Block::~CRC6Block()
{
	dwData[0] = 0;								// Cleanup
	dwData[1] = 0;
}

void CRC6Block::Encrypt(const CRC6Key &cRC6Key)
{
	Word A = LoWORD(dwData[0]);
	Word B = HiWORD(dwData[0]) + cRC6Key.GetKey(0);
	Word C = LoWORD(dwData[1]);
	Word D = HiWORD(dwData[1]) + cRC6Key.GetKey(1);
	Word q, t, u;

	for (int i = 0; i < 20; i++) {
		t = rol(B * (2 * B + 1), 5);
		u = rol(D * (2 * D + 1), 5);
		A = rol(A ^ t, u) + cRC6Key.GetKey(i * 2 + 2);
		C = rol(C ^ u, t) + cRC6Key.GetKey(i * 2 + 3);
		q = A;
		A = B;
		B = C;
		C = D;
		D = q;
	}
	A += cRC6Key.GetKey(42);
	C += cRC6Key.GetKey(43);
	dwData[0] = ((Dword)B << 32) | A;
	dwData[1] = ((Dword)D << 32) | C;
	A = B = C = D = 0;						// Cleanup
}

void CRC6Block::Decrypt(const CRC6Key &cRC6Key)
{
	Word A = LoWORD(dwData[0]);
	Word B = HiWORD(dwData[0]);
	Word C = LoWORD(dwData[1]);
	Word D = HiWORD(dwData[1]);
	Word q, t, u;

	C -= cRC6Key.GetKey(43);
	A -= cRC6Key.GetKey(42);
	
	for (int i = 19; i >= 0; i--) {
		q = D;
		D = C;
		C = B;
		B = A;
		A = q;
		u = rol(D * (2 * D + 1), 5);
		t = rol(B * (2 * B + 1), 5);
		C = ror(C - cRC6Key.GetKey(i * 2 + 3), t) ^ u;
		A = ror(A - cRC6Key.GetKey(i * 2 + 2), u) ^ t;
	}
	D -= cRC6Key.GetKey(1);
	B -= cRC6Key.GetKey(0);
	dwData[0] = ((Dword)B << 32) | A;
	dwData[1] = ((Dword)D << 32) | C;
	A = B = C = D = 0;						// Cleanup
}

void CRC6Block::SetData(Dword dwValueLeft, Dword dwValueRight)
{
	dwData[0] = dwValueLeft;
	dwData[1] = dwValueRight;
}

Dword CRC6Block::GetData(Word wWord)
{
	return dwData[wWord];
}

void CRC6Block::SetData(const Byte *pbData, Word wLength)
{
	memcpy((Byte *)dwData, pbData, wLength);
	memset((Byte *)dwData + wLength, 0, sizeof(dwData) - wLength);
}

Byte *CRC6Block::GetData()
{
	return (Byte *)dwData;
}
