/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "methods.hpp"

METHOD_ID strMethods[] = 
{
	{ "serpent",		SERPENT,			CSerpentBlock::BLOCKSIZE,				CSerpentKey::KEYSIZE		},
	{ "rc6",				RC6,				CRC6Block::BLOCKSIZE,					CRC6Key::KEYSIZE			},
	{ "loki97",			LOKI97,			CLOKIBlock::BLOCKSIZE,					CLOKIKey::KEYSIZE			},
	{ "mars",			MARS,				CMARSBlock::BLOCKSIZE,					CMARSKey::KEYSIZE			},
	{ "twofish",		TWOFISH,			CTwofishBlock::BLOCKSIZE,				CTwofishKey::KEYSIZE		},
	{ "rijndael",		RIJNDAEL,		CRijndaelBlock::BLOCKSIZE,				CRijndaelKey::KEYSIZE	},
	{ "rc5",				RC5,				CRC5Block::BLOCKSIZE,					CRC5Key::KEYSIZE			},
	{ "tripledes",		TRIPLEDES,		C3DESBlock::BLOCKSIZE,					C3DESKey::KEYSIZE			},
	{ "idea",			IDEA,				CIDEABlock::BLOCKSIZE,					CIDEAKey::KEYSIZE			},
	{ "safer",			SAFER,			CSAFERBlock::BLOCKSIZE,					CSAFERKey::KEYSIZE		},
	{ "feal",			FEAL,				CFEALBlock::BLOCKSIZE,					CFEALKey::KEYSIZE			},
	{ "threeway",		THREEWAY,		C3WayBlock::BLOCKSIZE,					C3WayKey::KEYSIZE			},
	{ "blowfish",		BLOWFISH,		CBlowfishBlock::BLOCKSIZE, 			CBlowfishKey::KEYSIZE	},
	{ "diamond",		DIAMOND,			CDiamondBlock::BLOCKSIZE,				CDiamondKey::KEYSIZE		},
	{ "cast",			CAST,				CCASTBlock::BLOCKSIZE,					CCASTKey::KEYSIZE			},
	{ "shark",			SHARK,			CSHARKBlock::BLOCKSIZE,					CSHARKKey::KEYSIZE		},
	{ "des",				DES,				CDESBlock::BLOCKSIZE,					CDESKey::KEYSIZE			},
	{ "rc2",				RC2,				CRC2Block::BLOCKSIZE,					CRC2Key::KEYSIZE			},
	{ "gost",			GOST,				CGOSTBlock::BLOCKSIZE,					CGOSTKey::KEYSIZE			},
	{ "square",			SQUARE,			CSquareBlock::BLOCKSIZE,				CSquareKey::KEYSIZE			},

	{ "tiger",			TIGER,			CTigerDigest::FINGERPRINTSIZE,		0								},
	{ "md5",				MD5,				CMD5Digest::FINGERPRINTSIZE,			0								},
	{ "ripemd",			RIPEMD,			CRIPEMD160Digest::FINGERPRINTSIZE,	0								},
	{ "sha1",			SHA1,				CSHA1Digest::FINGERPRINTSIZE,			0								},
	{ "haval",			HAVAL,			CHavalDigest::FINGERPRINTSIZE,		0								},
	{ "md2",				MD2,				CMD2Digest::FINGERPRINTSIZE,			0								},
	{ "md4",				MD4,				CMD4Digest::FINGERPRINTSIZE,			0								},

	{ "rsa",				RSA,				0,												0								},
	{ "dh",				DH,				0,												0								},
	{ "mqv",				MQV,				0,												0								},
	{ "rabin",			RABIN,			0,												0								},
	{ "bg",				BG,				0,												0								},
	{ "elgamal",		ELGAMAL,			0,												0								},
};

char *FindName(Byte bID)
{
	for (Word i = 0; i < (sizeof(strMethods) / sizeof(METHOD_ID)); i++)
		if (bID == strMethods[i].bID)
			return strMethods[i].pszName;
	return NULL;
}

// Note that the interval bMin .. bMax is open at both ends.
Byte FindID(char *pszName, Byte bMin, Byte bMax)
{
	for (Word i = 0; i < (sizeof(strMethods) / sizeof(METHOD_ID)); i++)
		if ((NULL != strMethods[i].pszName) &&
			 (0 == strcmp(pszName, strMethods[i].pszName)) &&
			 (bMin < strMethods[i].bID) &&
			 (bMax > strMethods[i].bID))
			return strMethods[i].bID;
	return EMPTY;
}

Word FindParam(Byte bID, Byte bParam)
{
	for (Word i = 0; i < (sizeof(strMethods) / sizeof(METHOD_ID)); i++)
		if (bID == strMethods[i].bID)
			return (bParam == 0 ? strMethods[i].wParam1 : strMethods[i].wParam2);
	return 0;
}

CDigest *NewDigest(Byte bDigest, Byte *pbText, Word wTextSize)
{
	CDigest *pResult = NULL;
	switch (bDigest) {
		case TIGER:
			pResult = new CTigerDigest(pbText, wTextSize);
		break;
		case MD5:
			pResult = new CMD5Digest(pbText, wTextSize);
		break;
		case RIPEMD:
			pResult = new CRIPEMD160Digest(pbText, wTextSize);
		break;
		case SHA1:
			pResult = new CSHA1Digest(pbText, wTextSize);
		break;
		case HAVAL:
			pResult = new CHavalDigest(pbText, wTextSize);
		break;
		case MD2:
			pResult = new CMD2Digest(pbText, wTextSize);
		break;
		case MD4:
			pResult = new CMD4Digest(pbText, wTextSize);
		break;
	}
	return pResult;
}

CAsymmetricKey *NewAsymmetricKey(Byte bType)
{
	CAsymmetricKey *pResult = NULL;
	switch (bType) {
		case RSA:
			pResult = new CRSAKey;
		break;
		case RABIN:
			pResult = new CRabinKey;
		break;
		case BG:
			pResult = new CBGKey;
		break;
		case ELGAMAL:
			pResult = new CElGamalKey;
		break;
		case MQV:
		case DH:
			pResult = new CDSAKey;
		break;
	}
	return pResult;
}

CAsymmetricBlock *NewAsymmetricBlock(Byte bType, CAsymmetricKey *pKey)
{
	CAsymmetricBlock *pResult = NULL;
	switch (bType) {
		case RSA:
			pResult = new CRSABlock((CRSAKey &)*pKey);
		break;
		case RABIN:
			pResult = new CRabinBlock((CRabinKey &)*pKey);
		break;
		case BG:
			pResult = new CBGBlock((CBGKey &)*pKey);
		break;
		case ELGAMAL:
			pResult = new CElGamalBlock((CElGamalKey &)*pKey);
		break;
	}
	return pResult;
}

CBlockKey *NewSymmetricKey(Byte bType, Byte *pbKey, Word wKeySize)
{
	CBlockKey *pResult = NULL;
	switch (bType) {
		case FEAL:
			pResult = new CFEALKey(pbKey, wKeySize);
		break;
		case CAST:
			pResult = new CCASTKey(pbKey, wKeySize);
		break;
		case IDEA:
			pResult = new CIDEAKey(pbKey, wKeySize);
		break;
		case THREEWAY:
			pResult = new C3WayKey(pbKey, wKeySize);
		break;
		case DES:
			pResult = new CDESKey(pbKey, wKeySize);
		break;
		case TRIPLEDES:
			pResult = new C3DESKey(pbKey, wKeySize);
		break;
		case RIJNDAEL:
			pResult = new CRijndaelKey(pbKey, wKeySize);
		break;
		case MARS:
			pResult = new CMARSKey(pbKey, wKeySize);
		break;
		case TWOFISH:
			pResult = new CTwofishKey(pbKey, wKeySize);
		break;
		case SERPENT:
			pResult = new CSerpentKey(pbKey, wKeySize);
		break;
		case RC6:
			pResult = new CRC6Key(pbKey, wKeySize);
		break;
		case RC5:
			pResult = new CRC5Key(pbKey, wKeySize);
		break;
		case RC2:
			pResult = new CRC2Key(pbKey, wKeySize);
		break;
		case SAFER:
			pResult = new CSAFERKey(pbKey, wKeySize);
		break;
		case GOST:
			pResult = new CGOSTKey(pbKey, wKeySize);
		break;
		case DIAMOND:
			pResult = new CDiamondKey(pbKey, wKeySize);
		break;
		case SQUARE:
			pResult = new CSquareKey(pbKey, wKeySize);
		break;
		case SHARK:
			pResult = new CSHARKKey(pbKey, wKeySize);
		break;
		case LOKI97:
			pResult = new CLOKIKey(pbKey, wKeySize);
		break;
		case BLOWFISH:
			pResult = new CBlowfishKey(pbKey, wKeySize);
		break;
	}
	return pResult;
}

CBlockObject *NewSymmetricBlock(Byte bType, Byte *pbIV)
{
	CBlockObject *pResult = NULL;
	switch (bType) {
		case FEAL:
			pResult = new CFEALBlock(pbIV);
		break;
		case CAST:
			pResult = new CCASTBlock(pbIV);
		break;
		case IDEA:
			pResult = new CIDEABlock(pbIV);
		break;
		case THREEWAY:
			pResult = new C3WayBlock(pbIV);
		break;
		case DES:
			pResult = new CDESBlock(pbIV);
		break;
		case TRIPLEDES:
			pResult = new C3DESBlock(pbIV);
		break;
		case RIJNDAEL:
			pResult = new CRijndaelBlock(pbIV);
		break;
		case MARS:
			pResult = new CMARSBlock(pbIV);
		break;
		case TWOFISH:
			pResult = new CTwofishBlock(pbIV);
		break;
		case SERPENT:
			pResult = new CSerpentBlock(pbIV);
		break;
		case RC6:
			pResult = new CRC6Block(pbIV);
		break;
		case RC5:
			pResult = new CRC5Block(pbIV);
		break;
		case RC2:
			pResult = new CRC2Block(pbIV);
		break;
		case SAFER:
			pResult = new CSAFERBlock(pbIV);
		break;
		case GOST:
			pResult = new CGOSTBlock(pbIV);
		break;
		case DIAMOND:
			pResult = new CDiamondBlock(pbIV);
		break;
		case SQUARE:
			pResult = new CSquareBlock(pbIV);
		break;
		case SHARK:
			pResult = new CSHARKBlock(pbIV);
		break;
		case LOKI97:
			pResult = new CLOKIBlock(pbIV);
		break;
		case BLOWFISH:
			pResult = new CBlowfishBlock(pbIV);
		break;
	}
	return pResult;
}

CKeyExchange *NewKeyExchange(Byte bType, CAsymmetricKey *pKey)
{
	CKeyExchange *pResult = NULL;
	switch (bType) {
		case DH:
			pResult = new CDHKeyExchange(((CDSAKey *)pKey)->GetPrime(),
												  ((CDSAKey *)pKey)->GetGenerator(),
												  ((CDSAKey *)pKey)->GetPrivate());
		break;
		case MQV:
			pResult = new CMQVKeyExchange(((CDSAKey *)pKey)->GetPrime(),
												   ((CDSAKey *)pKey)->GetGenerator(),
												   ((CDSAKey *)pKey)->GetPrivate());
		break;
	}
	return pResult;
}
