/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "space.hpp"

CSSSLSocket::CSSSLSocket(int iSocket, Byte bSymmetric, Word wIVSize, Word wKeySize, Byte *pbOthersIV, Byte *pbOthersKey) : pCipherData(NULL), pCipherKey(NULL)
{
	CSSSLSocket::iSocket = iSocket;
	CSSSLSocket::wKeySize = wKeySize;
	CSSSLSocket::wIVSize = wIVSize;
	
	if (NULL == (CSSSLSocket::pbIV = new Byte[wIVSize]))
		throw MEMORYALLOCATION_ERROR;
	if (NULL == (CSSSLSocket::pbKey = new Byte[wKeySize]))
		throw MEMORYALLOCATION_ERROR;

	if (NULL == pbOthersIV)
		GetDefPRNG().GetRandomData(pbIV, wIVSize);
	else
		memcpy(pbIV, pbOthersIV, wIVSize);

	if (NULL == pbOthersKey)
		GetDefPRNG().GetRandomData(pbKey, wKeySize);
	else
		memcpy(pbKey, pbOthersKey, wKeySize);

	pCipherKey = NewSymmetricKey(bSymmetric, pbKey, wKeySize);
	pCipherData = NewSymmetricBlock(bSymmetric, pbIV);

	if (NULL == pCipherKey ||
		 NULL == pCipherData ||
		 NULL == (pCFBCryptographer = new CCFBCryptographer(*pCipherData, *pCipherKey, CSSSLSocket::pbIV)))
		throw MEMORYALLOCATION_ERROR;
}

CSSSLSocket::~CSSSLSocket()
{
	if (NULL != pCFBCryptographer)
		delete pCFBCryptographer;
	if (NULL != pCipherData)
		delete pCipherData;
	if (NULL != pCipherKey)
		delete pCipherKey;
	if (NULL != pbIV)
		delete pbIV;
	if (NULL != pbKey)
		delete pbKey;
}

CSSSLSpace::CSSSLSpace()
{
	pSSSLSockets = NULL;
	iAllSockets = 0;
}

CSSSLSpace::~CSSSLSpace()
{
	for (int i = 0; i < iAllSockets; i++)
		if (NULL != pSSSLSockets[i])
			delete pSSSLSockets[i];
	if (NULL != pSSSLSockets)
		delete pSSSLSockets;
}

bool CSSSLSpace::AddSocket(CSSSLSocket *pNewSSSLSocket)
{
	CSSSLSocket **pNewSSSLSockets = (CSSSLSocket **)realloc(pSSSLSockets, (iAllSockets + 1) * sizeof(CSSSLSocket *));
	if (NULL == pNewSSSLSockets)
		throw MEMORYALLOCATION_ERROR;
	pSSSLSockets = pNewSSSLSockets;
	pSSSLSockets[iAllSockets] = pNewSSSLSocket;
	iAllSockets += 1;
	return true;
}

bool CSSSLSpace::DeleteSocket(int j)
{
	if (0 == iAllSockets)
		return false;
	for (int i = 0; i < iAllSockets; i++)
		if (pSSSLSockets[i]->GetSocket() == j) {
			delete pSSSLSockets[i];
			pSSSLSockets[i] = pSSSLSockets[iAllSockets - 1];
			iAllSockets -= 1;
			return true;
		}
	return false;
}

CSSSLSocket *CSSSLSpace::FindSocket(int j)
{
	for (int i = 0; i < iAllSockets; i++)
		if (pSSSLSockets[i]->GetSocket() == j)
			return pSSSLSockets[i];
	return NULL;
}

CSSSLSpace cSSSLSpace;
