%{
/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "block.hpp"
#include "cast.hpp"
#include "idea.hpp"
#include "3way.hpp"
#include "rijndael.hpp"
#include "des.hpp"
#include "mars.hpp"
#include "twofish.hpp"
#include "serpent.hpp"
#include "rc6.hpp"
#include "rc5.hpp"
#include "rc2.hpp"
#include "safer.hpp"
#include "gost.hpp"
#include "diamond.hpp"
#include "square.hpp"
#include "shark.hpp"
#include "feal.hpp"
#include "loki.hpp"
#include "blowfish.hpp"
#include "md2.hpp"
#include "md4.hpp"
#include "md5.hpp"
#include "ripemd160.hpp"
#include "sha1.hpp"
#include "tiger.hpp"
#include "haval.hpp"

char *ppszDigestNames[] = {
									"MD2",
									"MD4",
									"MD5",
									"RIPEMD-160",
									"SHA-1",
									"Tiger",
									"Haval"
								  };
char *ppszBlockNames[] = {
								  "CAST",
								  "IDEA",
								  "3-Way",
								  "DES",
								  "Rijndael",
								  "MARS",
								  "Twofish",
								  "Serpent",
								  "RC6",
								  "RC5",
								  "RC2",
								  "SAFER",
								  "GOST",
								  "Diamond",
								  "Square",
								  "Shark",
								  "LOKI",
								  "Blowfish",
								  "FEAL"
								 };

#define MD2					0
#define MD4					1
#define MD5					2
#define RIPEMD160			3
#define SHA1				4
#define TIGER				5
#define HAVAL				6

#define CAST			1000
#define IDEA			1001
#define THREEWAY		1002
#define DES				1003
#define RIJNDAEL		1004
#define MARS			1005
#define TWOFISH 	1006
#define SERPENT 	1007
#define RC6				1008
#define RC5				1009
#define RC2				1010
#define SAFER			1011
#define GOST			1012
#define DIAMOND 	1013
#define SQUARE			1014
#define SHARK			1015
#define LOKI			1016
#define BLOWFISH		1017
#define FEAL			1018

#define ctoi(c) 	(strchr(szHex, toupper(c)) - szHex)

char szHex[] = "0123456789ABCDEF";

Byte *pbVerify = NULL;
Byte *pbPlain = NULL;
Byte *pbKey = NULL;
Byte *pbBytes = NULL;
Word wVerifyBytes;
Word wPlainBytes;
Word wKeyBytes;
Word wMessageBytes;
Word wBytes;
int iAlgorithm = -1;
int iTest = 0;
int iPassed = 0;
int iFailed = 0;
int iSkipped = 0;

CDigest *pDigest = NULL;
CBlockKey *pCipherKey = NULL;
CBlockObject *pCipherData = NULL;

void dump(const Byte *pbData, int iLength)
{
	for (int i = 0; i < iLength; i++)
		printf("%02X", pbData[i]);
	printf(" ");
}
%}

HEXDIGIT				[0-9a-fA-F]
SPACE					[ \t]
NEWLINE 			[\n\r]

%%

{HEXDIGIT}+			{
							wBytes = yyleng / 2;
							pbBytes = new Byte[wBytes];
							for (Word i = 0; i < wBytes; i++)
								pbBytes[i] = ctoi(yytext[i * 2]) * 16 + ctoi(yytext[i * 2 + 1]);
							if (iAlgorithm >= 0 && iAlgorithm < 1000) {
								wVerifyBytes = wBytes;
								pbVerify = pbBytes;
							} else if (iAlgorithm >= 1000 && iAlgorithm < 2000) {
								if (NULL == pbPlain) {
									wPlainBytes = wBytes;
									pbPlain = pbBytes;
								} else if (NULL == pbKey) {
									wKeyBytes = wBytes;
									pbKey = pbBytes;
								} else if (NULL == pbVerify) {
									wVerifyBytes = wBytes;
									pbVerify = pbBytes;
								}
							} else {
								if (NULL != pbBytes)
									delete pbBytes;
							}
						}

\".*\"				{
							yytext++;
							wMessageBytes = strlen(yytext) - 1;
							switch (iAlgorithm) {
								case MD2:
									pDigest = new CMD2Digest((const Byte *)yytext, wMessageBytes);
								break;
								case MD4:
									pDigest = new CMD4Digest((const Byte *)yytext, wMessageBytes);
								break;
								case MD5:
									pDigest = new CMD5Digest((const Byte *)yytext, wMessageBytes);
								break;
								case RIPEMD160:
									pDigest = new CRIPEMD160Digest((const Byte *)yytext, wMessageBytes);
								break;
								case SHA1:
									pDigest = new CSHA1Digest((const Byte *)yytext, wMessageBytes);
								break;
								case TIGER:
									pDigest = new CTigerDigest((const Byte *)yytext, wMessageBytes);
								break;
								case HAVAL:
									pDigest = new CHavalDigest((const Byte *)yytext, wMessageBytes);
								break;
							}
						}

{SPACE}+

{NEWLINE}+			{
							if (iAlgorithm >= 0 && iAlgorithm < 1000) {
								if (NULL == pDigest || NULL == pbVerify) {
									printf("%5d: Skipping test.\n", iTest + 1);
									iSkipped += 1;
								} else {
									bool fgMatch = true;
									for (Word i = 0; i < pDigest->GetFingerPrintSize(); i++)
										if (pbVerify[i] != pDigest->GetFingerPrint()[i])
											fgMatch = false;
									if (true == fgMatch)
										iPassed += 1;
									else
										iFailed += 1;
									printf("%5d: Digest '%s' (message size = %d, digest size = %d): %s\n",
											 iTest + 1,
											 ppszDigestNames[iAlgorithm],
											 wMessageBytes,
											 pDigest->GetFingerPrintSize(),
											 fgMatch ? "Ok." : "Failed.");
								}
							} else if (iAlgorithm >= 1000 && iAlgorithm < 2000) {
								switch (iAlgorithm) {
									case CAST:
										pCipherKey = new CCASTKey(pbKey);
										pCipherData = new CCASTBlock(pbPlain);
									break;
									case IDEA:
										pCipherKey = new CIDEAKey(pbKey);
										pCipherData = new CIDEABlock(pbPlain);
									break;
									case THREEWAY:
										pCipherKey = new C3WayKey(pbKey);
										pCipherData = new C3WayBlock(pbPlain);
									break;
									case DES:
										pCipherKey = new CDESKey(pbKey);
										pCipherData = new CDESBlock(pbPlain);
									break;
									case RIJNDAEL:
										pCipherKey = new CRijndaelKey(pbKey, wKeyBytes);
										pCipherData = new CRijndaelBlock(pbPlain);
									break;
									case MARS:
										pCipherKey = new CMARSKey(pbKey, wKeyBytes);
										pCipherData = new CMARSBlock(pbPlain);
									break;
									case TWOFISH:
										pCipherKey = new CTwofishKey(pbKey, wKeyBytes);
										pCipherData = new CTwofishBlock(pbPlain);
									break;
									case SERPENT:
										pCipherKey = new CSerpentKey(pbKey, wKeyBytes);
										pCipherData = new CSerpentBlock(pbPlain);
									break;
									case RC6:
										pCipherKey = new CRC6Key(pbKey, wKeyBytes);
										pCipherData = new CRC6Block(pbPlain);
									break;
									case RC5:
										pCipherKey = new CRC5Key(pbKey, wKeyBytes);
										pCipherData = new CRC5Block(pbPlain);
									break;
									case RC2:
										pCipherKey = new CRC2Key(pbKey, wKeyBytes);
										pCipherData = new CRC2Block(pbPlain);
									break;
									case SAFER:
										pCipherKey = new CSAFERKey(pbKey, wKeyBytes);
										pCipherData = new CSAFERBlock(pbPlain);
									break;
									case GOST:
										pCipherKey = new CGOSTKey(pbKey, wKeyBytes);
										pCipherData = new CGOSTBlock(pbPlain);
									break;
									case DIAMOND:
										pCipherKey = new CDiamondKey(pbKey, wKeyBytes);
										pCipherData = new CDiamondBlock(pbPlain);
									break;
									case SQUARE:
										pCipherKey = new CSquareKey(pbKey, wKeyBytes);
										pCipherData = new CSquareBlock(pbPlain);
									break;
									case SHARK:
										pCipherKey = new CSHARKKey(pbKey, wKeyBytes);
										pCipherData = new CSHARKBlock(pbPlain);
									break;
									case LOKI:
										pCipherKey = new CLOKIKey(pbKey, wKeyBytes);
										pCipherData = new CLOKIBlock(pbPlain);
									break;
									case BLOWFISH:
										pCipherKey = new CBlowfishKey(pbKey, wKeyBytes);
										pCipherData = new CBlowfishBlock(pbPlain);
									break;
									case FEAL:
										pCipherKey = new CFEALKey(pbKey, wKeyBytes);
										pCipherData = new CFEALBlock(pbPlain);
									break;
								}

								if (NULL == pCipherKey ||
									 NULL == pCipherData ||
									 NULL == pbPlain ||
									 NULL == pbKey ||
									 NULL == pbVerify) {
									printf("%5d: Skipping test.\n", iTest + 1);
									iSkipped += 1;
								} else {
									bool fgMatch = true;
									pCipherData->Encrypt(*pCipherKey);
									/*
									dump(pbPlain, pCipherData->GetBlockSize());
									dump(pbKey, pCipherKey->GetKeySize());
									dump(pbVerify, pCipherData->GetBlockSize());
									dump(pCipherData->GetData(), pCipherData->GetBlockSize());
									printf("\n");
									*/
									Word i;
									for (i = 0; i < pCipherData->GetBlockSize(); i++)
									if (pbVerify[i] != pCipherData->GetData()[i])
										fgMatch = false;
									pCipherData->Decrypt(*pCipherKey);
									for (i = 0; i < pCipherData->GetBlockSize(); i++)
										if (pbPlain[i] != pCipherData->GetData()[i])
											fgMatch = false;
									if (true == fgMatch)
										iPassed += 1;
									else
										iFailed += 1;
									printf("%5d: Symmetric algorithm '%s' (block size = %d, key size = %d): %s\n",
											 iTest + 1,
											 ppszBlockNames[iAlgorithm - 1000],
											 pCipherData->GetBlockSize(),
											 pCipherKey->GetKeySize(),
											 fgMatch ? "Ok." : "Failed.");
								}
							}
							if (NULL != pDigest)
								delete pDigest;
							if (NULL != pCipherKey)
								delete pCipherKey;
							if (NULL != pCipherData)
								delete pCipherData;

							if (NULL != pbPlain)
								delete pbPlain;
							if (NULL != pbKey)
								delete pbKey;
							if (NULL != pbVerify)
								delete pbVerify;

							pDigest = NULL;
							pCipherKey = NULL;
							pCipherData = NULL;
							pbPlain = NULL;
							pbKey = NULL;
							pbVerify = NULL;
							iAlgorithm = -1;

							iTest += 1;
						}

(#.*{NEWLINE}+)+

MD2					iAlgorithm = MD2;
MD4					iAlgorithm = MD4;
MD5					iAlgorithm = MD5;
RIPEMD160			iAlgorithm = RIPEMD160;
SHA1					iAlgorithm = SHA1;
TIGER					iAlgorithm = TIGER;
HAVAL					iAlgorithm = HAVAL;

CAST					iAlgorithm = CAST;
IDEA					iAlgorithm = IDEA;
3WAY					iAlgorithm = THREEWAY;
DES					iAlgorithm = DES;
RIJNDAEL				iAlgorithm = RIJNDAEL;
MARS					iAlgorithm = MARS;
TWOFISH 			iAlgorithm = TWOFISH;
SERPENT 			iAlgorithm = SERPENT;
RC6					iAlgorithm = RC6;
RC5					iAlgorithm = RC5;
RC2					iAlgorithm = RC2;
SAFER					iAlgorithm = SAFER;
GOST					iAlgorithm = GOST;
DIAMOND 			iAlgorithm = DIAMOND;
SQUARE				iAlgorithm = SQUARE;
SHARK					iAlgorithm = SHARK;
LOKI					iAlgorithm = LOKI;
BLOWFISH				iAlgorithm = BLOWFISH;
FEAL					iAlgorithm = FEAL;

.

%%

int main(int /*argc*/, char** /*argv*/)
{
	while(yylex() != 0);
	printf("===================\n");
	printf("      Summary	   \n");
	printf("===================\n");
	printf("All tests: %d\n", iTest);
	printf("Passed: %d\n", iPassed);
	printf("Failed: %d\n", iFailed);
	printf("Skipped: %d\n", iSkipped);
	return 0;
}

#ifdef WIN32
int isatty(int)
{
	return 0;
}
#endif
