/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* Getopt for GNU.
   NOTE: getopt is now part of the C library, so if you don't know what
   "Keep this file name-space clean" means, talk to roland@gnu.ai.mit.edu
   before changing it!

   Copyright (C) 1987, 88, 89, 90, 91, 92, 1993
	   Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.	*/

/* This code was specially modified for the WIN32 version of passwdc. The
 * coding style was changed to match the other sources. All comments
 * were stripped because who needs them can refer to the original sources.
 * No changes are planned in this source as well. For UNIX getopt is a part
 * of the standard C library.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "getopt.hpp"

#ifndef HAVE_GETOPT_H

char *optarg = 0;
int optind = 0;
static char *nextchar;
int opterr = 1;
int optopt = '?';

static enum { REQUIRE_ORDER, PERMUTE, RETURN_IN_ORDER } ordering;

static int first_nonopt;
static int last_nonopt;

static void exchange(char **argv)
{
	int nonopts_size = (last_nonopt - first_nonopt) * sizeof(char *);
	char **temp = (char **)malloc(nonopts_size);

// Interchange the two blocks of data in ARGV.
	memcpy((char *)&argv[first_nonopt], (char *)temp, nonopts_size);
	memcpy((char *)&argv[last_nonopt], (char *)&argv[first_nonopt], (optind - last_nonopt) * sizeof (char *));
	memcpy((char *)temp, (char *)&argv[first_nonopt + optind - last_nonopt], nonopts_size);

// Update records for the slots the non-options now occupy.
	first_nonopt += (optind - last_nonopt);
	last_nonopt = optind;
}

int _getopt_internal(int argc, char **argv, const char *optstring, const struct option *longopts, int *longind, int long_only)
{
	int option_index;

	optarg = 0;

	if (optind == 0) {
		first_nonopt = last_nonopt = optind = 1;

      nextchar = NULL;

		if (optstring[0] == '-') {
			ordering = RETURN_IN_ORDER;
			++optstring;
		} else if (optstring[0] == '+') {
			ordering = REQUIRE_ORDER;
			++optstring;
		} else
			ordering = PERMUTE;
	}

	if (nextchar == NULL || *nextchar == '\0') {
		if (ordering == PERMUTE) {
			if (first_nonopt != last_nonopt && last_nonopt != optind)
				exchange ((char **) argv);
			else if (last_nonopt != optind)
				first_nonopt = optind;

			while (optind < argc && (argv[optind][0] != '-' || argv[optind][1] == '\0'))
				optind++;
		  last_nonopt = optind;
		}

		if (optind != argc && !strcmp(argv[optind], "--")) {
			optind++;

			if (first_nonopt != last_nonopt && last_nonopt != optind)
				exchange((char **) argv);
			else if (first_nonopt == last_nonopt)
				first_nonopt = optind;
			last_nonopt = argc;

			optind = argc;
		}

		if (optind == argc) {
			if (first_nonopt != last_nonopt)
				optind = first_nonopt;
			return EOF;
		}

		if ((argv[optind][0] != '-' || argv[optind][1] == '\0')) {
			if (ordering == REQUIRE_ORDER)
				return EOF;
			optarg = argv[optind++];
			return 1;
		}

		nextchar = (argv[optind] + 1 + (longopts != NULL && argv[optind][1] == '-'));
	}

	if (longopts != NULL && ((argv[optind][0] == '-' && (argv[optind][1] == '-' || long_only)))) {
		const struct option *p;
		char *s = nextchar;
		int exact = 0;
		int ambig = 0;
		const struct option *pfound = NULL;
		int indfound;

		indfound = 0;

		while (*s && *s != '=')
			s++;

		for (p = longopts, option_index = 0; p->name; p++, option_index++)
			if (!strncmp(p->name, nextchar, s - nextchar)) {
				if (s - nextchar == (int)strlen(p->name)) {
					pfound = p;
					indfound = option_index;
					exact = 1;
					break;
				} else if (pfound == NULL) {
					pfound = p;
					indfound = option_index;
				} else
					ambig = 1;
			}

		if (ambig && !exact) {
			if (opterr)
				fprintf(stderr, "%s: option `%s' is ambiguous\n", argv[0], argv[optind]);
			nextchar += strlen (nextchar);
			optind++;
			return '?';
		}

		if (pfound != NULL) {
			option_index = indfound;
			optind++;
			if (*s) {
				if (pfound->has_arg)
					optarg = s + 1;
				else {
					if (opterr) {
						if (argv[optind - 1][1] == '-')
							fprintf(stderr, "%s: option `--%s' doesn't allow an argument\n", argv[0], pfound->name);
						else
							fprintf(stderr, "%s: option `%c%s' doesn't allow an argument\n", argv[0], argv[optind - 1][0], pfound->name);
					}
					nextchar += strlen(nextchar);
					return '?';
				}
			} else if (pfound->has_arg == 1) {
				if (optind < argc)
					optarg = argv[optind++];
				else {
					if (opterr)
						fprintf(stderr, "%s: option `%s' requires an argument\n", argv[0], argv[optind - 1]);
					nextchar += strlen(nextchar);
					return optstring[0] == ':' ? ':' : '?';
				}
			}
			nextchar += strlen(nextchar);
			if (longind != NULL)
				*longind = option_index;
			if (pfound->flag) {
				*(pfound->flag) = pfound->val;
				return 0;
			}
			return pfound->val;
		}
		if (!long_only || argv[optind][1] == '-' || strchr(optstring, *nextchar) == NULL) {
			if (opterr) {
				if (argv[optind][1] == '-')
					fprintf(stderr, "%s: unrecognized option `--%s'\n", argv[0], nextchar);
				else
					fprintf(stderr, "%s: unrecognized option `%c%s'\n", argv[0], argv[optind][0], nextchar);
			}
			nextchar = (char *) "";
			optind++;
			return '?';
		}
	}

	{
		char c = *nextchar++;
		char *temp = strchr(optstring, c);

		if (*nextchar == '\0')
			++optind;

		if (temp == NULL || c == ':') {
			if (opterr)
				fprintf(stderr, "%s: illegal option -- %c\n", argv[0], c);
			optopt = c;
			return '?';
		}
		if (temp[1] == ':') {
			if (temp[2] == ':') {
				if (*nextchar != '\0') {
					optarg = nextchar;
					optind++;
				} else
					optarg = 0;
				nextchar = NULL;
			} else {
				if (*nextchar != '\0') {
					optarg = nextchar;
					optind++;
				} else if (optind == argc) {
					if (opterr)
						fprintf(stderr, "%s: option requires an argument -- %c\n", argv[0], c);
					optopt = c;
					if (optstring[0] == ':')
						c = ':';
					else
						c = '?';
				} else
					optarg = argv[optind++];
				nextchar = NULL;
			}
		}
		return c;
	}
}

int getopt_long(int argc, char **argv, const char *shortopts, const struct option *longopts, int *longind)
{
	 return _getopt_internal(argc, argv, shortopts, longopts, longind, 0);
}

int getopt(int argc, char **argv, const char *optstring)
{
	return _getopt_internal(argc,
									argv,
									optstring,
			   (const struct option *)0,
			   (int *)0,
			   0);
}

int getopt_long_only(int argc,
							char **argv,
							const char *options,
							const struct option *long_options,
							int *opt_index)
{
  return _getopt_internal(argc, argv, options, long_options, opt_index, 1);
}

#endif // HAVE_GETOPT_H
