/*
 * Copyright 1999, Alexander Feldman <alex@varna.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Alexander Feldman nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ALEXANDER FELDMAN AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL ALEXANDER FELDMAN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef WIN32
#include <unistd.h>
#endif

#include "rsa.hpp"
#include "version.hpp"
#include "initrnd.hpp"
#include "getopt.hpp"

char szInputFile[256] = "";
char szOutputFile[256] = "";
char szPublicFile[256] = "";

Word wModulusSize = 768;
Word wExponent = 65537;

bool fgBinary = false;
bool fgPublic = false;
bool fgCheck = false;

void callback1()
{
	static bool f;
	f = !f;
	fprintf(stderr, f ? "<>\b\b" : "><\b\b");
}

void callback2()
{
	fprintf(stderr, ">");
}

void callback3()
{
	fprintf(stderr, "\n");
}

void help()
{
	fprintf(stderr,
			  "usage: rsakeys [-m <bits>] [-o <file>] [-p [<file>]] [-e <3 | 65537>] [-b]\n"
			  "	  rsakeys [-bc] [-i <file>] [-p [<file>]]\n"
			  "	  rsakeys -v\n\n"
			  "	  -m, --modulus	  create key with <bits> (default is 768) bit modulus\n"
			  "	  -i, --in	  read private key from <file>\n"
			  "	  -o, --out	  write private key to <file>\n"
			  "	  -e, --exponent  use <value> for the E value (default is 0x10001)\n"
			  "	  -c, --check	  check the target private key\n"
			  "	  -p, --public	  extract the public key to <file> (stdout if not given)\n"
			  "	  -b, --binary	  do not base 64 encode keys\n"
			  "	  -v, --version	  show version and exit\n");
}

void version()
{
	fprintf(stderr, szVersion);
}

int main(int argc, char **argv)
{
	int iResult = 0;

	static struct option long_options[] =
	{
		{ "modulus", required_argument, 0, 'm' },
		{ "in", required_argument, 0, 'i' },
		{ "out", required_argument, 0, 'o' },
		{ "check", no_argument, 0, 'c' },
		{ "public", optional_argument, 0, 'p' },
		{ "exponent", required_argument, 0, 'e' },
		{ "binary", no_argument, 0, 'b' },
		{ "version", no_argument, 0, 'v' },
		{ 0, 0, 0, 0 }
	};

	int w;
	while (-1 != (w = getopt_long_only(argc, argv, "", long_options, NULL)))
		switch (w) {
			case 'm':
				wModulusSize = (Word)atoi(optarg);
				if ((wModulusSize < 512) || (wModulusSize > 3072)) {
					fprintf(stderr, "Invalid modulus size (not in the range 512 - 3072) specified: %s...\n", optarg);
					return 1;
				}
			break;
			case 'i':
				strncpy(szInputFile, optarg, sizeof(szInputFile));
			break;
			case 'o':
				strncpy(szOutputFile, optarg, sizeof(szOutputFile));
			break;
			case 'c':
				fgCheck = true;
			break;
			case 'p':
				fgPublic = true;
				if ((NULL != argv[optind]) && ('-' != *argv[optind])) {
					strncpy(szPublicFile, argv[optind], sizeof(szPublicFile));
					optind += 1;
				}
			break;
			case 'e':
				if (0 != strcmp("3", optarg) &&
					 0 != strcmp("65537", optarg)) {
					fprintf(stderr, "Invalid public exponent specified: %s...\n", optarg);
					return 1;
				}
				wExponent = (Word)atoi(optarg);
			break;
			case 'b':
				fgBinary = true;
			break;
			case 'v':
				version();
				return 0;						// Successful exit
			break;
			default:
				help();
				return 0;						// Successful exit
			break;
		}
	argc -= optind;							// optind is declared in <unistd.h>
	argv += optind;

	if (0 != argc) {
		help();
		return 0;
	}

	if ((true != fgCheck) && ('\0' == szInputFile[0])) {

		InitDefPRNG();

		fprintf(stderr, "Generating RSA key with %d bit modulus...\n", wModulusSize);

		pfnCallback1 = callback1;
		pfnCallback2 = callback2;
		pfnCallback3 = callback3;
		CRSAKey cRSAKey(wModulusSize, wExponent);

		try
		{
			cRSAKey.WritePrivate('\0' == szOutputFile[0] ? NULL : szOutputFile, !fgBinary);
			if (true == fgPublic)
				cRSAKey.WritePublic('\0' == szPublicFile[0] ? NULL : szPublicFile, !fgBinary);
		}
		catch (const char *pszMessage)
		{
			fprintf(stderr, "%s\n", pszMessage);
			iResult = 1;
		}

	} else {

		CRSAKey cRSAKey;
		try
		{
			cRSAKey.ReadPrivate('\0' == szInputFile[0] ? NULL : szInputFile, !fgBinary);
		}
		catch (const char *pszMessage)
		{
			fprintf(stderr, "%s\n", pszMessage);
			return 1;
		}

		if (true == fgCheck) {
			fprintf(stderr, "Checking RSA key...\n");

			Word wCheck = cRSAKey.Check();
			if (0 == wCheck)
				fprintf(stderr, "%s\n", ppszRSACheckResults[0]);
			Word i = 1;
			while (0 != wCheck) {
				if (1 == (wCheck & 1))
					fprintf(stderr, "%s\n", ppszRSACheckResults[i]);
				wCheck >>= 1;
				i += 1;
			}
		}
		try
		{
			if (true == fgPublic)
				cRSAKey.WritePublic('\0' == szPublicFile[0] ? NULL : szPublicFile, !fgBinary);
		}
		catch (const char *pszMessage)
		{
			fprintf(stderr, "%s\n", pszMessage);
			iResult = 1;
		}

	}

	return iResult;
}
