/* Copyright 1994, Roger Smith for Sterling Software and NASA-Ames Research Center
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of NASA and Sterling Software (collectively 
 * or individually) are not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  NASA and Sterling Software makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * NASA & STERLING SOFTWARE DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL NASA & STERLING SOFTWARE
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * 1994 Revision
 * Author:           Roger Smith, Sterling Software @ NASA-Ames Research Center
 *                   Moffett Field, California, rsmith@proteus.arc.nasa.gov
 */
#include <stdio.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xlib.h>
#include "system.h"
#include "StripCharP.h"
#include "TimeCharP.h"

extern char *calloc(), *malloc();
extern void exit();
extern Widget perfmon[NUM_GRAPHS], timechart, labelBox, pappaBox;
extern void draw_string();
extern void draw_graph();
extern void update_stats();
extern AppData appData;

static void adjust_timeout();
extern int dbg;
extern int version_major, version_minor, version_dot;

/* This routine interprets the key that was pressed on top of the window. 
 * It returns TRUE if Quit has been selected, signaling that the program
 * is done.
 */

void HandleKey(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
#define STRBUFSIZE 64
    char strbuf[STRBUFSIZE];
    int i, keycode, length = 0;
    Window rwin, chwin;
    int rx, ry, wx,wy;
    unsigned int key;
/*    PerfmonWidget pw = (PerfmonWidget) w;*/

    length = XLookupString((XKeyEvent *)event, strbuf, STRBUFSIZE, (KeySym *)&keycode, NULL);
    switch (keycode) {
    case 'Q':
    case 'q':
	exit(0);
	break;
    case 's':
	/* Maybe the following adjustments should be calls to SetValues? */
	adjust_timeout(SMALL_ADD_TIME);
	break;
    case 'S':
	adjust_timeout(LARGE_ADD_TIME);
	break;
    case 'f':
	adjust_timeout(SMALL_SUBTRACT_TIME);
	break;
    case 'F':
	adjust_timeout(LARGE_SUBTRACT_TIME);
	break;
    case 'd':
    case 'D':
    	if( dbg ) dbg = FALSE;
    	else dbg = TRUE;
    	break;
    case 'v':
    case 'V':
    	fprintf( stderr, "xperfmon++ version %s, Linux version %d.%d.%d\n",
    				XPERFVERSION, version_major, version_minor, version_dot );
		fflush( stderr );
		break;
    case 'C':
    case 'c':
	for ( i=0; i<NUM_GRAPHS; i++ ) {
	    if ( perfmon[i] ) {
		PerfChartWidget pw = (PerfChartWidget)perfmon[i];
		       pw->strip_chart.hi_max = 0;
		       pw->strip_chart.lo_min = 99999;
	    }
	}
	break;
    case '?':
	puts("Q|q - Quit");
	puts("L|l - Print out a chart of current limit settings");
	puts("M|m - Print out a chart of lifetime min and max values");
	puts("C|c - Clear the table of lifetime min and max values");
	puts("D|d - Toggle debugging printout");
	puts("H|h - Print a description of the graph the pointer is over");
	puts("s - Decrease update interval (slower) by a small amount");
	puts("S - Decrease update interval (slower) by a large amount");
	puts("f - Increase update interval (faster) by a small amount");
	puts("F - Increase update interval (faster) by a large amount");
	puts("? - Help");
	break;
    case 'L':
    case 'l':			/* print out a chart of current limit settings */
	printf("\n%10s %5s %5s %5s %5s\n",
	       "Widget", "Hi ", "Hi ", "Low", "Low ");
	printf("%10s %5s %5s %5s %5s\n",
	       "Name ", "Warn", "Alarm", "Warn", "Alarm");
	printf("%10s %5s %5s %5s %5s\n",
	       "------", "-----","-----","-----","-----");
	for ( i=0; i<NUM_GRAPHS; i++ ) {
	    if ( perfmon[i] ) {
		PerfChartWidget pw = (PerfChartWidget)perfmon[i];
		printf("%10s %5d %5d %5d %5d\n",
		       widgetLabels[i],
		       pw->strip_chart.highWarn,
		       pw->strip_chart.highAlarm,
		       pw->strip_chart.lowWarn,
		       pw->strip_chart.lowAlarm);
	    }
	}
	break;
    case 'm':
    case 'M':
	printf("\n%10s %5s %5s\n",
	       "Widget", "Hi ", "Low");
	printf("%10s %5s %5s\n",
	       "Name ", "Max", "Min");
	printf("%10s %5s %5s\n",
	       "------", "-----","-----");
	for ( i=0; i<NUM_GRAPHS; i++ ) {
	    if ( perfmon[i] ) {
		PerfChartWidget pw = (PerfChartWidget)perfmon[i];
		printf("%10s %5d %5d\n",
		       widgetLabels[i],
		       (int)pw->strip_chart.hi_max,
		       (int)pw->strip_chart.lo_min);
	    }
	}
	break;
    case 'h':
    case 'H':

	XQueryPointer(XtDisplay(appData.toplevel),XtWindow(pappaBox),&rwin,&chwin,&rx,&ry,
		      &wx,&wy,&key);
	if ( chwin ) {
	    Bool persec=FALSE;
	    Widget pw = XtWindowToWidget(XtDisplay(appData.toplevel), chwin);
	    if ( !pw ) 
		break;
	    for ( i=0; i<NUM_GRAPHS; i++ ) {
		if ( pw == perfmon[i] ) 
		    break;
	    }
	    switch ( i ) {
	    case LOAD_AVG:
		printf("Shows the system 1 minute load average *100. This value is\n");
		printf("updated by the system every 5 seconds. To obtain the actual\n");
		printf("value, insert a decimal point two places to the left.\n");
		break;
	    case USR_CPU:
		printf("\nShows the PERCENT of wall clock time spent in User CPU mode.\n");
		break;
	    case SYS_CPU:
		printf("\nShows the PERCENT of wall clock time spent in System CPU mode.\n");
		break;
	    case IDL_CPU:
		printf("\nShows the PERCENT of wall clock time spend in Idle CPU mode.\n");
		break;
	    case FRE_MEM:
		printf("\nShows the number of Kilobytes of memory free and available.\n");
		break;
	    case FREE_SWAP:
		printf("\nShows the number of Kilobytes of swap free and available.\n");
		break;
	    case SWAP_IN:
		printf("\nShows the number of Pages Swapped In PER SECOND.\n");
		persec=TRUE;
		break;
	    case SWAP_OUT:
		printf("\nShows the number of Pages Swapped Out PER SECOND.\n");
		persec=TRUE;
		break;
	    case PAGE_IN:
		printf("\nShows the number of Kilobytes Paged In PER SECOND.\n");
		persec=TRUE;
		break;
	    case PAGE_OUT:
		printf("\nShows the number of Kilobytes Paged Out PER SECOND.\n");
		persec=TRUE;
		break;
	    case DSK_XFR:
		printf("\nShows the number of Disk Transfers PER SECOND for all\n");
		printf("magnetic disks in the system.\n");
		persec=TRUE;
		break;
	    case INTRPTS:
		printf("\nShows the number of system Interrupts PER SECOND.\n");
		persec=TRUE;
		break;
/*
	    case SYSTEM_CALLS:
		printf("\nShows the number of system CALLS executed PER SECOND.\n");
		persec=TRUE;
		break;
*/
	    case CPU_SWITCH:
		printf("\nShows the number of CPU Context Switches executed PER SECOND.\n");
		persec=TRUE;
		break;
	    case INP_PKT:
		printf("\nShows the number of network Input Packets received PER SECOND\n");
		printf("on all network interfaces.\n");
		persec=TRUE;
		break;
	    case OUT_PKT:
		printf("\nShows the number of network Output Packets sent PER SECOND\n");
		printf("for all network interfaces.\n");
		persec=TRUE;
		break;
	    case COL_PKT:
		printf("\nShows the PERCENT network Collision Rate PER SECOND for\n");
		printf("all network interfaces..\n");
		persec=TRUE;
		break;
/*SC
	    case NFS_CLT:
		printf("\nShows the number of NFS Client Calls processed PER SECOND.\n");
		persec=TRUE;
		break;
	    case NFS_SRV:
		printf("\nShows the number of NFS Server Calls processed PER SECOND.\n");
		persec=TRUE;
		break;
SC*/
	    }
	    if ( persec && appData.fast) {
		printf("NOTE: When the interval is in milliseconds (-fast is set) then all\n");
		printf("PER SECOND values are converted to interval counts, i.e., the current\n");
		printf("value is the actual count for the interval period.\n");
	    }
	}
	break;
    default:
	break;
    }
}

/* when the master application timeout expires the following routine
 * will kick off all active graphs in 1 millisecond.*/

void start_graphs(client_data, id)
XtPointer client_data;
XtIntervalId *id;
{
    int i;
    TimeChartWidget tc = (TimeChartWidget)timechart;

    tc->time_chart.interval_id =
	XtAppAddTimeOut(XtWidgetToApplicationContext( (Widget) tc),
			1, draw_string, (caddr_t) tc);
    update_stats();

    for ( i=0; i<NUM_GRAPHS; i++ ) {
	if ( perfmon[i] ) {
	    PerfChartWidget pc = (PerfChartWidget)perfmon[i];
	    pc->strip_chart.interval_id =
		XtAppAddTimeOut(XtWidgetToApplicationContext( (Widget) pc),
				1, draw_graph, (caddr_t) pc);
	}
    }
    appData.interval_id = XtAppAddTimeOut(appData.app_context,
		    appData.interval*appData.ms_per_sec, start_graphs, (caddr_t) appData.toplevel);
}

static void
adjust_timeout(delta)
int delta;
{
    int i;
    char hostname[132], *c;

    if ( appData.ms_per_sec == 1 )
	delta *= 100;		/* increments of 100 milliseconds minimum */

    if ( (appData.interval+delta)*appData.ms_per_sec >= 100 ) {	/* no less than .1 secs */
	appData.interval += delta;

	XtRemoveTimeOut(appData.interval_id);
		    
	appData.interval_id =
	    XtAppAddTimeOut(appData.app_context,
			    appData.interval*appData.ms_per_sec, start_graphs, (caddr_t) appData.toplevel);
    }

    (void)gethostname(hostname, sizeof(hostname));
    c = (char *) ((long) &hostname[0] + (int) strlen(hostname));
    sprintf(c, "\nUpdate Interval = %5.1f secs", 
	    (float)(appData.interval*appData.ms_per_sec)/1000.0);
    XtVaSetValues(labelBox, XtNlabel, hostname, NULL);
}
