// Stream buffer classes -*- C++ -*-

// Copyright (C) 1997,1998 Cygnus Solutions
//
// This file is part of the libstdc++ version 3 distribution.
//
// This software is a copyrighted work licensed under the terms of the
// Cygnus libstdc++ license. Please consult the file LICENSE.STD for
// details.

//
// ISO C++ working draft paper: 27.5  Stream buffers
//

#ifndef _CPP_STREAMBUF
#define _CPP_STREAMBUF	1

#include <bits/c++config.h>
#include <bits/std_ios.h>

#ifdef _G_USE_LIBIO
//#include <libio.h>  /* XXX non-standard */
#endif

namespace std {

  // For the order of the virtual functions in this class we have not
  // much choice.  It must match the order of entries in the jump
  // table of the _IO_FILE structure.

  // note this stuff doesn't need to be a template;
  //   the out-of-line members can be moved into streambuf.cc, later.
  template <class _Dummy = int>
  struct _Streambuf_base
#ifdef _G_USE_LIBIO
    : public _IO_FILE
#endif
  {
  protected:
#ifdef _G_USE_LIBIO
    // LIBIO should be the basis only of a filebuf<char>
    // specialization, if that.
    const void*& _M_vtable ()
      { return *(const void**) ((_IO_FILE*) this + 1); }
#endif
    virtual ~_Streambuf_base();
    locale _M_locale;
    virtual void imbue (const locale& __loc);

  public:
    locale pubimbue (const locale& __loc);
    locale getloc () const;
  };

  template <class _Dummy>
    locale
    _Streambuf_base<_Dummy>::pubimbue (const locale& __loc)
  {
    locale __tmp(getloc());
    this->imbue (__loc);
    _M_locale = __loc;
    return __tmp;
  }

  template <class _Dummy>
    locale
    _Streambuf_base<_Dummy>::getloc () const
  {
    return _M_locale; //XXX not fully implemented
  }

  template <class _Dummy>
    void
    _Streambuf_base<_Dummy>::imbue (const locale& __loc)
  {  }

  template <class _Dummy>
    _Streambuf_base<_Dummy>::~_Streambuf_base() {}

  // basic_streambuf

  template <class _CharT, class _Traits>
  class basic_streambuf : public _Streambuf_base<>
  {
  public:
    // Types:
    typedef _CharT char_type;
    typedef typename _Traits::int_type int_type;
    typedef typename _Traits::pos_type pos_type;
    typedef typename _Traits::off_type off_type;
    typedef _Traits traits_type;

    virtual ~basic_streambuf ();

  private:
#ifndef _G_USE_LIBIO
#undef   _IO_read_ptr
#undef   _IO_read_end
#undef   _IO_read_base
#undef   _IO_write_base
#undef   _IO_write_ptr
#undef   _IO_write_end
#undef   _IO_buf_base
#undef   _IO_buf_end
#undef   _IO_save_base
#undef   _IO_backup_base
#undef   _IO_save_end
#undef   _IO_file_flags

    _CharT* _IO_read_ptr;	/* Current read pointer */
    _CharT* _IO_read_end;	/* End of get area. */
    _CharT* _IO_read_base;    /* Start of putback+get area. */
    _CharT* _IO_write_base;   /* Start of put area. */
    _CharT* _IO_write_ptr;    /* Current put pointer. */
    _CharT* _IO_write_end;    /* End of put area. */
    _CharT* _IO_buf_base;	    /* Start of reserve area. */
    _CharT* _IO_buf_end;	    /* End of reserve area. */
    /* The following fields are used to support backing up and undo. */
    _CharT* _IO_save_base;    /* Pointer to start of non-current get area. */
    _CharT* _IO_backup_base;  /*  first valid character of backup area */
    _CharT* _IO_save_end; /* Pointer to end of non-current get area. */
    int     _IO_file_flags;
#endif
  public:
    //  locales:
    locale   pubimbue(const locale &loc)
      { return _Streambuf_base::pubimbue (loc); }

    locale   getloc() const
      { return _Streambuf_base::getloc (); }

    // Buffer and positioning:
    basic_streambuf<char_type,_Traits>*
    pubsetbuf (char_type* __s, streamsize __n) { return setbuf (__s, __n); }

    pos_type pubseekoff (off_type __off, ios_base::seekdir __way,
			 ios_base::openmode __which = (ios_base::in
						     | ios_base::out))
      { return seekoff (__off, __way, __which); }

    pos_type pubseekpos (pos_type __sp,
			 ios_base::openmode __which = (ios_base::in
						     | ios_base::out))
      { return seekpos (__sp, __which); }

    int pubsync () { return sync (); }

    // Get and put areas:
    // Get area:
    streamsize in_avail () { return _IO_read_end - _IO_read_ptr; }

  private:
    int_type _M_snextc_helper ();

  public:

    int_type snextc ()
      {
	return (_IO_read_end - _IO_read_ptr > 1) ?
	  _Traits::to_int_type(*++_IO_read_ptr) : _M_snextc_helper ();
      }

    int_type sbumpc ()
      {
        return  (_IO_read_ptr < _IO_read_end) ?
	  _Traits::to_int_type (*_IO_read_ptr++) : uflow ();
      }

    int_type sgetc ()
      {
         return (_IO_read_ptr < _IO_read_end) ?
	   _Traits::to_int_type (*_IO_read_ptr) : underflow ();
      }

    // XXX deprecated, wrap in an #ifdef so we can warn pedantically.
    void stossc () { _M_stossc (); }

    void _M_stossc ()
      {
	 if (_IO_read_ptr < _IO_read_end) ++_IO_read_ptr;
	 else uflow ();
      }

    streamsize sgetn (char_type* __s, streamsize __n)
      { return xsgetn(__s, __n); }

    // Putback:
    int_type sputbackc (char_type __c);
    int_type sungetc ();

    // Put area:
    int_type sputc (char_type __c)
      {
        return (_IO_write_ptr < _IO_write_end) ?
	  _Traits::to_int_type(*_IO_write_ptr++ = __c) : overflow(__c);
      }

    streamsize sputn (const char_type* __s, streamsize __n)
      { return xsputn (__s, __n); }

  protected:
    basic_streambuf ();

    // Get area:
    char_type* eback () const
      { return _IO_file_flags & _IO_IN_BACKUP
	  ? _IO_save_base : _IO_read_base; }
    char_type* gptr ()  const
      { return _IO_file_flags & _IO_IN_BACKUP ? _IO_save_base : _IO_read_ptr; }
    char_type* egptr () const
      { return _IO_file_flags & _IO_IN_BACKUP ? _IO_save_end : _IO_read_end; }
    void gbump (int __n)
      { _IO_file_flags & _IO_IN_BACKUP
	  ? (_IO_save_base += __n) : (_IO_read_ptr += __n); }
    void setg (char_type* __gbeg, char_type* __gnext, char_type* __gend)
      {
#ifdef _G_USE_LIBIO  /* XXX eh? */
	if (_IO_file_flags & _IO_IN_BACKUP)
	  _IO_free_backup_area (this);
#endif
	_IO_read_base = __gbeg;
	_IO_read_ptr = __gnext;
	_IO_read_end = __gend;
      }

    // Put area:
    char_type* pbase () const { return _IO_write_base; }
    char_type* pptr () const { return _IO_write_ptr; }
    char_type* epptr () const { return _IO_write_end; }
    void pbump (int __n) { _IO_write_ptr += __n; }
    void setp (char_type* __pbeg, char_type* __pend)
      { _IO_write_base = _IO_write_ptr = __pbeg; _IO_write_end = __pend; }


    // Virtual functions:

    // Put area:
    virtual int_type overflow (int_type __c = _Traits::eof ());

    // Get area:
    virtual int_type underflow ();
    virtual int_type uflow ();

    // Putback:
    virtual int_type pbackfail (int_type __c = _Traits::eof ());

    // Put area:
    virtual streamsize xsputn (const char_type* __s, streamsize __n);

    // Get area:
    virtual streamsize xsgetn (char_type* __s, streamsize __n);


    // Buffer management and positioning:
    virtual pos_type seekoff (off_type __off, ios_base::seekdir __way,
			      ios_base::openmode __which = (ios_base::in
							  | ios_base::out));
    virtual pos_type seekpos (pos_type __sp,
			      ios_base::openmode __which = (ios_base::in
							  | ios_base::out));

    virtual basic_streambuf<char_type,_Traits>* setbuf (char_type* __s,
							streamsize __n);

    virtual int sync ();


#if 0
    // The next functions are defined in the jump table but are not
    // required by the ISO C++ streambuf implementation.
    virtual int doallocate ();

    virtual streamsize sys_read (char* __buf, streamsize __size);
    virtual streamsize sys_write (const char* __buf, streamsize);
    virtual pos_type sys_seek (streamoff __off, ios_base::seekdir __way);
    virtual int sys_close ();
    virtual int sys_stat (void* __st_buf); // Actually, a (struct stat*)
#endif


    // The following fields are required by the ISO C++ standard but
    // they are not present in the pre-templatized I/O implementation.
    // Since we must take care for the compatibility of the C code so
    // put them here at the end.

    // Get area:
    virtual int showmanyc ();

    friend /* template<> */ class basic_istream<_CharT,_Traits>;
    friend /* template<> */ class basic_ostream<_CharT,_Traits>;
    friend /* template<> */ class istreambuf_iterator<_CharT,_Traits>;
    friend /* template<> */ class ostreambuf_iterator<_CharT,_Traits>;
  };

  template <class _CharT, class _Traits>
    typename _Traits::int_type
    basic_streambuf<_CharT,_Traits>::_M_snextc_helper ()
      {
	if (_IO_read_ptr == _IO_read_end)
	  {
	    if (_Traits::eq_int_type (uflow (),_Traits::eof ()))
	      return _Traits::eof();
	    if (_IO_read_ptr != _IO_read_end)
	      return _Traits::to_int_type(*_IO_read_ptr);
	  }
	else
	  _IO_read_ptr = _IO_read_end;
	return underflow ();
      }

} // namespace std

#ifdef _G_NO_TEMPLATE_EXPORT
# define export
#ifndef _POSIX_SOURCE
#define _POSIX_SOURCE
#endif
// #include <bits/std_cerrno.h>     // XXX delete these if unused.
// #include <bits/std_streambuf.h>
// #include <bits/std_string.h>
#include <bits/streambuf.tcc>
#endif

#endif	/* _CPP_STREAMBUF */
