use strict;

package HTML::FormFu::Deflator::Strftime;
$HTML::FormFu::Deflator::Strftime::VERSION = '2.07';
# ABSTRACT: Strftime deflator

use Moose;
use MooseX::Attribute::Chained;
extends 'HTML::FormFu::Deflator';

has strftime => ( is => 'rw', traits => ['Chained'] );

sub deflator {
    my ( $self, $value ) = @_;

    my $return;

    eval {
        my $locale = $self->locale;

        $value->set_locale($locale) if defined $locale;
    };

    eval { $return = $value->strftime( $self->strftime ) };

    if ($@) {
        $return = $value;
    }

    return $return;
}

__PACKAGE__->meta->make_immutable;

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

HTML::FormFu::Deflator::Strftime - Strftime deflator

=head1 VERSION

version 2.07

=head1 SYNOPSIS

    $form->deflator( Strftime => 'start_time' )
        ->strftime( '%d/%m/%Y' );

    ---
    elements:
        - type: Text
          inflators:
              - type: DateTime
                parser:
                    strptime: "%Y/%m/%d"
          deflator:
              - type: Strftime
                strftime: "%Y/%m/%d"

=head1 DESCRIPTION

Strftime deflator for L<DateTime> objects.

When you redisplay a form to the user following an invalid submission,
any fields with DateTime inflators will stringify to something like
'1970-01-01T00:00:00'. In most cases it makes more sense to use the same
format you've asked the user for. This deflator allows you to specify a
more suitable and user-friendly format.

This deflator calls L<DateTime>'s C<strftime> method. Possible values for
the format string are documented at
L<http://search.cpan.org/dist/DateTime/lib/DateTime.pm#strftime_Patterns>.

If you set the form's locale (see L<HTML::FormFu/locale>) this is set on the DateTime object. Now you can use C<%x> to get the default date or C<%X> for the default time for the object's locale.

=head1 AUTHOR

Carl Franks, C<cfranks@cpan.org>

=head1 LICENSE

This library is free software, you can redistribute it and/or modify it under
the same terms as Perl itself.

=head1 AUTHOR

Carl Franks <cpan@fireartist.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018 by Carl Franks.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
