/*++
/* NAME
/*	lmtp_trouble 3
/* SUMMARY
/*	error handler policies
/* SYNOPSIS
/*	#include "lmtp.h"
/*
/*	int	lmtp_sess_fail(state, why)
/*	SMTP_STATE *state;
/*	DSN_BUF	*why;
/*
/*	int	lmtp_site_fail(state, mta_name, resp, format, ...)
/*	LMTP_STATE *state;
/*	const char *mta_name;
/*	LMTP_RESP *resp;
/*	const char *format;
/*
/*	int	lmtp_mesg_fail(state, mta_name, resp, format, ...)
/*	LMTP_STATE *state;
/*	const char *mta_name;
/*	LMTP_RESP *resp;
/*	const char *format;
/*
/*	void	lmtp_rcpt_fail(state, mta_name, resp, recipient, format, ...)
/*	LMTP_STATE *state;
/*	const char *mta_name;
/*	LMTP_RESP *resp;
/*	RECIPIENT *recipient;
/*	const char *format;
/*
/*	int	lmtp_stream_except(state, exception, description)
/*	LMTP_STATE *state;
/*	int	exception;
/*	const char *description;
/* DESCRIPTION
/*	This module handles all non-fatal errors that can happen while
/*	attempting to deliver mail via LMTP, and implements the policy
/*	of how to deal with the error. Depending on the nature of
/*	the problem, delivery of a single message is deferred, delivery
/*	of all messages to the same domain is deferred, or one or more
/*	recipients are given up as non-deliverable and a bounce log is
/*	updated.
/*
/*	In addition, when an unexpected response code is seen such
/*	as 3xx where only 4xx or 5xx are expected, or any error code
/*	that suggests a syntax error or something similar, the
/*	protocol error flag is set so that the postmaster receives
/*	a transcript of the session. No notification is generated for
/*	what appear to be configuration errors - very likely, they
/*	would suffer the same problem and just cause more trouble.
/*
/*	lmtp_sess_fail() takes a pre-formatted error report after
/*	failure to complete some protocol handshake.  The policy is
/*	as with lmtp_site_fail().
/*
/*	lmtp_site_fail() handles the case where the program fails to
/*	complete some protocol handshake: the server is not reachable,
/*	is not running, does not want talk to us, or we talk to ourselves.
/*	The \fIcode\fR gives an error status code; the \fIformat\fR
/*	argument gives a textual description.  The policy is: soft
/*	error: defer delivery of all messages to this domain; hard
/*	error: bounce all recipients of this message.
/*	The result is non-zero.
/*
/*	lmtp_mesg_fail() handles the case where the lmtp server
/*	does not accept the sender address or the message data.
/*	The policy is: soft errors: defer delivery of this message;
/*	hard error: bounce all recipients of this message.
/*	The result is non-zero.
/*
/*	lmtp_rcpt_fail() handles the case where a recipient is not
/*	accepted by the server for reasons other than that the server
/*	recipient limit is reached. The policy is: soft error: defer
/*	delivery to this recipient; hard error: bounce this recipient.
/*
/*	lmtp_stream_except() handles the exceptions generated by
/*	the smtp_stream(3) module (i.e. timeouts and I/O errors).
/*	The \fIexception\fR argument specifies the type of problem.
/*	The \fIdescription\fR argument describes at what stage of
/*	the LMTP dialog the problem happened. The policy is to defer
/*	delivery of all messages to the same domain. The result is non-zero.
/*
/*	Arguments:
/* .IP state
/*	LMTP client state per delivery request.
/* .IP resp
/*	Server response including reply code and text.
/* .IP recipient
/*	Undeliverable recipient address information.
/* .IP format
/*	Human-readable description of why mail is not deliverable.
/* DIAGNOSTICS
/*	Panic: unknown exception code.
/* SEE ALSO
/*	lmtp_proto(3) lmtp high-level protocol
/*	smtp_stream(3) lmtp low-level protocol
/*	defer(3) basic message defer interface
/*	bounce(3) basic message bounce interface
/* LICENSE
/* .ad
/* .fi
/*	The Secure Mailer license must be distributed with this software.
/* AUTHOR(S)
/*	Wietse Venema
/*	IBM T.J. Watson Research
/*	P.O. Box 704
/*	Yorktown Heights, NY 10598, USA
/*
/*	Alterations for LMTP by:
/*	Philip A. Prindeville
/*	Mirapoint, Inc.
/*	USA.
/*
/*	Additional work on LMTP by:
/*	Amos Gouaux
/*	University of Texas at Dallas
/*	P.O. Box 830688, MC34
/*	Richardson, TX 75083, USA
/*--*/

/* System library. */

#include <sys_defs.h>
#include <stdlib.h>			/* 44BSD stdarg.h uses abort() */
#include <stdarg.h>
#include <string.h>

/* Utility library. */

#include <msg.h>
#include <vstring.h>
#include <stringops.h>

/* Global library. */

#include <smtp_stream.h>
#include <deliver_request.h>
#include <deliver_completed.h>
#include <bounce.h>
#include <defer.h>
#include <mail_error.h>
#include <dsn_buf.h>
#include <dsn.h>

/* Application-specific. */

#include "lmtp.h"

#define LMTP_THROTTLE	1
#define LMTP_NOTHROTTLE	0

/* lmtp_check_code - check response code */

static void lmtp_check_code(LMTP_STATE *state, int code)
{

    /*
     * The intention of this code is to alert the postmaster when the local
     * Postfix LMTP client screws up, protocol wise. RFC 821 says that x0z
     * replies "refer to syntax errors, syntactically correct commands that
     * don't fit any functional category, and unimplemented or superfluous
     * commands". Unfortunately, this also triggers postmaster notices when
     * remote servers screw up, protocol wise. This is becoming a common
     * problem now that response codes are configured manually as part of
     * anti-UCE systems, by people who aren't aware of RFC details.
     */
    if (code < 400 || code > 599
	|| code == 555			/* RFC 1869, section 6.1. */
	|| (code >= 500 && code < 510))
	state->error_mask |= MAIL_ERROR_PROTOCOL;
}

/* lmtp_bulk_fail - skip, defer or bounce recipients, maybe throttle queue */

static int lmtp_bulk_fail(LMTP_STATE *state, DSN *dsn, int throttle_queue)
{
    DELIVER_REQUEST *request = state->request;
    LMTP_SESSION *session = state->session;
    RECIPIENT *rcpt;
    int     status;
    int     soft_error = (dsn->dtext[0] == '4');
    int     nrcpt;

    /*
     * If we are still in the connection set-up phase, update the set-up
     * completion time here, otherwise the time spent in set-up latency will
     * be attributed as message transfer latency.
     * 
     * All remaining recipients failed at this point, so we update the delivery
     * completion time stamp so that multiple recipient status records show
     * the same delay values.
     */
    if (request->msg_stats.conn_setup_done.tv_sec == 0) {
	GETTIMEOFDAY(&request->msg_stats.conn_setup_done);
	request->msg_stats.deliver_done =
	    request->msg_stats.conn_setup_done;
    } else
	GETTIMEOFDAY(&request->msg_stats.deliver_done);

    /*
     * If this is a soft error, postpone further deliveries to this domain.
     * Otherwise, generate a bounce record for each recipient.
     */
    for (nrcpt = 0; nrcpt < request->rcpt_list.len; nrcpt++) {
	rcpt = request->rcpt_list.info + nrcpt;
	if (rcpt->offset == 0)
	    continue;
	status = (soft_error ? defer_append : bounce_append)
	    (DEL_REQ_TRACE_FLAGS(request->flags), request->queue_id,
	     &request->msg_stats, rcpt,
	     session ? session->namaddr : "none", dsn);
	if (status == 0) {
	    deliver_completed(state->src, rcpt->offset);
	    rcpt->offset = 0;
	}
	state->status |= status;
    }
    if (throttle_queue && soft_error && request->hop_status == 0)
	request->hop_status = DSN_COPY(dsn);

    return (-1);
}

/* lmtp_sess_fail - skip site, defer or bounce all recipients */

int     lmtp_sess_fail(LMTP_STATE *state, DSN_BUF *why)
{
    DSN     dsn;

    /*
     * We need to incur the expense of copying lots of strings into VSTRING
     * buffers when the error information is collected by a routine that
     * terminates BEFORE the error is reported. If no copies were made, the
     * information would not be frozen in time.
     */
    return (lmtp_bulk_fail(state, DSN_FROM_DSN_BUF(&dsn, why), LMTP_THROTTLE));
}

/* vlmtp_fill_dsn - fill in temporary DSN structure */

static void vlmtp_fill_dsn(LMTP_STATE *state, DSN *dsn, const char *mta_name,
			           const char *status, const char *reply,
			           const char *format, va_list ap)
{

    /*
     * We can avoid the cost of copying lots of strings into VSTRING buffers
     * when the error information is collected by the routine that terminates
     * AFTER the error is reported. In this case, the information is already
     * frozen in time, so we don't need to make copies.
     */
    if (state->dsn_reason == 0)
	state->dsn_reason = vstring_alloc(100);
    else
	VSTRING_RESET(state->dsn_reason);
    if (mta_name && reply[0] != '4' && reply[0] != '5') {
	vstring_strcpy(state->dsn_reason, "Protocol error: ");
	mta_name = DSN_BY_LOCAL_MTA;
	status = "5.5.0";
	reply = "501 Protocol error in server reply";
    }
    vstring_vsprintf_append(state->dsn_reason, format, ap);
    LMTP_DSN_ASSIGN(dsn, mta_name, status, reply, STR(state->dsn_reason));
}

/* lmtp_fill_dsn - fill in temporary DSN structure */

static void lmtp_fill_dsn(LMTP_STATE *state, DSN *dsn, const char *mta_name,
			          const char *status, const char *reply,
			          const char *format,...)
{
    va_list ap;

    va_start(ap, format);
    vlmtp_fill_dsn(state, dsn, mta_name, status, reply, format, ap);
    va_end(ap);
}

/* lmtp_site_fail - defer site or bounce recipients */

int     lmtp_site_fail(LMTP_STATE *state, const char *mta_name, LMTP_RESP *resp,
		               const char *format,...)
{
    DSN     dsn;
    va_list ap;

    /*
     * Initialize.
     */
    va_start(ap, format);
    vlmtp_fill_dsn(state, &dsn, mta_name, resp->dsn, resp->str, format, ap);
    va_end(ap);

    if (state->session && mta_name)
	lmtp_check_code(state, resp->code);

    /*
     * Skip, defer or bounce recipients, and throttle this queue.
     */
    return (lmtp_bulk_fail(state, &dsn, LMTP_THROTTLE));
}

/* lmtp_mesg_fail - defer message or bounce all recipients */

int     lmtp_mesg_fail(LMTP_STATE *state, const char *mta_name, LMTP_RESP *resp,
		               const char *format,...)
{
    va_list ap;
    DSN     dsn;

    /*
     * Initialize.
     */
    va_start(ap, format);
    vlmtp_fill_dsn(state, &dsn, mta_name, resp->dsn, resp->str, format, ap);
    va_end(ap);

    if (state->session && mta_name)
	lmtp_check_code(state, resp->code);

    /*
     * Skip, defer or bounce recipients, but don't throttle this queue.
     */
    return (lmtp_bulk_fail(state, &dsn, LMTP_NOTHROTTLE));
}

/* lmtp_rcpt_fail - defer or bounce recipient */

void    lmtp_rcpt_fail(LMTP_STATE *state, const char *mta_name, LMTP_RESP *resp,
		               RECIPIENT *rcpt, const char *format,...)
{
    DELIVER_REQUEST *request = state->request;
    LMTP_SESSION *session = state->session;
    int     soft_error;
    int     status;
    DSN     dsn;
    va_list ap;

    /*
     * Initialize.
     */
    va_start(ap, format);
    vlmtp_fill_dsn(state, &dsn, mta_name, resp->dsn, resp->str, format, ap);
    va_end(ap);
    soft_error = dsn.dtext[0] == '4';

    if (state->session && mta_name)
	lmtp_check_code(state, resp->code);

    /*
     * If this is a soft error, postpone delivery to this recipient.
     * Otherwise, generate a bounce record for this recipient.
     */
    status = (soft_error ? defer_append : bounce_append)
	(DEL_REQ_TRACE_FLAGS(request->flags), request->queue_id,
	 &request->msg_stats, rcpt,
	 session ? session->namaddr : "none", &dsn);
    if (status == 0) {
	deliver_completed(state->src, rcpt->offset);
	rcpt->offset = 0;
    }
    state->status |= status;
}

/* lmtp_stream_except - defer domain after I/O problem */

int     lmtp_stream_except(LMTP_STATE *state, int code, const char *description)
{
    LMTP_SESSION *session = state->session;
    DSN     dsn;

    /*
     * Sanity check.
     */
    if (session == 0)
	msg_panic("lmtp_stream_except: no session");

    /*
     * Initialize.
     */
    switch (code) {
    default:
	msg_panic("lmtp_stream_except: unknown exception %d", code);
    case SMTP_ERR_EOF:
	lmtp_fill_dsn(state, &dsn, DSN_BY_LOCAL_MTA,
		      "4.4.2", "421 lost connection",
		      "lost connection with %s while %s",
		      session->namaddr, description);
	break;
    case SMTP_ERR_TIME:
	lmtp_fill_dsn(state, &dsn, DSN_BY_LOCAL_MTA,
		      "4.4.2", "426 conversation timed out",
		      "conversation with %s timed out while %s",
		      session->namaddr, description);
	break;
    case SMTP_ERR_PROTO:
	lmtp_fill_dsn(state, &dsn, DSN_BY_LOCAL_MTA,
		      "4.5.0", "403 remote protocol error",
		      "remote protocol error in reply from %s while %s",
		      session->namaddr, description);
	break;
    }
    return (lmtp_bulk_fail(state, &dsn, LMTP_THROTTLE));
}
