/*++
/* NAME
/*	cyrus_server 3
/* SUMMARY
/*	Cyrus SASL server-side plug-in
/* SYNOPSIS
/*	#include <cyrus_server.h>
/*
/*	XSASL_SERVER_IMPL *cyrus_server_init(appl_name)
/*	const char *appl_name;
/* DESCRIPTION
/*	This module implements the SASL server-side authentication plug-in.
/*
/*	cyrus_server_init() initializes the Cyrus SASL library and
/*	returns an implementation handle that can be used to generate
/*	SASL server instances.
/* DIAGNOSTICS
/*	All errors are fatal.
/* LICENSE
/* .ad
/* .fi
/*	The Secure Mailer license must be distributed with this software.
/* AUTHOR(S)
/*	Initial implementation by:
/*	Till Franke
/*	SuSE Rhein/Main AG
/*	65760 Eschborn, Germany
/*
/*	Adopted by:
/*	Wietse Venema
/*	IBM T.J. Watson Research
/*	P.O. Box 704
/*	Yorktown Heights, NY 10598, USA
/*--*/

/* System library. */

#include <sys_defs.h>
#include <stdlib.h>
#include <string.h>

/* Utility library. */

#include <msg.h>
#include <mymalloc.h>
#include <name_mask.h>
#include <stringops.h>

/* Global library. */

#include <mail_params.h>

/* Application-specific. */

#include <xsasl_server.h>
#include <cyrus_common.h>
#include <cyrus_server.h>

#ifdef USE_SASL_AUTH

#include <sasl.h>
#include <saslutil.h>

/*
 * Silly little macros.
 */
#define STR(s)	vstring_str(s)

 /*
  * Macros to handle API differences between SASLv1 and SASLv2. Specifics:
  * 
  * The SASL_LOG_* constants were renamed in SASLv2.
  * 
  * SASLv2's sasl_server_new takes two new parameters to specify local and
  * remote IP addresses for auth mechs that use them.
  * 
  * SASLv2's sasl_server_start and sasl_server_step no longer have the errstr
  * parameter.
  * 
  * SASLv2's sasl_decode64 function takes an extra parameter for the length of
  * the output buffer.
  * 
  * The other major change is that SASLv2 now takes more responsibility for
  * deallocating memory that it allocates internally.  Thus, some of the
  * function parameters are now 'const', to make sure we don't try to free
  * them too.  This is dealt with in the code later on.
  */

#if SASL_VERSION_MAJOR < 2
/* SASL version 1.x */
#define SASL_SERVER_NEW(srv, fqdn, rlm, lport, rport, cb, secflags, pconn) \
	sasl_server_new(srv, fqdn, rlm, cb, secflags, pconn)
#define SASL_SERVER_START(conn, mech, clin, clinlen, srvout, srvoutlen, err) \
	sasl_server_start(conn, mech, clin, clinlen, srvout, srvoutlen, err)
#define SASL_SERVER_STEP(conn, clin, clinlen, srvout, srvoutlen, err) \
	sasl_server_step(conn, clin, clinlen, srvout, srvoutlen, err)
#define SASL_DECODE64(in, inlen, out, outmaxlen, outlen) \
	sasl_decode64(in, inlen, out, outlen)
typedef unsigned MECHANISM_COUNT_TYPE;
typedef char *SERVEROUT_TYPE;
typedef void *VOID_SERVEROUT_TYPE;

#endif

#if SASL_VERSION_MAJOR >= 2
/* SASL version > 2.x */
#define SASL_SERVER_NEW(srv, fqdn, rlm, lport, rport, cb, secflags, pconn) \
	sasl_server_new(srv, fqdn, rlm, lport, rport, cb, secflags, pconn)
#define SASL_SERVER_START(conn, mech, clin, clinlen, srvout, srvoutlen, err) \
	sasl_server_start(conn, mech, clin, clinlen, srvout, srvoutlen)
#define SASL_SERVER_STEP(conn, clin, clinlen, srvout, srvoutlen, err) \
	sasl_server_step(conn, clin, clinlen, srvout, srvoutlen)
#define SASL_DECODE64(in, inlen, out, outmaxlen, outlen) \
	sasl_decode64(in, inlen, out, outmaxlen, outlen)
typedef int MECHANISM_COUNT_TYPE;
typedef const char *SERVEROUT_TYPE;
typedef const void *VOID_SERVEROUT_TYPE;

#endif

 /*
  * The CYRUS_SERVER object is derived from the generic XSASL_SERVER object.
  */
typedef struct {
    XSASL_SERVER xsasl;			/* generic members, must be first */
    sasl_conn_t *sasl_conn;		/* SASL context */
    VSTRING *decoded;			/* decoded challenge or response */
    char   *username;			/* authenticated user */
    char   *mechanism_list;		/* applicable mechanisms */
} CYRUS_SERVER;

 /*
  * Forward declarations.
  */
static XSASL_SERVER *cyrus_server_create(const char *, const char *);
static void cyrus_server_free(XSASL_SERVER *);
static int cyrus_server_first(XSASL_SERVER *, const char *, const char *, VSTRING *);
static int cyrus_server_next(XSASL_SERVER *, const char *, VSTRING *);
static int cyrus_server_set_security(XSASL_SERVER *, const char *, const char *);
static const char *cyrus_server_get_mechanism_list(XSASL_SERVER *);
static const char *cyrus_server_get_username(XSASL_SERVER *);

 /*
  * SASL callback interface structure. These call-backs have no per-session
  * context.
  */
#define NO_CALLBACK_CONTEXT	0

static sasl_callback_t callbacks[] = {
    {SASL_CB_LOG, &cyrus_log, NO_CALLBACK_CONTEXT},
    {SASL_CB_LIST_END, 0, 0}
};

/* cyrus_server_init - create implementation handle */

XSASL_SERVER_IMPL *cyrus_server_init(const char *appl_name)
{
    const char *myname = "cyrus_server_init";
    XSASL_SERVER_IMPL *xp;
    int     cyrus_status;

#if SASL_VERSION_MAJOR >= 2 && (SASL_VERSION_MINOR >= 2 \
    || (SASL_VERSION_MINOR == 1 && SASL_VERSION_STEP >= 19))
    int     sasl_major;
    int     sasl_minor;
    int     sasl_step;

    /*
     * DLL hell guard.
     */
    sasl_version_info((const char **) 0, (const char **) 0,
		      &sasl_major, &sasl_minor,
		      &sasl_step, (int *) 0);
    if (sasl_major != SASL_VERSION_MAJOR
#if 0
	|| sasl_minor != SASL_VERSION_MINOR
	|| sasl_step != SASL_VERSION_STEP
#endif
	)
	msg_fatal("incorrect SASL library version. "
	      "Postfix was built with include files from version %d.%d.%d, "
		  "but the run-time library version is %d.%d.%d",
		  SASL_VERSION_MAJOR, SASL_VERSION_MINOR, SASL_VERSION_STEP,
		  sasl_major, sasl_minor, sasl_step);
#endif

    /*
     * Initialize the library: load SASL plug-in routines, etc.
     */
    if (msg_verbose)
	msg_info("%s: SASL config file is %s.conf",
		 myname, appl_name);
    if ((cyrus_status = sasl_server_init(callbacks, appl_name)) != SASL_OK)
	msg_fatal("SASL per-process initialization failed: %s",
		  cyrus_strerror(cyrus_status));

    xp = (XSASL_SERVER_IMPL *) mymalloc(sizeof(*xp));
    xp->create = cyrus_server_create;
    return (xp);
}

/* cyrus_server_create - create server instance */

static XSASL_SERVER *cyrus_server_create(const char *service, const char *realm)
{
    const char *myname = "cyrus_server_create";
    char   *server_address;
    char   *client_address;
    sasl_conn_t *sasl_conn;
    CYRUS_SERVER *cp;
    int     cyrus_status;

    if (msg_verbose)
	msg_info("%s: SASL service=%s, realm=%s",
		 myname, service, realm ? realm : "(null)");

    /*
     * Set up a new server context.
     */
#define NO_SECURITY_LAYERS	(0)
#define NO_SESSION_CALLBACKS	((sasl_callback_t *) 0)
#define NO_AUTH_REALM		((char *) 0)

#if SASL_VERSION_MAJOR >= 2 && defined(USE_SASL_IP_AUTH)

    /*
     * Get IP addresses of local and remote endpoints for SASL.
     */
#error "USE_SASL_IP_AUTH is not implemented"

#else

    /*
     * Don't give any IP address information to SASL.  SASLv1 doesn't use it,
     * and in SASLv2 this will disable any mechaniams that do.
     */
    server_address = 0;
    client_address = 0;
#endif

    if ((cyrus_status =
	 SASL_SERVER_NEW(service, var_myhostname,
			 realm ? realm : NO_AUTH_REALM,
			 server_address, client_address,
			 NO_SESSION_CALLBACKS, NO_SECURITY_LAYERS,
			 &sasl_conn)) != SASL_OK)
	msg_fatal("SASL per-connection server initialization: %s",
		  cyrus_strerror(cyrus_status));

    cp = (CYRUS_SERVER *) mymalloc(sizeof(*cp));
    cp->xsasl.free = cyrus_server_free;
    cp->xsasl.first = cyrus_server_first;
    cp->xsasl.next = cyrus_server_next;
    cp->xsasl.set_security = cyrus_server_set_security;
    cp->xsasl.get_mechanism_list = cyrus_server_get_mechanism_list;
    cp->xsasl.get_username = cyrus_server_get_username;
    cp->sasl_conn = sasl_conn;
    cp->decoded = vstring_alloc(20);
    cp->username = 0;
    cp->mechanism_list = 0;

    return (&cp->xsasl);
}

/* cyrus_server_set_security - set security properties */

static int cyrus_server_set_security(XSASL_SERVER *xp,
				             const char *sasl_opts_name,
				             const char *sasl_opts_val)
{
    CYRUS_SERVER *cp = (CYRUS_SERVER *) xp;
    sasl_security_properties_t sec_props;
    int     cyrus_status;

    /*
     * Security options. Some information can be found in the sasl.h include
     * file.
     */
    memset(&sec_props, 0, sizeof(sec_props));
    sec_props.min_ssf = 0;
    sec_props.max_ssf = 0;			/* don't allow real SASL
						 * security layer */
    sec_props.security_flags =
	cyrus_security_parse_opts(sasl_opts_name, sasl_opts_val);
    sec_props.maxbufsize = 0;
    sec_props.property_names = 0;
    sec_props.property_values = 0;

    if ((cyrus_status = sasl_setprop(cp->sasl_conn, SASL_SEC_PROPS,
				     &sec_props)) != SASL_OK)
	msg_fatal("SASL per-connection security setup; %s",
		  cyrus_strerror(cyrus_status));
    return (0);
}

/* cyrus_server_get_mechanism_list - get available authentication mechanisms */

static const char *cyrus_server_get_mechanism_list(XSASL_SERVER *xp)
{
    const char *myname = "cyrus_server_get_mechanism_list";
    CYRUS_SERVER *cp = (CYRUS_SERVER *) xp;
    const char *mechanism_list;
    MECHANISM_COUNT_TYPE mechanism_count;
    int     cyrus_status;

    /*
     * Get the list of authentication mechanisms.
     */
#define UNSUPPORTED_USER	((char *) 0)
#define IGNORE_MECHANISM_LEN	((unsigned *) 0)

    if ((cyrus_status = sasl_listmech(cp->sasl_conn, UNSUPPORTED_USER,
				      "", " ", "",
				      &mechanism_list,
				      IGNORE_MECHANISM_LEN,
				      &mechanism_count)) != SASL_OK)
	msg_fatal("%s: %s", myname, cyrus_strerror(cyrus_status));
    if (mechanism_count <= 0)
	msg_fatal("%s: no applicable SASL mechanisms", myname);
    cp->mechanism_list = mystrdup(mechanism_list);
#if SASL_VERSION_MAJOR < 2
    /* SASL version 1 doesn't free memory that it allocates. */
    free(mechanism_list);
#endif
    return (cp->mechanism_list);
}

/* cyrus_server_free - destroy server instance */

static void cyrus_server_free(XSASL_SERVER *xp)
{
    CYRUS_SERVER *cp = (CYRUS_SERVER *) xp;

    sasl_dispose(&cp->sasl_conn);
    vstring_free(cp->decoded);
    if (cp->username)
	myfree(cp->username);
    if (cp->mechanism_list)
	myfree(cp->mechanism_list);
    myfree((char *) cp);
}

/* cyrus_auth_response - encode server first/next response */

static int cyrus_auth_response(int cyrus_status, SERVEROUT_TYPE serverout,
			               unsigned serveroutlen, VSTRING *reply)
{
    const char *myname = "cyrus_auth_response";
    unsigned enc_length;
    unsigned enc_length_out;

    /*
     * Encode the server first/next non-error response; otherwise return the
     * unencoded error text that corresponds to the SASL error status.
     * 
     * Regarding the hairy expression below: output from sasl_encode64() comes
     * in multiples of four bytes for each triple of input bytes, plus four
     * bytes for any incomplete last triple, plus one byte for the null
     * terminator.
     */
    if (cyrus_status == SASL_OK) {
	vstring_strcpy(reply, "");
	return (XSASL_SERVER_AUTH_DONE);
    } else if (cyrus_status == SASL_CONTINUE) {
	if (msg_verbose)
	    msg_info("%s: uncoded server challenge: %.*s",
		     myname, (int) serveroutlen, serverout);
	enc_length = ((serveroutlen + 2) / 3) * 4 + 1;
	VSTRING_RESET(reply);			/* Fix 200512 */
	VSTRING_SPACE(reply, enc_length);
	if ((cyrus_status = sasl_encode64(serverout, serveroutlen,
					  STR(reply), vstring_avail(reply),
					  &enc_length_out)) != SASL_OK)
	    msg_panic("%s: sasl_encode64 botch: %s",
		      myname, cyrus_strerror(cyrus_status));
	return (XSASL_SERVER_AUTH_MORE);
    } else {
	vstring_strcpy(reply, cyrus_strerror(cyrus_status));
	return (XSASL_SERVER_AUTH_FAIL);
    }
}

/* cyrus_server_first - per-session authentication */

int     cyrus_server_first(XSASL_SERVER *xp, const char *sasl_method,
			           const char *init_response, VSTRING *reply)
{
    char   *myname = "cyrus_server_first";
    CYRUS_SERVER *cp = (CYRUS_SERVER *) xp;
    char   *dec_buffer;
    unsigned dec_length;
    unsigned reply_len;
    unsigned serveroutlen;
    int     cyrus_status;
    SERVEROUT_TYPE serverout = 0;
    int     xsasl_status;

#if SASL_VERSION_MAJOR < 2
    const char *errstr = 0;

#endif

#define IFELSE(e1,e2,e3) ((e1) ? (e2) : (e3))

    if (msg_verbose)
	msg_info("%s: sasl_method %s%s%s", myname, sasl_method,
		 IFELSE(init_response, ", init_response ", ""),
		 IFELSE(init_response, init_response, ""));

    /*
     * SASL authentication protocol start-up. Process any initial client
     * response that was sent along in the AUTH command.
     */
    if (init_response) {
	reply_len = strlen(init_response);
	VSTRING_RESET(cp->decoded);		/* Fix 200512 */
	VSTRING_SPACE(cp->decoded, reply_len);
	if ((cyrus_status = SASL_DECODE64(init_response, reply_len,
					  dec_buffer = STR(cp->decoded),
					  vstring_avail(cp->decoded),
					  &dec_length)) != SASL_OK) {
	    vstring_strcpy(reply, cyrus_strerror(cyrus_status));
	    return (XSASL_SERVER_AUTH_FORM);
	}
	if (msg_verbose)
	    msg_info("%s: decoded initial response %s", myname, dec_buffer);
    } else {
	dec_buffer = 0;
	dec_length = 0;
    }
    cyrus_status = SASL_SERVER_START(cp->sasl_conn, sasl_method, dec_buffer,
				     dec_length, &serverout,
				     &serveroutlen, &errstr);
    xsasl_status = cyrus_auth_response(cyrus_status, serverout,
				       serveroutlen, reply);
#if SASL_VERSION_MAJOR < 2
    /* SASL version 1 doesn't free memory that it allocates. */
    free(serverout);
#endif
    return (xsasl_status);
}

/* cyrus_server_next - continue authentication */

static int cyrus_server_next(XSASL_SERVER *xp, const char *request,
			             VSTRING *reply)
{
    char   *myname = "cyrus_server_next";
    CYRUS_SERVER *cp = (CYRUS_SERVER *) xp;
    unsigned dec_length;
    unsigned request_len;
    unsigned serveroutlen;
    int     cyrus_status;
    SERVEROUT_TYPE serverout = 0;
    int     xsasl_status;

#if SASL_VERSION_MAJOR < 2
    const char *errstr = 0;

#endif

    request_len = strlen(request);
    VSTRING_RESET(cp->decoded);			/* Fix 200512 */
    VSTRING_SPACE(cp->decoded, request_len);
    if ((cyrus_status = SASL_DECODE64(request, request_len,
				      STR(cp->decoded),
				      vstring_avail(cp->decoded),
				      &dec_length)) != SASL_OK) {
	vstring_strcpy(reply, cyrus_strerror(cyrus_status));
	return (XSASL_SERVER_AUTH_FORM);
    }
    if (msg_verbose)
	msg_info("%s: decoded response: %.*s",
		 myname, (int) dec_length, STR(cp->decoded));
    cyrus_status = SASL_SERVER_STEP(cp->sasl_conn, STR(cp->decoded),
				    dec_length, &serverout,
				    &serveroutlen, &errstr);
    xsasl_status = cyrus_auth_response(cyrus_status, serverout,
				       serveroutlen, reply);
#if SASL_VERSION_MAJOR < 2
    /* SASL version 1 doesn't free memory that it allocates. */
    free(serverout);
#endif
    return (xsasl_status);
}

/* cyrus_server_get_username - get authenticated username */

static const char *cyrus_server_get_username(XSASL_SERVER *xp)
{
    const char *myname = "cyrus_server_get_username";
    CYRUS_SERVER *cp = (CYRUS_SERVER *) xp;
    VOID_SERVEROUT_TYPE serverout = 0;
    int     cyrus_status;

    /*
     * XXX Do not free(serverout).
     */
    cyrus_status = sasl_getprop(cp->sasl_conn, SASL_USERNAME, &serverout);
    if (cyrus_status != SASL_OK || serverout == 0)
	msg_panic("%s: sasl_getprop SASL_USERNAME botch: %s",
		  myname, cyrus_strerror(cyrus_status));
    if (cp->username)
	myfree(cp->username);
    cp->username = mystrdup(serverout);
    printable(cp->username, '?');
    return (cp->username);
}

#endif
