/*++
/* NAME
/*	xsasl-server 3
/* SUMMARY
/*	Postfix SASL server plug-in interface
/* SYNOPSIS
/*	#include <xsasl_server.h>
/*
/*	XSASL_SERVER_IMPL *xsasl_server_init(server_type, appl_name)
/*	const char *server_type;
/*	const char *appl_name;
/*
/*	XSASL_SERVER *xsasl_server_create(implementation, service, user_realm)
/*	XSASL_SERVER_IMPL *implementation;
/*	const char *service;
/*	const char *user_realm;
/*
/*	void xsasl_server_free(server)
/*	XSASL_SERVER *server;
/*
/*	int	xsasl_server_first(server, auth_method, init_resp, server_reply)
/*	XSASL_SERVER *server;
/*	const char *auth_method;
/*	const char *init_resp;
/*	VSTRING *server_reply;
/*
/*	int	xsasl_server_next(server, client_request, server_reply)
/*	XSASL_SERVER *server;
/*	const char *client_request;
/*	VSTRING *server_reply;
/* AUXILIARY FUNCTIONS
/*	int	xsasl_server_set_security(server, paramname, properties)
/*	XSASL_SERVER *server;
/*	const char *paramname;
/*	const char *properties;
/*
/*	const char *xsasl_server_get_mechanism_list(server)
/*	XSASL_SERVER *server;
/*
/*	const char *xsasl_server_get_username(server)
/*	XSASL_SERVER *server;
/* DESCRIPTION
/*	The XSASL_SERVER implements a generic interface to one or
/*	more SASL authentication implementations.
/*
/*	xsasl_server_init() is called once during process
/*	initialization.  It selects a SASL service implementation
/*	by name, specifies the application name, and returns an
/*	implementation handle that can be used to generate SASL
/*	server instances.
/*
/*	xsasl_server_create() is called at the start of an SMTP
/*	session. It generates a SASL server instance for the specified
/*	service and authentication realm. Specify a null pointer
/*	when no realm should be used.
/*
/*	xsasl_server_free() is called at the end of an SMTP session.
/*	It destroys a SASL server instance.
/*
/*	xsasl_server_first() produces the server reponse for the
/*	client AUTH command. The client input are an authentication
/*	method, and an optional initial response or null pointer.
/*	The initial response and server non-error replies are BASE64
/*	encoded.  Server error replies are 7-bit ASCII text without
/*	control characters, without BASE64 encoding, and without
/*	SMTP reply code or enhanced status code.
/*
/*	The result is one of the following:
/* .IP XSASL_SERVER_MORE
/*	More client input is needed. The server reply specifies what.
/* .IP XSASL_SERVER_DONE
/*	Authentication completed successfully.
/* .IP XSASL_SERVER_FORM
/*	The client input is incorrectly formatted. The server error
/*	reply explains why.
/* .IP XSASL_SERVER_FAIL
/*	Authentication failed. The server error reply explains why.
/* .PP
/*	xsasl_server_next() supports the subsequent stages of the
/*	client-server AUTH protocol. Both the client input and 
/*	server non-error responses are BASE64 encoded.  See
/*	xsasl_server_first() for other details.
/*
/*	xsasl_server_set_security() specifies the security requirements
/*	for authentication methods in the form of a parameter name and
/*	value. The possible values are implementation dependent.
/*
/*	xsasl_server_get_mechanism_list() returns the authentication
/*	mechanisms that match the security properties, as a white-space
/*	separated list. This is meant to be used in the SMTP EHLO reply.
/*
/*	xsasl_server_get_username() returns the stored username after
/*	successful authentication.
/* SECURITY
/*	The caller does not sanitize client input. It is the
/*	responsibility of the SASL server implementation to produce
/*	7-bit ASCII without control characters as server non-error
/*	and error replies, and as the result from xsasl_server_method()
/*	and xsasl_server_username().
/* DIAGNOSTICS
/*	xsasl_server_init() and xsasl_server_create()
/*	return a null pointer upon failure.
/*
/*	Fatal errors: out of memory.
/* SEE ALSO
/*	cyrus_security(3) Cyrus SASL security features
/* LICENSE
/* .ad
/* .fi
/*	The Secure Mailer license must be distributed with this
/*	software.
/* AUTHOR(S)
/*	Wietse Venema
/*	IBM T.J. Watson Research
/*	P.O. Box 704
/*	Yorktown Heights, NY 10598, USA
/*--*/

/* System library. */

#include <sys_defs.h>
#include <string.h>

/* Utility library. */

#include <msg.h>
#include <mymalloc.h>

/* SASL implementations. */

#include <xsasl_server.h>
#include <cyrus_server.h>

 /*
  * Lookup table for available SASL server implementations.
  */
typedef struct {
    char   *server_type;
    struct XSASL_SERVER_IMPL *(*server_init) (const char *);
} XSASL_SERVER_IMPL_INFO;

static XSASL_SERVER_IMPL_INFO server_impl_info[] = {
#ifdef XSASL_SERVER_TYPE_CYRUS
    XSASL_SERVER_TYPE_CYRUS, cyrus_server_init,
#endif
    0,
};

/* xsasl_server_init - look up server implementation by name */

XSASL_SERVER_IMPL *xsasl_server_init(const char *server_type,
				             const char *appl_name)
{
    XSASL_SERVER_IMPL_INFO *xp;

    for (xp = server_impl_info; xp->server_type; xp++)
	if (strcmp(server_type, xp->server_type) == 0)
	    return (xp->server_init(appl_name));
    msg_fatal("unsupported SASL authentication implementation: %s",
	      server_type);
}
