/*++
/* NAME
/*	dsn
/* SUMMARY
/*	RFC-compliant delivery status information
/* SYNOPSIS
/*	#include <dsn.h>
/*
/*	typedef struct {
/* .in +4
/*		char *status;	/* RFC 3463 status */
/*		char *action;	/* null or RFC 3464 action */
/*		char *reason;	/* human-readable text */
/*		char *dtype;	/* null or diagnostic type */
/*		char *dtext;	/* null or diagnostic code */
/*		char *mtype;	/* null or MTA type */
/*		char *mname;	/* null or remote MTA */
/* .in -4
/*	} DSN;
/*
/*	DSN *create(status, action, reason, dtype, dtext, mtype, mname)
/*	const char *status;
/*	const char *action;
/*	const char *reason;
/*	const char *dtype;
/*	const char *dtext;
/*	const char *mtype;
/*	const char *mname;
/*
/*	DSN *copy(dsn)
/*	DSN *dsn;
/*
/*	DSN *DSN_ASSIGN(dsn, status, action, reason, dtype, dtext, mtype, mname)
/*	DSN *dsn;
/*	const char *status;
/*	const char *action;
/*	const char *reason;
/*	const char *dtype;
/*	const char *dtext;
/*	const char *mtype;
/*	const char *mname;
/*
/*	DSN *DSN_SIMPLE(dsn, status, action, reason)
/*	DSN *dsn;
/*	const char *status;
/*	const char *action;
/*	const char *reason;
/*
/*	DSN *DSN_SMTP(dsn, status, action, smtp_dtext, reason)
/*	DSN *dsn;
/*	const char *status;
/*	const char *action;
/*	const char *smtp_dtext;
/*	const char *reason;
/*
/*	void	dsn_free(dsn)
/*	DSN *dsn;
/* DESCRIPTION
/*	This module maintains delivery error information. For a
/*	description of structure field members see "Arguments"
/*	below.  Function-like names spelled in upper case are macros.
/*	These may evaluate some arguments more than once.
/*
/*	dsn_create() creates a DSN structure and copies its arguments.
/*	The DSN structure should be destroyed with dsn_free().
/*
/*	DSN_COPY() creates a deep copy of its argument.
/*
/*	dsn_free() destroys a DSN structure and makes its storage
/*	available for reuse.
/*
/*	DSN_ASSIGN() updates a DSN structure and DOES NOT copy
/*	arguments or free memory.  The result DSN structure must
/*	NOT be passed to dsn_free(). DSN_ASSIGN() is typically used
/*	for stack-based short-lived storage.
/*
/*	DSN_SIMPLE() takes the minimally required subset of all the
/*	parameters and resets the rest to zero.
/*
/*	DSN_SMTP() handles the common case of an SMTP-type
/*	diagnostic code that was generated by the local MTA.
/*
/*	Arguments:
/* .IP reason
/*	Human-readable text, used for logging purposes, and for
/*	updating the message-specific \fBbounce\fR or \fIdefer\fR
/*	logfile.
/* .IP status
/*	Enhanced status code as specified in RFC 3463.
/* .IP action
/*	Action as defined in RFC 3464. If null, a default action
/*	is chosen.
/* .IP dtype
/*	DSN_NO_DTYPE, empty string, or diagnostic code type as
/*	specified in RFC 3464.
/* .IP dtext
/*	DSN_NO_DTEXT, empty string, or diagnostic code as specified
/*	in RFC 3464.
/* .IP mtype
/*	DSN_NO_MTYPE, empty string, DSN_MTYPE_DNS or DSN_MTYPE_UNIX.
/* .IP mname
/*	DSN_NO_MNAME, empty string, or remote MTA as specified in
/*	RFC 3464.
/* DIAGNOSTICS
/*	Panic: null status or reason.
/*	Fatal: out of memory.
/* LICENSE
/* .ad
/* .fi
/*	The Secure Mailer license must be distributed with this software.
/* AUTHOR(S)
/*	Wietse Venema
/*	IBM T.J. Watson Research
/*	P.O. Box 704
/*	Yorktown Heights, NY 10598, USA
/*--*/

/* System library. */

#include <sys_defs.h>

/* Utility library. */

#include <msg.h>
#include <mymalloc.h>

/* Global library. */

#include <dsn.h>

/* dsn_create - create DSN structure */

DSN    *dsn_create(const char *status, const char *action, const char *reason,
		           const char *dtype, const char *dtext,
		           const char *mtype, const char *mname)
{
    const char *myname = "dsn_create";
    DSN    *dsn;

    dsn = (DSN *) mymalloc(sizeof(*dsn));

    /*
     * Status and reason must not be null. Other members may be null pointers
     * or empty strings. Null/empty members are represented as null pointers.
     */
#define NULL_OR_EMPTY(s) ((s) == 0 || *(s) == 0)

    if (NULL_OR_EMPTY(status))
	msg_panic("%s: null dsn status", myname);
    else
	dsn->status = mystrdup(status);

    if (NULL_OR_EMPTY(action))
	dsn->action = 0;
    else
	dsn->action = mystrdup(action);

    if (NULL_OR_EMPTY(reason))
	msg_panic("%s: null dsn reason", myname);
    else
	dsn->reason = mystrdup(reason);

    if (NULL_OR_EMPTY(dtype) || NULL_OR_EMPTY(dtext)) {
	dsn->dtype = 0;
	dsn->dtext = 0;
    } else {
	dsn->dtype = mystrdup(dtype);
	dsn->dtext = mystrdup(dtext);
    }
    if (NULL_OR_EMPTY(mtype) || NULL_OR_EMPTY(mname)) {
	dsn->mtype = 0;
	dsn->mname = 0;
    } else {
	dsn->mtype = mystrdup(mtype);
	dsn->mname = mystrdup(mname);
    }
    return (dsn);
}

/* dsn_free - destroy DSN structure */

void    dsn_free(DSN *dsn)
{
    myfree((char *) dsn->status);
    if (dsn->action)
	myfree((char *) dsn->action);
    myfree((char *) dsn->reason);
    if (dsn->dtype)
	myfree((char *) dsn->dtype);
    if (dsn->dtext)
	myfree((char *) dsn->dtext);
    if (dsn->mtype)
	myfree((char *) dsn->mtype);
    if (dsn->mname)
	myfree((char *) dsn->mname);
    myfree((char *) dsn);
}
