/*
 * Copyright (c) 2002-2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: smtpsh.c,v 1.32 2005/10/27 21:19:26 ca Exp $")

#include "sm/assert.h"
#include "sm/error.h"
#include "sm/ctype.h"
#include "sm/memops.h"
#include "sm/io.h"
#include "sm/str.h"
#include "sm/string.h"
#include "sm/time.h"
#include "sm/limits.h"
#include "sm/types.h"
#include "sm/net.h"
#include "statethreads/st.h"
#define SM_USE_STATETHREADS 1
#include "sm/stsock.h"
#include "sm/tls.h"
#include "sm/tlsbio.h"
#include "sm/sasl.h"
#include "sm/rfc2821.h"

#include "s2q.h"
#include "smtps.h"
#include "s2m.h"
#include "smtpsrv.h"
#include "smtpsh.h"
#include "log.h"

/*
**  SSM_MAIL_NEW -- create new mail address
**
**	Parameters:
**		ss_ta -- SMTP server transaction context
**		alloc_pa -- also create sm_str for printable address?
**		pmail -- mail (sender) address (output)
**
**	Returns:
**		usual return code
*/

sm_ret_T
ssm_mail_new(ss_ta_P ss_ta, bool alloc_pa, ss_mail_P *pmail)
{
	ss_mail_P mail;

	SM_REQUIRE(pmail != NULL);
	mail = (ss_mail_P) sm_rpool_zalloc(ss_ta->ssta_rpool, sizeof(*mail));
	if (mail == NULL)
		return sm_error_temp(SM_EM_SMTPS, ENOMEM);
	A2821_INIT_RP(&(mail->ssm_a2821), ss_ta->ssta_rpool);
	if (alloc_pa)
	{
		mail->ssm_pa = sm_str_new(ss_ta->ssta_rpool, MAXADDRLEN,
					MAXADDRLEN);
		if (mail->ssm_pa == NULL)
		{
			ssm_mail_free(ss_ta, mail);
			return sm_error_temp(SM_EM_SMTPS, ENOMEM);
		}
	}
	*pmail = mail;
	return SM_SUCCESS;
}

/*
**  SSM_MAIL_FREE -- free mail address
**
**	Parameters:
**		ss_ta -- SMTP server transaction context
**		mail -- mail (sender) address
**
**	Returns:
**		usual return code
*/

sm_ret_T
ssm_mail_free(ss_ta_P ss_ta, ss_mail_P mail)
{
	if (mail == NULL)
		return SM_SUCCESS;
	a2821_free(&(mail->ssm_a2821));
	SM_STR_FREE(mail->ssm_pa);
	sm_rpool_free(ss_ta->ssta_rpool, mail);
	return SM_SUCCESS;
}

/*
**  SSR_RCPTS_NEW -- add a new recipient to the recipient list;
**	if the latter doesn't exist: create it
**
**	Parameters:
**		ss_ta -- SMTP server transaction context
**		rcpts -- list of recipients
**		prcpt -- new recipient (output)
**
**	Returns:
**		usual return code
*/

sm_ret_T
ssr_rcpts_new(ss_ta_P ss_ta, ss_rcpts_P rcpts, ss_rcpt_P *prcpt)
{
	SM_REQUIRE(rcpts != NULL);
	SM_REQUIRE(prcpt != NULL);
	*prcpt = (ss_rcpt_P) sm_rpool_zalloc(ss_ta->ssta_rpool,
						sizeof(**prcpt));
	if (*prcpt == NULL)
		goto error;
	A2821_INIT_RP(&((*prcpt)->ssr_a2821), ss_ta->ssta_rpool);
	SS_RCPTS_INSERT_TAIL(rcpts, *prcpt);
	(*prcpt)->ssr_idx = ss_ta->ssta_rcpts_tot++;
	return SM_SUCCESS;

  error:
	SM_RPOOL_FREE(ss_ta->ssta_rpool, *prcpt);
	return sm_error_temp(SM_EM_SMTPS, ENOMEM);
}

/*
**  SSR_RCPT_FREE -- free a single recipient address
**
**	Parameters:
**		ss_ta -- SMTP server transaction context
**		rcpt -- recipient to free
**
**	Returns:
**		usual return code
*/

sm_ret_T
ssr_rcpt_free(ss_ta_P ss_ta, ss_rcpt_P rcpt)
{
	if (rcpt == NULL)
		return SM_SUCCESS;
	a2821_free(&(rcpt->ssr_a2821));
	sm_rpool_free(ss_ta->ssta_rpool, rcpt);

	/* remove adr from list? it's done by SS_RCPTS_REMOVE_FREE() */
	/* XXX adjust some counter? */

	return SM_SUCCESS;
}

/*
**  SSR_RCPTS_FREE -- free an entire recipient list
**
**	Parameters:
**		ss_ta -- SMTP server transaction context
**		rcpts -- recipient list to free
**
**	Returns:
**		usual return code
*/

static sm_ret_T
ssr_rcpts_free(ss_ta_P ss_ta, ss_rcpts_P rcpts)
{
	ss_rcpt_P rcpt, nxt;

	if (SS_RCPTS_EMPTY(rcpts))
		return SM_SUCCESS;
	for (rcpt = SS_RCPTS_FIRST(rcpts); rcpt != SS_RCPTS_END(rcpts); rcpt = nxt)
	{
		nxt = SS_RCPTS_NEXT(rcpt);

		/* remove adr from list? use SS_RCPTS_REMOVE_FREE()? */
		ssr_rcpt_free(ss_ta, rcpt);
	}
	SS_RCPTS_INIT(rcpts);
	return SM_SUCCESS;
}

/*
**  SS_TA_CLR -- clear out a transaction context (for reuse)
**
**	Parameters:
**		ss_ta -- SMTP server transaction context
**
**	Returns:
**		usual return code
*/

sm_ret_T
ss_ta_clr(ss_ta_P ss_ta)
{
	SM_IS_SS_TA(ss_ta);

	ssm_mail_free(ss_ta, ss_ta->ssta_mail);
	ssr_rcpts_free(ss_ta, &(ss_ta->ssta_rcpts));
	SM_STR_FREE(ss_ta->ssta_mail_acc.ssa_reply_text);
	SM_STR_FREE(ss_ta->ssta_rcpt_acc.ssa_reply_text);
#if SM_USE_PMILTER
	SM_STR_FREE(ss_ta->ssta_msg_acc.ssa_reply_text);
#endif
	SM_STR_FREE(ss_ta->ssta_msgid);
	if (ss_ta->ssta_rpool != NULL)
	{
		sm_rpool_delete(ss_ta->ssta_rpool);
		ss_ta->ssta_rpool = NULL;
	}

	/* clear transaction, also resets counters */
	sm_memzero(ss_ta, sizeof(*ss_ta));
	SS_RCPTS_INIT(&(ss_ta->ssta_rcpts));
#if SS_TA_CHECK
	ss_ta->sm_magic = SM_SS_TA_MAGIC;
#endif
	return SM_SUCCESS;
}

/*
**  SS_TA_ABORT -- abort a transaction
**
**	Parameters:
**		ss_sess -- SMTP server session context
**		ss_ta -- SMTP server transaction context
**
**	Returns:
**		usual return code
*/

sm_ret_T
ss_ta_abort(ss_sess_P ss_sess, ss_ta_P ss_ta)
{
	sm_ret_T ret;

	SM_IS_SS_TA(ss_ta);

	/* check existing transaction */
	if (ss_ta->ssta_state != SSTA_ST_NONE)
	{
#if SM_USE_PMILTER
		if (SSTA_IS_FLAG(ss_ta, SSTA_FL_PM_CALLED) &&
		    SSSE_IS_FLAG(ss_sess, SSSE_FL_PM_USE))
		{
			ss_ctx_P ss_ctx;

			ss_ctx = ss_sess->ssse_sctx;
			ret = sm_s2m_abort(ss_sess,
					ss_ctx->ssc_s2m_ctx,
					ss_ctx->ssc_cnf.ss_cnf_id,
					ss_sess->ssse_id);
			if (sm_is_err(ret))
			{
				sm_log_write(ss_ctx->ssc_lctx,
					SS_LCAT_SERVER, SS_LMOD_SERVER,
					SM_LOG_ERROR, 6,
					"sev=ERROR, func=ss_ta_abort, ss_sess=%s, sm_s2m_abort=%m"
					, ss_sess->ssse_id, ret);
			}
		}
#endif /* SM_USE_PMILTER */

		/* XXX ... */
		/* abort all performed actions */
		if (ss_ta->ssta_state >= SSTA_ST_MAIL)
		{

			(void) sm_rcb_close_decn(ss_sess->ssse_rcb);
			ret = sm_s2q_dtaid(ss_sess,
				ss_sess->ssse_sctx->ssc_s2q_ctx,
				ss_sess->ssse_sctx->ssc_cnf.ss_cnf_id,
				ss_ta->ssta_id);
			if (sm_is_err(ret))
				goto ssd;
			ret = sm_w4q2s_reply(ss_sess, TMO_W4Q2S,
					ss_sess->ssse_sctx->ssc_s2q_ctx);
			if (sm_is_err(ret))
			{
				sm_log_write(ss_sess->ssse_sctx->ssc_lctx,
					SS_LCAT_SERVER, SS_LMOD_SERVER,
					SM_LOG_ERR, 3,
					"sev=ERROR, func=ss_ta_abort, ss_sess=%s, ss_ta=%s, func=ss_ta_abort, sm_w4q2s_reply=%m\n"
					, ss_sess->ssse_id, ss_ta->ssta_id
					, ret);
				goto ssd;	/* XXX ? */
			}
		}
	}
	ss_ta_clr(ss_ta);
	return SM_SUCCESS;

  ssd:
	/* always? */
	sm_str_scopy(ss_sess->ssse_wr, SS_R_SSD);
	return ret;
}

/*
**  SS_TA_INIT -- initialize transaction context
**	ss_ta is NOT allocated, it is a local variable in ss_hdl_session()
**
**	Parameters:
**		ss_sess -- SMTP server session context
**		ss_ta -- SMTP server transaction context
**
**	Returns:
**		usual return code
*/

sm_ret_T
ss_ta_init(ss_sess_P ss_sess, ss_ta_P ss_ta)
{
	sm_rpool_P rp;
	sm_ret_T ret;

	rp = NULL;
	ret = SM_SUCCESS;
	sm_memzero(ss_ta, sizeof(*ss_ta));
	if (Rpools)
	{
		rp = sm_rpool_new(NULL);
		if (rp == NULL)
		{
			ret = sm_error_temp(SM_EM_SMTPS, ENOMEM);
			goto error;
		}
		ss_ta->ssta_rpool = rp;
	}
	ret = ss_id_next(ss_sess->ssse_sctx->ssc_cnf.ss_cnf_id,
			ss_ta->ssta_id);
	if (sm_is_err(ret))
		goto error;
	SS_RCPTS_INIT(&(ss_ta->ssta_rcpts));
	ss_ta->ssta_state = SSTA_ST_INIT;

	/* inherit some flags from session; better (efficient) way?? */
	if (SSSE_IS_FLAG(ss_sess, SSSE_FL_DISCARD))
		SSTA_SET_FLAG(ss_ta, SSTA_FL_DISCARD);
	if (SSSE_IS_FLAG(ss_sess, SSSE_FL_DELAY_CHKS))
		SSTA_SET_FLAG(ss_ta, SSTA_FL_DELAY_CHKS);
#if SM_USE_PMILTER
	if (SSSE_IS_FLAG(ss_sess, SSSE_FL_PM_USE))
		SSTA_SET_FLAG(ss_ta, SSTA_FL_PM_USE);
#endif

#if SS_TA_CHECK
	ss_ta->sm_magic = SM_SS_TA_MAGIC;
#endif
	return SM_SUCCESS;

  error:
	if (rp != NULL)
	{
		sm_rpool_delete(rp);
		ss_ta->ssta_rpool = NULL;
	}
	return ret;
}

/*
**  SS_SESS_FREE -- free a SMTP server session context
**
**	Parameters:
**		ss_sess -- SMTP server session context
**
**	Returns:
**		usual return code
*/

sm_ret_T
ss_sess_free(ss_sess_P ss_sess)
{
	ss_ta_P ss_ta;

	if (ss_sess == NULL)
		return SM_SUCCESS;

	/* make sure ss_sess isn't referenced anymore */
	if (ss_sess->ssse_sctx != NULL &&
	    ss_sess->ssse_sctx->ssc_s2q_ctx != NULL)
		(void) ss_clr_sess_rq(ss_sess->ssse_sctx->ssc_s2q_ctx, ss_sess);

	/* should this free the ss_ta inside? */
	ss_ta = ss_sess->ssse_ta;
	if (ss_ta != NULL)
		(void) ss_ta_abort(ss_sess, ss_ta);
	if (ss_sess->ssse_fp != NULL)
	{
		sm_io_close(ss_sess->ssse_fp);
		ss_sess->ssse_fp = NULL;
#if SM_USE_TLS
		ss_sess->ssse_fptls = NULL;
		(void) tlsi_free(ss_sess->ssse_tlsi);
#endif
	}

#if SM_USE_SASL
	if (SSSE_IS_FLAG(ss_sess, SSSE_FL_SASL_OK))
	{
		ss_ctx_P ss_ctx;

		ss_ctx = ss_sess->ssse_sctx;
		if (ss_ctx != NULL &&
		    ss_ctx->ssc_sasl_ctx != NULL &&
		    ss_sess->ssse_sasl_conn != NULL)
		{
			sasl_dispose(&(ss_sess->ssse_sasl_conn));
		}
		SSSE_CLR_FLAG(ss_sess, SSSE_FL_SASL_OK);
	}
#endif /* SM_USE_SASL */

	SM_STR_FREE(ss_sess->ssse_rd);
	SM_STR_FREE(ss_sess->ssse_wr);
	SM_STR_FREE(ss_sess->ssse_str);
	SM_STR_FREE(ss_sess->ssse_helo);
	SM_STR_FREE(ss_sess->ssse_acc.ssa_reply_text);
	SM_CSTR_FREE(ss_sess->ssse_cltname);
	SM_RCB_FREE(ss_sess->ssse_rcb);
#if SM_USE_PMILTER
	/* SM_STR_FREE(ss_sess->ssse_ehlo_acc.ssa_reply_text); */
#endif
	if (ss_sess->ssse_cond_rd != NULL)
		st_cond_destroy(ss_sess->ssse_cond_rd);

	ss_sess->ssse_sctx = NULL;
#if SS_SESS_CHECK
	ss_sess->sm_magic = SM_MAGIC_NULL;
#endif
	sm_free_size(ss_sess, sizeof(*ss_sess));
	return SM_SUCCESS;
}

/*
**  SS_SESS_NEW -- create a new SMTP server session context
**
**	Parameters:
**		ss_ctx -- SMTP server context
**		ss_sess -- SMTP server session context
**
**	Returns:
**		usual return code
*/

sm_ret_T
ss_sess_new(ss_ctx_P ss_ctx, ss_sess_P *pss_sess)
{
	sm_str_P str;
	sm_ret_T ret;
	ss_sess_P ss_sess;
	uint u;
	extern sm_cstr_P HostnameNone;

	SM_REQUIRE(pss_sess != NULL);
	ss_sess = (ss_sess_P) sm_zalloc(sizeof(*ss_sess));
	if (ss_sess == NULL)
		goto errnomem;
	str = sm_str_new(NULL, SMTPBUFSIZE, SMTPMAXSIZE);
	if (str == NULL)
		goto errnomem;
	ss_sess->ssse_rd = str;
	str = sm_str_new(NULL, SMTPBUFSIZE, SMTPMAXSIZE);
	if (str == NULL)
		goto errnomem;
	ss_sess->ssse_wr = str;
	str = sm_str_new(NULL, SMTPBUFSIZE, SMTPMAXSIZE);
	if (str == NULL)
		goto errnomem;
	ss_sess->ssse_str = str;
	str = sm_str_new(NULL, 32, SMTPMAXSIZE);
	if (str == NULL)
		goto errnomem;
	ss_sess->ssse_helo = str;

#define NO_EHLONAME "Did_not_use_EHLO/HELO"
	sm_str_scat(ss_sess->ssse_helo, NO_EHLONAME);

	str = sm_str_new(NULL, 64, SMTPMAXSIZE);
	if (str == NULL)
		goto errnomem;

	ss_sess->ssse_acc.ssa_reply_text = str;
	ss_sess->ssse_rcb = sm_rcb_new(NULL, S2Q_RCB_SIZE, QSS_RC_MAXSZ);
	if (ss_sess->ssse_rcb == NULL)
		goto errnomem;
	ss_sess->ssse_cond_rd = st_cond_new();
	if (ss_sess->ssse_cond_rd == NULL)
	{
		ret = sm_error_temp(SM_EM_SMTPS, errno);
		goto error;
	}
	ss_sess->ssse_s2q_idx = SSSE_S2Q_IDX_NONE;
#if SS_SESS_CHECK
	ss_sess->sm_magic = SM_SS_SESS_MAGIC;
#endif
	ss_sess->ssse_cltname = SM_CSTR_DUP(HostnameNone);

#if SM_USE_TLS
	ret = tlsi_new(&(ss_sess->ssse_tlsi));
#endif

#if SM_USE_SASL
	if (SSC_IS_FLAG(ss_ctx, SSC_FL_SASL_OK))
		SSSE_SET_FLAG(ss_sess, SSSE_FL_SASL_OK);
	if (SSSE_IS_FLAG(ss_sess, SSSE_FL_SASL_OK))
	{
		ret = sasl_server_new("smtp",
			sm_str_getdata(ss_ctx->ssc_hostname),
			NULL, NULL, NULL, NULL, 0,
			&(ss_sess->ssse_sasl_conn));
		if (ret != SASL_OK)
		{
			SSSE_CLR_FLAG(ss_sess, SSSE_FL_SASL_OK);
			sm_log_write(ss_ctx->ssc_lctx,
				SS_LCAT_SERVER, SS_LMOD_SERVER,
				SM_LOG_WARN, 9,
				"sev=WARN, func=ss_sess_new, sasl_server_new=%r",
				ret);
		}
		else
		{
			sm_log_write(ss_ctx->ssc_lctx,
				SS_LCAT_SERVER, SS_LMOD_SERVER,
				SM_LOG_INFO, 19,
				"sev=INFO, func=ss_sess_new, sasl_server_new=%p",
				ss_sess->ssse_sasl_conn);
		}
	}
	if (SSSE_IS_FLAG(ss_sess, SSSE_FL_SASL_OK))
	{

		/*
		**  SASL set properties for sasl
		**  set local/remote IP
		**
		**  XXX where exactly are these used/required?
		**  Kerberos_v4
		*/

		/* set properties */
		sm_memzero(&ss_ctx->ssc_sasl_ctx->sm_sasl_ssp,
			sizeof(ss_ctx->ssc_sasl_ctx->sm_sasl_ssp));

		ss_ctx->ssc_sasl_ctx->sm_sasl_ssp.security_flags =
				ss_ctx->ssc_sasl_ctx->sm_sasl_sec_flags
					& SASL_SEC_MAXIMUM;
		if (sasl_setprop(ss_sess->ssse_sasl_conn,
				SASL_SEC_PROPS,
				&ss_ctx->ssc_sasl_ctx->sm_sasl_ssp) != SASL_OK)
			SSSE_CLR_FLAG(ss_sess, SSSE_FL_SASL_OK);

		if (SSSE_IS_FLAG(ss_sess, SSSE_FL_SASL_OK))
		{
			int ext_ssf;

			/*
			**  external security strength factor;
			**	currently we have none so zero
			*/

			ext_ssf = 0;
			if (!((sasl_setprop(ss_sess->ssse_sasl_conn,
					SASL_SSF_EXTERNAL, &ext_ssf)
					== SASL_OK) &&
			      (sasl_setprop(ss_sess->ssse_sasl_conn,
					SASL_AUTH_EXTERNAL, NULL) == SASL_OK)))
				SSSE_CLR_FLAG(ss_sess, SSSE_FL_SASL_OK);
		}
		if (SSSE_IS_FLAG(ss_sess, SSSE_FL_SASL_OK))
		{
			ss_ctx->ssc_sasl_ctx->sm_sasl_n_mechs =
				sm_saslmechs(ss_ctx->ssc_sasl_ctx,
					ss_sess->ssse_sasl_conn,
					&(ss_sess->ssse_sasl_mech_list));
			if (ss_ctx->ssc_sasl_ctx->sm_sasl_n_mechs <= 0)
				SSSE_CLR_FLAG(ss_sess, SSSE_FL_SASL_OK);
		}
	}
	sm_log_write(ss_ctx->ssc_lctx,
		SS_LCAT_SERVER, SS_LMOD_SERVER,
		SM_LOG_INFO, 19,
		"sev=INFO, func=ss_sess_new, sasl_conn=%p, flags=%x",
		ss_sess->ssse_sasl_conn, ss_sess->ssse_flags);
#endif /* SM_USE_SASL */

	/* inherit some flags from context; better (efficient) way?? */
	if (SSC_IS_CFLAG(ss_ctx, SSC_CFL_DELAY_CHKS))
		SSSE_SET_FLAG(ss_sess, SSSE_FL_DELAY_CHKS);

	for (u = 0; u < SS_MAX_COMM_SRVS; u++)
		ss_sess->ssse_s2q_id[u] = S2Q_ID_NONE;

	ss_sess->ssse_sctx = ss_ctx;
	ret = ss_id_next(ss_ctx->ssc_cnf.ss_cnf_id, ss_sess->ssse_id);
	*pss_sess = ss_sess;
	return ret;

  errnomem:
	ret = sm_error_temp(SM_EM_SMTPS, ENOMEM);
  error:
	if (ss_sess != NULL)
	{
		/* free all allocated data! */
		SM_STR_FREE(ss_sess->ssse_rd);
		SM_STR_FREE(ss_sess->ssse_wr);
		SM_STR_FREE(ss_sess->ssse_str);
		SM_STR_FREE(ss_sess->ssse_helo);
		SM_STR_FREE(ss_sess->ssse_acc.ssa_reply_text);
		SM_CSTR_FREE(ss_sess->ssse_cltname);
		SM_RCB_FREE(ss_sess->ssse_rcb);
		if (ss_sess->ssse_cond_rd != NULL)
			st_cond_destroy(ss_sess->ssse_cond_rd);
	}

	/* complain */
	sm_log_write(ss_ctx->ssc_lctx,
		SS_LCAT_SERVER, SS_LMOD_SERVER,
		SM_LOG_ERR, 1,
		"sev=ERROR, func=ss_sess_new, status=cannot_create_session_context");
	ss_sess_free(ss_sess);
	*pss_sess = NULL;
	return ret;
}
