//
// MODTOOL
//
// Copyright (c) !998 David Lindauer (LADSOFT)
//
// see license.txt for licensing info
//
//===================================================================
//
// MODTOOL.CPP
//
// This is the windproc, the main line code (arg parsing), and the about box
//
// This program runs in two modes, interactive and batch
//
// in the interactive mode you choose selections from menus,
// in the batch mode (initiated by command line params) everything
// happens automatically *EXCEPT* the program will stop with an
// error if one occurs in the batch mode
//
#define STRICT
#include <windows.h>
#include <commctrl.h>
#include <winsock.h>
#include <stdio.h>

#include "effects.h"
#include "winconst.h"
#include "property.h"
#include "profile.h"
#include "progress.h"
#include "nntp.h"
#include "pop3.h"
#include "network.h"
#include "analyze.h"
#include "wargs.h"
#include "help.h"
#include "helpids.h"
#include "mthi.h"
#include "send.h"
#include "mailfile.h"

extern HWND hWndProgress,hWndModList;
extern int networkinuse;

HMENU hMenu;		// Main window menu bar
WNDCLASS wc; 		// class registry
HWND hWndMain; 		// Main window 
MSG hMsg; 		// Message handling loop 
HINSTANCE hInst;	// Program instance 

// char strings
char *szAppName="LADSoft_ModTool";
char *szProgTitle="ModTool";

//===================================================================
// Batch mode
//
// These are the batch mode flags
#define BF_READMAIL 1
#define BF_READNEWS 2
#define BF_SENDMAIL 4
#define BF_POSTNEWS 8
#define BF_ANALYZE  16
#define BF_SEND 32

int batchflags;		// one or more flags indicates in batch mode
int help;		// set to one if user requested help

static char rcpt[256];		// mail recepient
static char mailfile[256];	// post or mail file
static char *rcptptr;		// pointer to rcpt
//===================================================================
// argument handling
//
// prototypes for handlers
//
void boolargs(char select, char *string);
void groupsetup(char select, char *string);
void MailSetup(char select, char *string);
void PostSetup(char select, char *string);
//
// Possible arguments
//
ARGLIST ArgList[] = { 
	{'r',ARG_BOOL,boolargs },
	{'a',ARG_BOOL,boolargs },
	{'A',ARG_BOOL,boolargs },
	{'s',ARG_CONCATSTRING,MailSetup },
	{'p',ARG_CONCATSTRING,PostSetup },
	{'G',ARG_CONCATSTRING,groupsetup },
	{'H',ARG_BOOL,boolargs },
	{'S',ARG_BOOL,boolargs },
	{'?',ARG_BOOL,boolargs },
	{0,0,0} 
};
// Boolean args */
static void boolargs(char select, char *string)
{
	switch (select) {
		case 'r':
			batchflags |= BF_READMAIL;
			break;
		case 'a':
			batchflags |= BF_READNEWS;
			break;
		case 'A':
			batchflags |= BF_ANALYZE;
			break;
		case 'S':
			batchflags |= BF_SEND;
			break;
		case 'H':
		case '?':
			help = 1;
			break;
	}
}
// Mail file args
//
// This arg is in the form of:
//    -mstring1|string2
//
// string1 is the recipient
// string2 is the mail file
//
void MailSetup(char select, char *string)
{
	char *t;
 	batchflags |= BF_SENDMAIL;
	strcpy(rcpt,string);
	if ((t = strstr(rcpt,":")) != 0)
		*t++ = 0;
	if (!t)     
		t = string + strlen(string);
	strcpy(mailfile,t);
}
// Post args
//
// -pfilename
//
void PostSetup(char select, char *string)
{
 	batchflags |= BF_POSTNEWS;
	strcpy(mailfile,string);
}
// if they change the group to send to
void groupsetup(char select, char *string)
{
	strcpy(groupid,string);
}
//===================================================================
//
// Display an about box
// 
int FAR PASCAL AboutProc (HWND hWnd,UINT wmsg,WPARAM wparam,LPARAM lparam)

{

	switch (wmsg) {
		case WM_INITDIALOG:
			CenterWindow(hWnd);
			return TRUE;

		case WM_CLOSE:
			EndDialog(hWnd,0);
			break;
        
		case WM_COMMAND:
			switch(LOWORD (wparam)) {
				case IDOK:
					PostMessage(hWnd,WM_CLOSE,0,0L);
					break;
			}
			break;
	}
	return 0;

}
//===================================================================
//
// Utility to enable/disable a menu item
//
void MenuItemEnable(int control, int enabled)
{
	if (enabled)
		EnableMenuItem(hMenu,control,MF_ENABLED | MF_BYCOMMAND);
	else
		EnableMenuItem(hMenu,control,MF_GRAYED | MF_BYCOMMAND);
}
//===================================================================
//
// windows procedure
//
// handles menu items, callbacks from the winsock system
//
long FAR PASCAL  WndProc (HWND hWnd,UINT wmsg,WPARAM wparam,LPARAM lparam)

{
	switch (wmsg) {
		case WM_USER:
			// WINSOCK CALLBACK
			NetworkCallback(hWnd,WSAGETSELECTEVENT(lparam),
					WSAGETSELECTERROR(lparam));
			break;
		case WM_USENDSUCCESS:
			if (MessageSave(hWnd,TRUE) != 1)
				SendNext(hWnd);
			break;
		case WM_USENDFAILURE:
			MessageSave(hWnd,FALSE);
			SendNext(hWnd);
			break;
		case WM_SETFOCUS:
			break;
		case WM_PAINT:
			break;
		case WM_LBUTTONDOWN:
			break;
		case WM_RBUTTONDOWN:
			break;
		case WM_INITMENU:
			// Selected a menu, refresh what is viewable
			MenuItemEnable(IDM_ARCHIVE,!networkinuse && archivefromserver);
			MenuItemEnable(IDM_POP3,!networkinuse);
			MenuItemEnable(IDM_EXIT,!networkinuse);
			MenuItemEnable(IDM_PROPERTIES,!networkinuse);
			MenuItemEnable(IDM_ANALYZE,!networkinuse);
			break;
		case WM_DESTROY:
			PostQuitMessage(0);
			break;
		case WM_COMMAND:
			switch(LOWORD(wparam)) {
				case IDM_ANALYZE:
					if (ApproveAll())
						ExtendedMessageBox("Analyzsis",MB_ICONERROR,"Analysis failed");
					else
						ExtendedMessageBox("Analyzsis",MB_ICONINFORMATION,"Analysis Succeeded");
					break;
				case IDM_SEND:
					SendOutbox(hWnd);
					break;
				case IDM_POP3:
					ReceivePOP3Messages(TRUE);
					break;
				case IDM_ARCHIVE:
					ReceiveNewsMessages();
					break;
				case IDM_ABOUT:
					DialogBox(hInst,"DLG_ABOUT",hWnd,AboutProc);
					break;
				case IDM_PROPERTIES:
					DisplayProperties(hInst,hWnd);
					break;
				case IDM_EXIT:
					PostMessage(hWnd,WM_CLOSE,0,0L);
					break;
				case IDM_HELP:
					HelpIndex(hWnd);
					break;
				case IDM_OVERVIEW:
					Help(hWnd,HID_OVERVIEW);
					break;
				case IDM_USINGHELP:
					HelpWind(hWnd,HELP_INDEX);
					break;
			}
			break;
	}

	return DefWindowProc(hWnd,wmsg,wparam,lparam);

}

//===================================================================
//
// Message loop.  We reuse this several times, in contrast
// to normal windows programs
//
// modeless dialogs:
//   progress bar
//   nested group selection dialog box in properties page
//
void ScanMessages(void)
{
	while ( GetMessage(&hMsg,0,0,0) )
		if ((!IsWindow(hWndProgress) || !IsDialogMessage(hWndProgress,&hMsg))
			&& (!IsWindow(hWndModList)|| !IsDialogMessage(hWndModList,&hMsg))) {
			TranslateMessage(&hMsg);
			DispatchMessage(&hMsg);
		}
}
//===================================================================
//
// winmain
//
// read default group from ini file, parse command arguments,
// display a window, then enter batch or interactive mode based on
// the batchflags
int FAR PASCAL WinMain(HINSTANCE hInstance,HINSTANCE parent,char *cmdline,int l)

{
	char **argv;
	int argc,argerr = 0;

	// register class
	wc.lpszClassName = szAppName;
	hInst = wc.hInstance	 = hInstance;
	wc.lpfnWndProc   = WndProc;
	wc.hCursor	   = LoadCursor(NULL, IDC_ARROW);
	wc.hIcon		 = 0;
	wc.lpszMenuName  = szAppName;
	wc.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
	wc.style		 = CS_HREDRAW | CS_VREDRAW;
	wc.cbClsExtra	= 0;
	wc.cbWndExtra	= 0;
	RegisterClass(&wc);

	// Common controls
	InitCommonControls();

	// Read group now because command line can override it
	ReadDefaultGroup();

	// Parse command line
	argv=CmdLineToC(&argc,cmdline);
	if (argv) {
		argerr = !parse_args(&argc,argv,1);
		if (!argerr) {
			if (argc > 1)
				argerr = 1;
		}
	}
	else
		argerr = 1;

	// Read defaults from ini file
	GetNetworkServers();
	GetFolders();
	GetGroupProperties();
	GetListProperties();

	// show the window.  Minimized if in batch mode
	hWndMain=CreateWindow(szAppName,szProgTitle,WS_OVERLAPPEDWINDOW,
			CW_USEDEFAULT,CW_USEDEFAULT,CW_USEDEFAULT,CW_USEDEFAULT,
			0,0,hInstance,0);
	hMenu = LoadMenu(hInst,"IDM_MENU");
	SetMenu(hWndMain,hMenu);
	if (!batchflags) 
		ShowWindow(hWndMain,SW_SHOW);
	else
		ShowWindow(hWndMain,SW_MINIMIZE);
	UpdateWindow(hWndMain);
	WSADATA ws;

	// Now select the mode to run in
	if (argerr || help) {
		Help(hWndMain,HID_COMMANDLINE);
	}
	// Init for network.  PPP does not try to dial up until
	// we issue the connect command (see network.cpp)
	else if (WSAStartup(0x101,&ws)) {
		NetworkMessageBox(MB_ICONERROR,"Network not available");
	}
	else if (batchflags) {
		// Note in batch mode
		// we do one thing at a time
		// and run the message loop while it is running
		// if batchflags is nonzero the various
		// network routines know to post a quit message
		// when they are done
		//
		if (batchflags & BF_READNEWS) {
			ReceiveNewsMessages();
			ScanMessages();
		}
		if (batchflags & BF_READMAIL) {
			ReceivePOP3Messages(TRUE);
			ScanMessages();
		}
		if (batchflags & BF_ANALYZE) {
			if (ApproveAll())
				ExtendedMessageBox("Analysis",MB_ICONERROR,"Analysis failed");
			// Don't need to scan messages here
			// even though we may have
		}
		if (batchflags & BF_SENDMAIL) {
			MakeProgress(0,hInst, "Sending Mail",1000);
			rcptptr = litlate(rcpt);
			SendMail(1,&rcptptr,mailfile,TRUE,TRUE);
			ScanMessages();
			DeleteProgress();
		}
		if (batchflags & BF_POSTNEWS) {
			MakeProgress(0,hInst, "Sending News",1000);
			PostNews(mailfile,TRUE);
			ScanMessages();
			DeleteProgress();
		}
		if (batchflags & BF_SEND) {
			SendOutbox(hWndMain);
			ScanMessages();
		}
	}
	else
		// This is the normal message loop for interactive mode
		ScanMessages();
	WSACleanup();		
	return 1;

}