/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.entity.dns.geoscaling;

import static org.testng.Assert.assertEquals;
import static org.testng.Assert.assertTrue;

import java.util.Date;
import java.util.LinkedHashSet;
import java.util.Set;

import org.testng.annotations.Test;
import org.apache.brooklyn.core.location.geo.HostGeoInfo;
import org.apache.brooklyn.util.core.ResourceUtils;


/**
 * {@link GeoscalingScriptGenerator} unit tests.
 */
public class GeoscalingScriptGeneratorTest {
    
    private final static Set<HostGeoInfo> HOSTS = new LinkedHashSet<HostGeoInfo>();
    static {
        HOSTS.add(new HostGeoInfo("1.2.3.100", "Server 1", 40.0, -80.0));
        HOSTS.add(new HostGeoInfo("1.2.3.101", "Server 2", 30.0, 20.0));
    }
    
    
    @Test
    public void testScriptGeneration() {
        Date generationTime = new Date(0);
        String generatedScript = GeoscalingScriptGenerator.generateScriptString(generationTime, HOSTS);
        assertTrue(generatedScript.contains("1.2.3"));
        String expectedScript = ResourceUtils.create(this).getResourceAsString("org/apache/brooklyn/entity/dns/geoscaling/expectedScript.php");
        assertEqualsNormalizedEol(generatedScript, expectedScript);
        //also make sure leading slash is allowed
        String expectedScript2 = ResourceUtils.create(this).getResourceAsString("org/apache/brooklyn/entity/dns/geoscaling/expectedScript.php");
        assertEqualsNormalizedEol(generatedScript, expectedScript2);
    }


    private void assertEqualsNormalizedEol(String generatedScript, String expectedScript) {
        assertEquals(normalizeEol(generatedScript), normalizeEol(expectedScript));
    }


    private Object normalizeEol(String str) {
        // Remove CR in case the files are checked out on Windows.
        // That's just to satisfy the test condition, PHP doesn't care about line endings.
        return str.replace("\r\n", "\n");
    }

}
