/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.analysis.differentiation;

import java.util.Arrays;
import java.util.List;
import org.apache.commons.math3.ExtendedFieldElementAbstractTest;
import org.apache.commons.math3.TestUtils;
import org.apache.commons.math3.analysis.differentiation.DerivativeStructure;
import org.apache.commons.math3.analysis.polynomials.PolynomialFunction;
import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.exception.NumberIsTooLargeException;
import org.apache.commons.math3.random.Well1024a;
import org.apache.commons.math3.util.ArithmeticUtils;
import org.apache.commons.math3.util.CombinatoricsUtils;
import org.apache.commons.math3.util.FastMath;
import org.junit.Assert;
import org.junit.Test;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class DerivativeStructureTest
extends ExtendedFieldElementAbstractTest<DerivativeStructure> {
    @Override
    protected DerivativeStructure build(double x) {
        return new DerivativeStructure(2, 1, 0, x);
    }

    @Test(expected=NumberIsTooLargeException.class)
    public void testWrongVariableIndex() {
        new DerivativeStructure(3, 1, 3, 1.0);
    }

    @Test(expected=DimensionMismatchException.class)
    public void testMissingOrders() {
        new DerivativeStructure(3, 1, 0, 1.0).getPartialDerivative(new int[]{0, 1});
    }

    @Test(expected=NumberIsTooLargeException.class)
    public void testTooLargeOrder() {
        new DerivativeStructure(3, 1, 0, 1.0).getPartialDerivative(new int[]{1, 1, 2});
    }

    @Test
    public void testVariableWithoutDerivative0() {
        DerivativeStructure v = new DerivativeStructure(1, 0, 0, 1.0);
        Assert.assertEquals((double)1.0, (double)v.getValue(), (double)1.0E-15);
    }

    @Test(expected=NumberIsTooLargeException.class)
    public void testVariableWithoutDerivative1() {
        DerivativeStructure v = new DerivativeStructure(1, 0, 0, 1.0);
        Assert.assertEquals((double)1.0, (double)v.getPartialDerivative(new int[]{1}), (double)1.0E-15);
    }

    @Test
    public void testVariable() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 0, 1.0), 1.0, 1.0, 0.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 1, 2.0), 2.0, 0.0, 1.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 2, 3.0), 3.0, 0.0, 0.0, 1.0);
        }
    }

    @Test
    public void testConstant() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            this.checkF0F1(new DerivativeStructure(3, maxOrder, Math.PI), Math.PI, 0.0, 0.0, 0.0);
        }
    }

    @Test
    public void testCreateConstant() {
        DerivativeStructure a = new DerivativeStructure(3, 2, 0, 1.3);
        DerivativeStructure b = a.createConstant(2.5);
        Assert.assertEquals((long)a.getFreeParameters(), (long)b.getFreeParameters());
        Assert.assertEquals((long)a.getOrder(), (long)b.getOrder());
        this.checkEquals(((DerivativeStructure)a.getField().getOne()).multiply(2.5), b, 1.0E-15);
    }

    @Test
    public void testPrimitiveAdd() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 0, 1.0).add(5.0), 6.0, 1.0, 0.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 1, 2.0).add(5.0), 7.0, 0.0, 1.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 2, 3.0).add(5.0), 8.0, 0.0, 0.0, 1.0);
        }
    }

    @Test
    public void testAdd() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            DerivativeStructure x = new DerivativeStructure(3, maxOrder, 0, 1.0);
            DerivativeStructure y = new DerivativeStructure(3, maxOrder, 1, 2.0);
            DerivativeStructure z = new DerivativeStructure(3, maxOrder, 2, 3.0);
            DerivativeStructure xyz = x.add(y.add(z));
            this.checkF0F1(xyz, x.getValue() + y.getValue() + z.getValue(), 1.0, 1.0, 1.0);
        }
    }

    @Test
    public void testPrimitiveSubtract() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 0, 1.0).subtract(5.0), -4.0, 1.0, 0.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 1, 2.0).subtract(5.0), -3.0, 0.0, 1.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 2, 3.0).subtract(5.0), -2.0, 0.0, 0.0, 1.0);
        }
    }

    @Test
    public void testSubtract() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            DerivativeStructure x = new DerivativeStructure(3, maxOrder, 0, 1.0);
            DerivativeStructure y = new DerivativeStructure(3, maxOrder, 1, 2.0);
            DerivativeStructure z = new DerivativeStructure(3, maxOrder, 2, 3.0);
            DerivativeStructure xyz = x.subtract(y.subtract(z));
            this.checkF0F1(xyz, x.getValue() - (y.getValue() - z.getValue()), 1.0, -1.0, 1.0);
        }
    }

    @Test
    public void testPrimitiveMultiply() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 0, 1.0).multiply(5), 5.0, 5.0, 0.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 1, 2.0).multiply(5), 10.0, 0.0, 5.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 2, 3.0).multiply(5), 15.0, 0.0, 0.0, 5.0);
        }
    }

    @Test
    public void testMultiply() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            DerivativeStructure x = new DerivativeStructure(3, maxOrder, 0, 1.0);
            DerivativeStructure y = new DerivativeStructure(3, maxOrder, 1, 2.0);
            DerivativeStructure z = new DerivativeStructure(3, maxOrder, 2, 3.0);
            DerivativeStructure xyz = x.multiply(y.multiply(z));
            for (int i = 0; i <= maxOrder; ++i) {
                for (int j = 0; j <= maxOrder; ++j) {
                    for (int k = 0; k <= maxOrder; ++k) {
                        if (i + j + k > maxOrder) continue;
                        Assert.assertEquals((double)((i == 0 ? x.getValue() : (i == 1 ? 1.0 : 0.0)) * (j == 0 ? y.getValue() : (j == 1 ? 1.0 : 0.0)) * (k == 0 ? z.getValue() : (k == 1 ? 1.0 : 0.0))), (double)xyz.getPartialDerivative(new int[]{i, j, k}), (double)1.0E-15);
                    }
                }
            }
        }
    }

    @Test
    public void testNegate() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 0, 1.0).negate(), -1.0, -1.0, 0.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 1, 2.0).negate(), -2.0, 0.0, -1.0, 0.0);
            this.checkF0F1(new DerivativeStructure(3, maxOrder, 2, 3.0).negate(), -3.0, 0.0, 0.0, -1.0);
        }
    }

    @Test
    public void testReciprocal() {
        for (double x = 0.1; x < 1.2; x += 0.1) {
            DerivativeStructure r = new DerivativeStructure(1, 6, 0, x).reciprocal();
            Assert.assertEquals((double)(1.0 / x), (double)r.getValue(), (double)1.0E-15);
            int i = 1;
            while (i < r.getOrder()) {
                double expected = (double)((long)ArithmeticUtils.pow((int)-1, (int)i) * CombinatoricsUtils.factorial((int)i)) / FastMath.pow((double)x, (int)(i + 1));
                Assert.assertEquals((double)expected, (double)r.getPartialDerivative(new int[]{i++}), (double)(1.0E-15 * FastMath.abs((double)expected)));
            }
        }
    }

    @Test
    public void testPow() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            for (int n = 0; n < 10; ++n) {
                DerivativeStructure x = new DerivativeStructure(3, maxOrder, 0, 1.0);
                DerivativeStructure y = new DerivativeStructure(3, maxOrder, 1, 2.0);
                DerivativeStructure z = new DerivativeStructure(3, maxOrder, 2, 3.0);
                List<DerivativeStructure> list = Arrays.asList(x, y, z, x.add(y).add(z), x.multiply(y).multiply(z));
                if (n == 0) {
                    for (DerivativeStructure ds : list) {
                        this.checkEquals((DerivativeStructure)ds.getField().getOne(), ds.pow(n), 1.0E-15);
                    }
                    continue;
                }
                if (n == 1) {
                    for (DerivativeStructure ds : list) {
                        this.checkEquals(ds, ds.pow(n), 1.0E-15);
                    }
                    continue;
                }
                for (DerivativeStructure ds : list) {
                    DerivativeStructure p = (DerivativeStructure)ds.getField().getOne();
                    for (int i = 0; i < n; ++i) {
                        p = p.multiply(ds);
                    }
                    this.checkEquals(p, ds.pow(n), 1.0E-15);
                }
            }
        }
    }

    @Test
    public void testPowDoubleDS() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            DerivativeStructure x = new DerivativeStructure(3, maxOrder, 0, 0.1);
            DerivativeStructure y = new DerivativeStructure(3, maxOrder, 1, 0.2);
            DerivativeStructure z = new DerivativeStructure(3, maxOrder, 2, 0.3);
            List<DerivativeStructure> list = Arrays.asList(x, y, z, x.add(y).add(z), x.multiply(y).multiply(z));
            for (DerivativeStructure ds : list) {
                for (double a : new double[]{0.0, 0.1, 1.0, 2.0, 5.0}) {
                    DerivativeStructure reference = a == 0.0 ? (DerivativeStructure)x.getField().getZero() : new DerivativeStructure(3, maxOrder, a).pow(ds);
                    DerivativeStructure result = DerivativeStructure.pow((double)a, (DerivativeStructure)ds);
                    this.checkEquals(reference, result, 1.0E-15);
                }
            }
            DerivativeStructure negEvenInteger = DerivativeStructure.pow((double)-2.0, (DerivativeStructure)new DerivativeStructure(3, maxOrder, 0, 2.0));
            Assert.assertEquals((double)4.0, (double)negEvenInteger.getValue(), (double)1.0E-15);
            Assert.assertTrue((boolean)Double.isNaN(negEvenInteger.getPartialDerivative(new int[]{1, 0, 0})));
            DerivativeStructure negOddInteger = DerivativeStructure.pow((double)-2.0, (DerivativeStructure)new DerivativeStructure(3, maxOrder, 0, 3.0));
            Assert.assertEquals((double)-8.0, (double)negOddInteger.getValue(), (double)1.0E-15);
            Assert.assertTrue((boolean)Double.isNaN(negOddInteger.getPartialDerivative(new int[]{1, 0, 0})));
            DerivativeStructure negNonInteger = DerivativeStructure.pow((double)-2.0, (DerivativeStructure)new DerivativeStructure(3, maxOrder, 0, 2.001));
            Assert.assertTrue((boolean)Double.isNaN(negNonInteger.getValue()));
            Assert.assertTrue((boolean)Double.isNaN(negNonInteger.getPartialDerivative(new int[]{1, 0, 0})));
            DerivativeStructure zeroNeg = DerivativeStructure.pow((double)0.0, (DerivativeStructure)new DerivativeStructure(3, maxOrder, 0, -1.0));
            Assert.assertTrue((boolean)Double.isNaN(zeroNeg.getValue()));
            Assert.assertTrue((boolean)Double.isNaN(zeroNeg.getPartialDerivative(new int[]{1, 0, 0})));
            DerivativeStructure posNeg = DerivativeStructure.pow((double)2.0, (DerivativeStructure)new DerivativeStructure(3, maxOrder, 0, -2.0));
            Assert.assertEquals((double)0.25, (double)posNeg.getValue(), (double)1.0E-15);
            Assert.assertEquals((double)(FastMath.log((double)2.0) / 4.0), (double)posNeg.getPartialDerivative(new int[]{1, 0, 0}), (double)1.0E-15);
            DerivativeStructure zeroZero = DerivativeStructure.pow((double)0.0, (DerivativeStructure)new DerivativeStructure(3, maxOrder, 0, 0.0));
            Assert.assertEquals((double)1.0, (double)zeroZero.getValue(), (double)1.0E-15);
            Assert.assertEquals((double)Double.NEGATIVE_INFINITY, (double)zeroZero.getPartialDerivative(new int[]{1, 0, 0}), (double)1.0E-15);
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{0, 1, 0})));
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{0, 0, 1})));
            if (maxOrder <= 1) continue;
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{2, 0, 0})));
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{0, 2, 0})));
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{0, 0, 2})));
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{1, 1, 0})));
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{0, 1, 1})));
            Assert.assertTrue((boolean)Double.isNaN(zeroZero.getPartialDerivative(new int[]{1, 1, 0})));
        }
    }

    @Test
    public void testExpression() {
        double epsilon = 2.5E-13;
        for (double x = 0.0; x < 2.0; x += 0.2) {
            DerivativeStructure dsX = new DerivativeStructure(3, 5, 0, x);
            for (double y = 0.0; y < 2.0; y += 0.2) {
                DerivativeStructure dsY = new DerivativeStructure(3, 5, 1, y);
                for (double z = 0.0; z > -2.0; z -= 0.2) {
                    DerivativeStructure dsZ = new DerivativeStructure(3, 5, 2, z);
                    DerivativeStructure ds = new DerivativeStructure(1.0, dsX, 5.0, dsX.multiply(dsY), -2.0, dsZ, 1.0, new DerivativeStructure(8.0, dsZ.multiply(dsX), -1.0, dsY).pow(3));
                    DerivativeStructure dsOther = new DerivativeStructure(1.0, dsX, 5.0, dsX.multiply(dsY), -2.0, dsZ).add(new DerivativeStructure(8.0, dsZ.multiply(dsX), -1.0, dsY).pow(3));
                    double f = x + 5.0 * x * y - 2.0 * z + FastMath.pow((double)(8.0 * z * x - y), (int)3);
                    Assert.assertEquals((double)f, (double)ds.getValue(), (double)FastMath.abs((double)(epsilon * f)));
                    Assert.assertEquals((double)f, (double)dsOther.getValue(), (double)FastMath.abs((double)(epsilon * f)));
                    double dfdx = 1.0 + 5.0 * y + 24.0 * z * FastMath.pow((double)(8.0 * z * x - y), (int)2);
                    Assert.assertEquals((double)dfdx, (double)ds.getPartialDerivative(new int[]{1, 0, 0}), (double)FastMath.abs((double)(epsilon * dfdx)));
                    Assert.assertEquals((double)dfdx, (double)dsOther.getPartialDerivative(new int[]{1, 0, 0}), (double)FastMath.abs((double)(epsilon * dfdx)));
                    double dfdxdy = 5.0 + 48.0 * z * (y - 8.0 * z * x);
                    Assert.assertEquals((double)dfdxdy, (double)ds.getPartialDerivative(new int[]{1, 1, 0}), (double)FastMath.abs((double)(epsilon * dfdxdy)));
                    Assert.assertEquals((double)dfdxdy, (double)dsOther.getPartialDerivative(new int[]{1, 1, 0}), (double)FastMath.abs((double)(epsilon * dfdxdy)));
                    double dfdxdydz = 48.0 * (y - 16.0 * z * x);
                    Assert.assertEquals((double)dfdxdydz, (double)ds.getPartialDerivative(new int[]{1, 1, 1}), (double)FastMath.abs((double)(epsilon * dfdxdydz)));
                    Assert.assertEquals((double)dfdxdydz, (double)dsOther.getPartialDerivative(new int[]{1, 1, 1}), (double)FastMath.abs((double)(epsilon * dfdxdydz)));
                }
            }
        }
    }

    @Test
    public void testCompositionOneVariableX() {
        double epsilon = 1.0E-13;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.1) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                for (double y = 0.1; y < 1.2; y += 0.1) {
                    DerivativeStructure dsY = new DerivativeStructure(1, maxOrder, y);
                    DerivativeStructure f = dsX.divide(dsY).sqrt();
                    double f0 = FastMath.sqrt((double)(x / y));
                    Assert.assertEquals((double)f0, (double)f.getValue(), (double)FastMath.abs((double)(epsilon * f0)));
                    if (f.getOrder() <= 0) continue;
                    double f1 = 1.0 / (2.0 * FastMath.sqrt((double)(x * y)));
                    Assert.assertEquals((double)f1, (double)f.getPartialDerivative(new int[]{1}), (double)FastMath.abs((double)(epsilon * f1)));
                    if (f.getOrder() <= 1) continue;
                    double f2 = -f1 / (2.0 * x);
                    Assert.assertEquals((double)f2, (double)f.getPartialDerivative(new int[]{2}), (double)FastMath.abs((double)(epsilon * f2)));
                    if (f.getOrder() <= 2) continue;
                    double f3 = (f0 + x / (2.0 * y * f0)) / (4.0 * x * x * x);
                    Assert.assertEquals((double)f3, (double)f.getPartialDerivative(new int[]{3}), (double)FastMath.abs((double)(epsilon * f3)));
                }
            }
        }
    }

    @Test
    public void testTrigo() {
        double epsilon = 2.0E-12;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.1) {
                DerivativeStructure dsX = new DerivativeStructure(3, maxOrder, 0, x);
                for (double y = 0.1; y < 1.2; y += 0.1) {
                    DerivativeStructure dsY = new DerivativeStructure(3, maxOrder, 1, y);
                    for (double z = 0.1; z < 1.2; z += 0.1) {
                        DerivativeStructure dsZ = new DerivativeStructure(3, maxOrder, 2, z);
                        DerivativeStructure f = dsX.divide(dsY.cos().add(dsZ.tan())).sin();
                        double a = FastMath.cos((double)y) + FastMath.tan((double)z);
                        double f0 = FastMath.sin((double)(x / a));
                        Assert.assertEquals((double)f0, (double)f.getValue(), (double)FastMath.abs((double)(epsilon * f0)));
                        if (f.getOrder() <= 0) continue;
                        double dfdx = FastMath.cos((double)(x / a)) / a;
                        Assert.assertEquals((double)dfdx, (double)f.getPartialDerivative(new int[]{1, 0, 0}), (double)FastMath.abs((double)(epsilon * dfdx)));
                        double dfdy = x * FastMath.sin((double)y) * dfdx / a;
                        Assert.assertEquals((double)dfdy, (double)f.getPartialDerivative(new int[]{0, 1, 0}), (double)FastMath.abs((double)(epsilon * dfdy)));
                        double cz = FastMath.cos((double)z);
                        double cz2 = cz * cz;
                        double dfdz = -x * dfdx / (a * cz2);
                        Assert.assertEquals((double)dfdz, (double)f.getPartialDerivative(new int[]{0, 0, 1}), (double)FastMath.abs((double)(epsilon * dfdz)));
                        if (f.getOrder() <= 1) continue;
                        double df2dx2 = -(f0 / (a * a));
                        Assert.assertEquals((double)df2dx2, (double)f.getPartialDerivative(new int[]{2, 0, 0}), (double)FastMath.abs((double)(epsilon * df2dx2)));
                        double df2dy2 = x * FastMath.cos((double)y) * dfdx / a - x * x * FastMath.sin((double)y) * FastMath.sin((double)y) * f0 / (a * a * a * a) + 2.0 * FastMath.sin((double)y) * dfdy / a;
                        Assert.assertEquals((double)df2dy2, (double)f.getPartialDerivative(new int[]{0, 2, 0}), (double)FastMath.abs((double)(epsilon * df2dy2)));
                        double c4 = cz2 * cz2;
                        double df2dz2 = x * (2.0 * a * (1.0 - a * cz * FastMath.sin((double)z)) * dfdx - x * f0 / a) / (a * a * a * c4);
                        Assert.assertEquals((double)df2dz2, (double)f.getPartialDerivative(new int[]{0, 0, 2}), (double)FastMath.abs((double)(epsilon * df2dz2)));
                        double df2dxdy = dfdy / x - x * FastMath.sin((double)y) * f0 / (a * a * a);
                        Assert.assertEquals((double)df2dxdy, (double)f.getPartialDerivative(new int[]{1, 1, 0}), (double)FastMath.abs((double)(epsilon * df2dxdy)));
                    }
                }
            }
        }
    }

    @Test
    public void testSqrtDefinition() {
        double[] epsilon = new double[]{5.0E-16, 5.0E-16, 2.0E-15, 5.0E-14, 2.0E-12};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure sqrt1 = dsX.pow(0.5);
                DerivativeStructure sqrt2 = dsX.sqrt();
                DerivativeStructure zero = sqrt1.subtract(sqrt2);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testRootNSingularity() {
        for (int n = 2; n < 10; ++n) {
            for (int maxOrder = 0; maxOrder < 12; ++maxOrder) {
                DerivativeStructure dsZero = new DerivativeStructure(1, maxOrder, 0, 0.0);
                DerivativeStructure rootN = dsZero.rootN(n);
                Assert.assertEquals((double)0.0, (double)rootN.getValue(), (double)1.0E-20);
                if (maxOrder > 0) {
                    Assert.assertTrue((boolean)Double.isInfinite(rootN.getPartialDerivative(new int[]{1})));
                    Assert.assertTrue((rootN.getPartialDerivative(new int[]{1}) > 0.0 ? 1 : 0) != 0);
                    int order = 2;
                    while (order <= maxOrder) {
                        Assert.assertTrue((boolean)Double.isNaN(rootN.getPartialDerivative(new int[]{order++})));
                    }
                }
                double[] gDerivatives = new double[1 + maxOrder];
                gDerivatives[0] = 0.0;
                for (int k = 1; k <= maxOrder; ++k) {
                    gDerivatives[k] = FastMath.pow((double)-1.0, (int)(k + 1));
                }
                DerivativeStructure correctRoot = new DerivativeStructure(1, maxOrder, gDerivatives).rootN(n);
                Assert.assertEquals((double)0.0, (double)correctRoot.getValue(), (double)1.0E-20);
                if (maxOrder <= 0) continue;
                Assert.assertTrue((boolean)Double.isInfinite(correctRoot.getPartialDerivative(new int[]{1})));
                Assert.assertTrue((correctRoot.getPartialDerivative(new int[]{1}) > 0.0 ? 1 : 0) != 0);
                for (int order = 2; order <= maxOrder; ++order) {
                    Assert.assertTrue((boolean)Double.isInfinite(correctRoot.getPartialDerivative(new int[]{order})));
                    if (order % 2 == 0) {
                        Assert.assertTrue((correctRoot.getPartialDerivative(new int[]{order}) < 0.0 ? 1 : 0) != 0);
                        continue;
                    }
                    Assert.assertTrue((correctRoot.getPartialDerivative(new int[]{order}) > 0.0 ? 1 : 0) != 0);
                }
            }
        }
    }

    @Test
    public void testSqrtPow2() {
        double[] epsilon = new double[]{1.0E-16, 3.0E-16, 2.0E-15, 6.0E-14, 6.0E-12};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.multiply(dsX).sqrt();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testCbrtDefinition() {
        double[] epsilon = new double[]{4.0E-16, 9.0E-16, 6.0E-15, 2.0E-13, 4.0E-12};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure cbrt1 = dsX.pow(0.3333333333333333);
                DerivativeStructure cbrt2 = dsX.cbrt();
                DerivativeStructure zero = cbrt1.subtract(cbrt2);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testCbrtPow3() {
        double[] epsilon = new double[]{1.0E-16, 5.0E-16, 8.0E-15, 3.0E-13, 4.0E-11};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.multiply(dsX.multiply(dsX)).cbrt();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testPowReciprocalPow() {
        double[] epsilon = new double[]{2.0E-15, 2.0E-14, 3.0E-13, 8.0E-12, 3.0E-10};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.01) {
                DerivativeStructure dsX = new DerivativeStructure(2, maxOrder, 0, x);
                for (double y = 0.1; y < 1.2; y += 0.01) {
                    DerivativeStructure dsY = new DerivativeStructure(2, maxOrder, 1, y);
                    DerivativeStructure rebuiltX = dsX.pow(dsY).pow(dsY.reciprocal());
                    DerivativeStructure zero = rebuiltX.subtract(dsX);
                    for (int n = 0; n <= maxOrder; ++n) {
                        for (int m = 0; m <= maxOrder; ++m) {
                            if (n + m > maxOrder) continue;
                            Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n, m}), (double)epsilon[n + m]);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testHypotDefinition() {
        double epsilon = 1.0E-20;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = -1.7; x < 2.0; x += 0.2) {
                DerivativeStructure dsX = new DerivativeStructure(2, maxOrder, 0, x);
                for (double y = -1.7; y < 2.0; y += 0.2) {
                    DerivativeStructure dsY = new DerivativeStructure(2, maxOrder, 1, y);
                    DerivativeStructure hypot = DerivativeStructure.hypot((DerivativeStructure)dsY, (DerivativeStructure)dsX);
                    DerivativeStructure ref = dsX.multiply(dsX).add(dsY.multiply(dsY)).sqrt();
                    DerivativeStructure zero = hypot.subtract(ref);
                    for (int n = 0; n <= maxOrder; ++n) {
                        for (int m = 0; m <= maxOrder; ++m) {
                            if (n + m > maxOrder) continue;
                            Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n, m}), (double)epsilon);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testHypotNoOverflow() {
        DerivativeStructure dsX = new DerivativeStructure(2, 5, 0, 3.0E250);
        DerivativeStructure dsY = new DerivativeStructure(2, 5, 1, -4.0E250);
        DerivativeStructure hypot = DerivativeStructure.hypot((DerivativeStructure)dsX, (DerivativeStructure)dsY);
        Assert.assertEquals((double)5.0E250, (double)hypot.getValue(), (double)1.0E235);
        Assert.assertEquals((double)(dsX.getValue() / hypot.getValue()), (double)hypot.getPartialDerivative(new int[]{1, 0}), (double)1.0E-10);
        Assert.assertEquals((double)(dsY.getValue() / hypot.getValue()), (double)hypot.getPartialDerivative(new int[]{0, 1}), (double)1.0E-10);
        DerivativeStructure sqrt = dsX.multiply(dsX).add(dsY.multiply(dsY)).sqrt();
        Assert.assertTrue((boolean)Double.isInfinite(sqrt.getValue()));
    }

    @Test
    public void testHypotNeglectible() {
        DerivativeStructure dsSmall = new DerivativeStructure(2, 5, 0, 3.0E-10);
        DerivativeStructure dsLarge = new DerivativeStructure(2, 5, 1, -4.0E25);
        Assert.assertEquals((double)dsLarge.abs().getValue(), (double)DerivativeStructure.hypot((DerivativeStructure)dsSmall, (DerivativeStructure)dsLarge).getValue(), (double)1.0E-10);
        Assert.assertEquals((double)0.0, (double)DerivativeStructure.hypot((DerivativeStructure)dsSmall, (DerivativeStructure)dsLarge).getPartialDerivative(new int[]{1, 0}), (double)1.0E-10);
        Assert.assertEquals((double)-1.0, (double)DerivativeStructure.hypot((DerivativeStructure)dsSmall, (DerivativeStructure)dsLarge).getPartialDerivative(new int[]{0, 1}), (double)1.0E-10);
        Assert.assertEquals((double)dsLarge.abs().getValue(), (double)DerivativeStructure.hypot((DerivativeStructure)dsLarge, (DerivativeStructure)dsSmall).getValue(), (double)1.0E-10);
        Assert.assertEquals((double)0.0, (double)DerivativeStructure.hypot((DerivativeStructure)dsLarge, (DerivativeStructure)dsSmall).getPartialDerivative(new int[]{1, 0}), (double)1.0E-10);
        Assert.assertEquals((double)-1.0, (double)DerivativeStructure.hypot((DerivativeStructure)dsLarge, (DerivativeStructure)dsSmall).getPartialDerivative(new int[]{0, 1}), (double)1.0E-10);
    }

    @Test
    public void testHypotSpecial() {
        Assert.assertTrue((boolean)Double.isNaN(DerivativeStructure.hypot((DerivativeStructure)new DerivativeStructure(2, 5, 0, Double.NaN), (DerivativeStructure)new DerivativeStructure(2, 5, 0, 3.0E250)).getValue()));
        Assert.assertTrue((boolean)Double.isNaN(DerivativeStructure.hypot((DerivativeStructure)new DerivativeStructure(2, 5, 0, 3.0E250), (DerivativeStructure)new DerivativeStructure(2, 5, 0, Double.NaN)).getValue()));
        Assert.assertTrue((boolean)Double.isInfinite(DerivativeStructure.hypot((DerivativeStructure)new DerivativeStructure(2, 5, 0, Double.POSITIVE_INFINITY), (DerivativeStructure)new DerivativeStructure(2, 5, 0, 3.0E250)).getValue()));
        Assert.assertTrue((boolean)Double.isInfinite(DerivativeStructure.hypot((DerivativeStructure)new DerivativeStructure(2, 5, 0, 3.0E250), (DerivativeStructure)new DerivativeStructure(2, 5, 0, Double.POSITIVE_INFINITY)).getValue()));
    }

    @Test
    public void testPrimitiveRemainder() {
        double epsilon = 1.0E-15;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = -1.7; x < 2.0; x += 0.2) {
                DerivativeStructure dsX = new DerivativeStructure(2, maxOrder, 0, x);
                for (double y = -1.7; y < 2.0; y += 0.2) {
                    DerivativeStructure remainder = dsX.remainder(y);
                    DerivativeStructure ref = dsX.subtract(x - FastMath.IEEEremainder((double)x, (double)y));
                    DerivativeStructure zero = remainder.subtract(ref);
                    for (int n = 0; n <= maxOrder; ++n) {
                        for (int m = 0; m <= maxOrder; ++m) {
                            if (n + m > maxOrder) continue;
                            Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n, m}), (double)epsilon);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testRemainder() {
        double epsilon = 2.0E-15;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = -1.7; x < 2.0; x += 0.2) {
                DerivativeStructure dsX = new DerivativeStructure(2, maxOrder, 0, x);
                for (double y = -1.7; y < 2.0; y += 0.2) {
                    DerivativeStructure dsY = new DerivativeStructure(2, maxOrder, 1, y);
                    DerivativeStructure remainder = dsX.remainder(dsY);
                    DerivativeStructure ref = dsX.subtract(dsY.multiply((x - FastMath.IEEEremainder((double)x, (double)y)) / y));
                    DerivativeStructure zero = remainder.subtract(ref);
                    for (int n = 0; n <= maxOrder; ++n) {
                        for (int m = 0; m <= maxOrder; ++m) {
                            if (n + m > maxOrder) continue;
                            Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n, m}), (double)epsilon);
                        }
                    }
                }
            }
        }
    }

    @Override
    @Test
    public void testExp() {
        double[] epsilon = new double[]{1.0E-16, 1.0E-16, 1.0E-16, 1.0E-16, 1.0E-16};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                double refExp = FastMath.exp((double)x);
                DerivativeStructure exp = new DerivativeStructure(1, maxOrder, 0, x).exp();
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)refExp, (double)exp.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testExpm1Definition() {
        double epsilon = 3.0E-16;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure expm11 = dsX.expm1();
                DerivativeStructure expm12 = dsX.exp().subtract((DerivativeStructure)dsX.getField().getOne());
                DerivativeStructure zero = expm11.subtract(expm12);
                int n = 0;
                while (n <= maxOrder) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n++}), (double)epsilon);
                }
            }
        }
    }

    @Override
    @Test
    public void testLog() {
        double[] epsilon = new double[]{1.0E-16, 1.0E-16, 3.0E-14, 7.0E-13, 3.0E-11};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure log = new DerivativeStructure(1, maxOrder, 0, x).log();
                Assert.assertEquals((double)FastMath.log((double)x), (double)log.getValue(), (double)epsilon[0]);
                for (int n = 1; n <= maxOrder; ++n) {
                    double refDer = (double)(-CombinatoricsUtils.factorial((int)(n - 1))) / FastMath.pow((double)(-x), (int)n);
                    Assert.assertEquals((double)refDer, (double)log.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testLog1pDefinition() {
        double epsilon = 3.0E-16;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure log1p1 = dsX.log1p();
                DerivativeStructure log1p2 = dsX.add((DerivativeStructure)dsX.getField().getOne()).log();
                DerivativeStructure zero = log1p1.subtract(log1p2);
                int n = 0;
                while (n <= maxOrder) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n++}), (double)epsilon);
                }
            }
        }
    }

    @Test
    public void testLog10Definition() {
        double[] epsilon = new double[]{3.0E-16, 3.0E-16, 8.0E-15, 3.0E-13, 8.0E-12};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure log101 = dsX.log10();
                DerivativeStructure log102 = dsX.log().divide(FastMath.log((double)10.0));
                DerivativeStructure zero = log101.subtract(log102);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testLogExp() {
        double[] epsilon = new double[]{2.0E-16, 2.0E-16, 3.0E-16, 2.0E-15, 6.0E-15};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.exp().log();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testLog1pExpm1() {
        double[] epsilon = new double[]{6.0E-17, 3.0E-16, 5.0E-16, 9.0E-16, 6.0E-15};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.expm1().log1p();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testLog10Power() {
        double[] epsilon = new double[]{3.0E-16, 3.0E-16, 9.0E-16, 6.0E-15, 6.0E-14};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = new DerivativeStructure(1, maxOrder, 10.0).pow(dsX).log10();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testSinCos() {
        double epsilon = 5.0E-16;
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure sin = dsX.sin();
                DerivativeStructure cos = dsX.cos();
                double s = FastMath.sin((double)x);
                double c = FastMath.cos((double)x);
                block7: for (int n = 0; n <= maxOrder; ++n) {
                    switch (n % 4) {
                        case 0: {
                            Assert.assertEquals((double)s, (double)sin.getPartialDerivative(new int[]{n}), (double)epsilon);
                            Assert.assertEquals((double)c, (double)cos.getPartialDerivative(new int[]{n}), (double)epsilon);
                            continue block7;
                        }
                        case 1: {
                            Assert.assertEquals((double)c, (double)sin.getPartialDerivative(new int[]{n}), (double)epsilon);
                            Assert.assertEquals((double)(-s), (double)cos.getPartialDerivative(new int[]{n}), (double)epsilon);
                            continue block7;
                        }
                        case 2: {
                            Assert.assertEquals((double)(-s), (double)sin.getPartialDerivative(new int[]{n}), (double)epsilon);
                            Assert.assertEquals((double)(-c), (double)cos.getPartialDerivative(new int[]{n}), (double)epsilon);
                            continue block7;
                        }
                        default: {
                            Assert.assertEquals((double)(-c), (double)sin.getPartialDerivative(new int[]{n}), (double)epsilon);
                            Assert.assertEquals((double)s, (double)cos.getPartialDerivative(new int[]{n}), (double)epsilon);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testSinAsin() {
        double[] epsilon = new double[]{3.0E-16, 5.0E-16, 3.0E-15, 2.0E-14, 4.0E-13};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.sin().asin();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testCosAcos() {
        double[] epsilon = new double[]{6.0E-16, 6.0E-15, 2.0E-13, 4.0E-12, 2.0E-10};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.cos().acos();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testTanAtan() {
        double[] epsilon = new double[]{6.0E-17, 2.0E-16, 2.0E-15, 4.0E-14, 2.0E-12};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.tan().atan();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testTangentDefinition() {
        double[] epsilon = new double[]{5.0E-16, 2.0E-15, 3.0E-14, 5.0E-13, 2.0E-11};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure tan1 = dsX.sin().divide(dsX.cos());
                DerivativeStructure tan2 = dsX.tan();
                DerivativeStructure zero = tan1.subtract(tan2);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Override
    @Test
    public void testAtan2() {
        double[] epsilon = new double[]{5.0E-16, 3.0E-15, 2.2E-14, 1.0E-12, 8.0E-11};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = -1.7; x < 2.0; x += 0.2) {
                DerivativeStructure dsX = new DerivativeStructure(2, maxOrder, 0, x);
                for (double y = -1.7; y < 2.0; y += 0.2) {
                    DerivativeStructure dsY = new DerivativeStructure(2, maxOrder, 1, y);
                    DerivativeStructure atan2 = DerivativeStructure.atan2((DerivativeStructure)dsY, (DerivativeStructure)dsX);
                    DerivativeStructure ref = dsY.divide(dsX).atan();
                    if (x < 0.0) {
                        ref = y < 0.0 ? ref.subtract(Math.PI) : ref.add(Math.PI);
                    }
                    DerivativeStructure zero = atan2.subtract(ref);
                    for (int n = 0; n <= maxOrder; ++n) {
                        for (int m = 0; m <= maxOrder; ++m) {
                            if (n + m > maxOrder) continue;
                            Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n, m}), (double)epsilon[n + m]);
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testAtan2SpecialCases() {
        DerivativeStructure pp = DerivativeStructure.atan2((DerivativeStructure)new DerivativeStructure(2, 2, 1, 0.0), (DerivativeStructure)new DerivativeStructure(2, 2, 1, 0.0));
        Assert.assertEquals((double)0.0, (double)pp.getValue(), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)FastMath.copySign((double)1.0, (double)pp.getValue()), (double)1.0E-15);
        DerivativeStructure pn = DerivativeStructure.atan2((DerivativeStructure)new DerivativeStructure(2, 2, 1, 0.0), (DerivativeStructure)new DerivativeStructure(2, 2, 1, -0.0));
        Assert.assertEquals((double)Math.PI, (double)pn.getValue(), (double)1.0E-15);
        DerivativeStructure np = DerivativeStructure.atan2((DerivativeStructure)new DerivativeStructure(2, 2, 1, -0.0), (DerivativeStructure)new DerivativeStructure(2, 2, 1, 0.0));
        Assert.assertEquals((double)0.0, (double)np.getValue(), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)FastMath.copySign((double)1.0, (double)np.getValue()), (double)1.0E-15);
        DerivativeStructure nn = DerivativeStructure.atan2((DerivativeStructure)new DerivativeStructure(2, 2, 1, -0.0), (DerivativeStructure)new DerivativeStructure(2, 2, 1, -0.0));
        Assert.assertEquals((double)(-Math.PI), (double)nn.getValue(), (double)1.0E-15);
    }

    @Test
    public void testSinhDefinition() {
        double[] epsilon = new double[]{3.0E-16, 3.0E-16, 5.0E-16, 2.0E-15, 6.0E-15};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure sinh1 = dsX.exp().subtract(dsX.exp().reciprocal()).multiply(0.5);
                DerivativeStructure sinh2 = dsX.sinh();
                DerivativeStructure zero = sinh1.subtract(sinh2);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testCoshDefinition() {
        double[] epsilon = new double[]{3.0E-16, 3.0E-16, 5.0E-16, 2.0E-15, 6.0E-15};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure cosh1 = dsX.exp().add(dsX.exp().reciprocal()).multiply(0.5);
                DerivativeStructure cosh2 = dsX.cosh();
                DerivativeStructure zero = cosh1.subtract(cosh2);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testTanhDefinition() {
        double[] epsilon = new double[]{3.0E-16, 5.0E-16, 7.0E-16, 3.0E-15, 2.0E-14};
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure tanh1 = dsX.exp().subtract(dsX.exp().reciprocal()).divide(dsX.exp().add(dsX.exp().reciprocal()));
                DerivativeStructure tanh2 = dsX.tanh();
                DerivativeStructure zero = tanh1.subtract(tanh2);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testSinhAsinh() {
        double[] epsilon = new double[]{3.0E-16, 3.0E-16, 4.0E-16, 7.0E-16, 3.0E-15, 8.0E-15};
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.sinh().asinh();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testCoshAcosh() {
        double[] epsilon = new double[]{2.0E-15, 1.0E-14, 2.0E-13, 6.0E-12, 3.0E-10, 2.0E-8};
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.cosh().acosh();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testTanhAtanh() {
        double[] epsilon = new double[]{3.0E-16, 2.0E-16, 7.0E-16, 4.0E-15, 3.0E-14, 4.0E-13};
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.tanh().atanh();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testCompositionOneVariableY() {
        double epsilon = 1.0E-13;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.1) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, x);
                for (double y = 0.1; y < 1.2; y += 0.1) {
                    DerivativeStructure dsY = new DerivativeStructure(1, maxOrder, 0, y);
                    DerivativeStructure f = dsX.divide(dsY).sqrt();
                    double f0 = FastMath.sqrt((double)(x / y));
                    Assert.assertEquals((double)f0, (double)f.getValue(), (double)FastMath.abs((double)(epsilon * f0)));
                    if (f.getOrder() <= 0) continue;
                    double f1 = -x / (2.0 * y * y * f0);
                    Assert.assertEquals((double)f1, (double)f.getPartialDerivative(new int[]{1}), (double)FastMath.abs((double)(epsilon * f1)));
                    if (f.getOrder() <= 1) continue;
                    double f2 = (f0 - x / (4.0 * y * f0)) / (y * y);
                    Assert.assertEquals((double)f2, (double)f.getPartialDerivative(new int[]{2}), (double)FastMath.abs((double)(epsilon * f2)));
                    if (f.getOrder() <= 2) continue;
                    double f3 = (x / (8.0 * y * f0) - 2.0 * f0) / (y * y * y);
                    Assert.assertEquals((double)f3, (double)f.getPartialDerivative(new int[]{3}), (double)FastMath.abs((double)(epsilon * f3)));
                }
            }
        }
    }

    @Test
    public void testTaylorPolynomial() {
        for (double x = 0.0; x < 1.2; x += 0.1) {
            DerivativeStructure dsX = new DerivativeStructure(3, 4, 0, x);
            for (double y = 0.0; y < 1.2; y += 0.2) {
                DerivativeStructure dsY = new DerivativeStructure(3, 4, 1, y);
                for (double z = 0.0; z < 1.2; z += 0.2) {
                    DerivativeStructure dsZ = new DerivativeStructure(3, 4, 2, z);
                    DerivativeStructure f = dsX.multiply(dsY).add(dsZ).multiply(dsX).multiply(dsY);
                    for (double dx = -0.2; dx < 0.2; dx += 0.2) {
                        for (double dy = -0.2; dy < 0.2; dy += 0.1) {
                            for (double dz = -0.2; dz < 0.2; dz += 0.1) {
                                double ref = (x + dx) * (y + dy) * ((x + dx) * (y + dy) + (z + dz));
                                Assert.assertEquals((double)ref, (double)f.taylor(new double[]{dx, dy, dz}), (double)2.0E-15);
                            }
                        }
                    }
                }
            }
        }
    }

    @Test
    public void testTaylorAtan2() {
        double[] expected = new double[]{0.214, 0.0241, 0.00422, 6.48E-4, 8.04E-5};
        double x0 = 0.1;
        double y0 = -0.3;
        for (int maxOrder = 0; maxOrder < 5; ++maxOrder) {
            DerivativeStructure dsX = new DerivativeStructure(2, maxOrder, 0, x0);
            DerivativeStructure dsY = new DerivativeStructure(2, maxOrder, 1, y0);
            DerivativeStructure atan2 = DerivativeStructure.atan2((DerivativeStructure)dsY, (DerivativeStructure)dsX);
            double maxError = 0.0;
            for (double dx = -0.05; dx < 0.05; dx += 0.001) {
                for (double dy = -0.05; dy < 0.05; dy += 0.001) {
                    double ref = FastMath.atan2((double)(y0 + dy), (double)(x0 + dx));
                    maxError = FastMath.max((double)maxError, (double)FastMath.abs((double)(ref - atan2.taylor(new double[]{dx, dy}))));
                }
            }
            Assert.assertEquals((double)0.0, (double)(expected[maxOrder] - maxError), (double)(0.01 * expected[maxOrder]));
        }
    }

    @Override
    @Test
    public void testAbs() {
        DerivativeStructure minusOne = new DerivativeStructure(1, 1, 0, -1.0);
        Assert.assertEquals((double)1.0, (double)minusOne.abs().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)minusOne.abs().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        DerivativeStructure plusOne = new DerivativeStructure(1, 1, 0, 1.0);
        Assert.assertEquals((double)1.0, (double)plusOne.abs().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)plusOne.abs().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        DerivativeStructure minusZero = new DerivativeStructure(1, 1, 0, -0.0);
        Assert.assertEquals((double)0.0, (double)minusZero.abs().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)minusZero.abs().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        DerivativeStructure plusZero = new DerivativeStructure(1, 1, 0, 0.0);
        Assert.assertEquals((double)0.0, (double)plusZero.abs().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)plusZero.abs().getPartialDerivative(new int[]{1}), (double)1.0E-15);
    }

    @Override
    @Test
    public void testSignum() {
        DerivativeStructure minusOne = new DerivativeStructure(1, 1, 0, -1.0);
        Assert.assertEquals((double)-1.0, (double)minusOne.signum().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)0.0, (double)minusOne.signum().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        DerivativeStructure plusOne = new DerivativeStructure(1, 1, 0, 1.0);
        Assert.assertEquals((double)1.0, (double)plusOne.signum().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)0.0, (double)plusOne.signum().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        DerivativeStructure minusZero = new DerivativeStructure(1, 1, 0, -0.0);
        Assert.assertEquals((double)-0.0, (double)minusZero.signum().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertTrue((Double.doubleToLongBits(minusZero.signum().getValue()) < 0L ? 1 : 0) != 0);
        Assert.assertEquals((double)0.0, (double)minusZero.signum().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        DerivativeStructure plusZero = new DerivativeStructure(1, 1, 0, 0.0);
        Assert.assertEquals((double)0.0, (double)plusZero.signum().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertTrue((Double.doubleToLongBits(plusZero.signum().getValue()) == 0L ? 1 : 0) != 0);
        Assert.assertEquals((double)0.0, (double)plusZero.signum().getPartialDerivative(new int[]{1}), (double)1.0E-15);
    }

    @Test
    public void testCeilFloorRintLong() {
        DerivativeStructure x = new DerivativeStructure(1, 1, 0, -1.5);
        Assert.assertEquals((double)-1.5, (double)x.getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)x.getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)x.ceil().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)0.0, (double)x.ceil().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-2.0, (double)x.floor().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)0.0, (double)x.floor().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-2.0, (double)x.rint().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)0.0, (double)x.rint().getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-2.0, (double)x.subtract((DerivativeStructure)x.getField().getOne()).rint().getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((long)-1L, (long)x.round());
    }

    @Test
    public void testCopySign() {
        DerivativeStructure minusOne = new DerivativeStructure(1, 1, 0, -1.0);
        Assert.assertEquals((double)1.0, (double)minusOne.copySign(1.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)minusOne.copySign(1.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)minusOne.copySign(-1.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)minusOne.copySign(-1.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)minusOne.copySign(0.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)minusOne.copySign(0.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)minusOne.copySign(-0.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)minusOne.copySign(-0.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)minusOne.copySign(Double.NaN).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)minusOne.copySign(Double.NaN).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        DerivativeStructure plusOne = new DerivativeStructure(1, 1, 0, 1.0);
        Assert.assertEquals((double)1.0, (double)plusOne.copySign(1.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)plusOne.copySign(1.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)plusOne.copySign(-1.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)plusOne.copySign(-1.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)plusOne.copySign(0.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)plusOne.copySign(0.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)plusOne.copySign(-0.0).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)-1.0, (double)plusOne.copySign(-0.0).getPartialDerivative(new int[]{1}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)plusOne.copySign(Double.NaN).getPartialDerivative(new int[]{0}), (double)1.0E-15);
        Assert.assertEquals((double)1.0, (double)plusOne.copySign(Double.NaN).getPartialDerivative(new int[]{1}), (double)1.0E-15);
    }

    @Test
    public void testToDegreesDefinition() {
        double epsilon = 3.0E-16;
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                Assert.assertEquals((double)FastMath.toDegrees((double)x), (double)dsX.toDegrees().getValue(), (double)epsilon);
                for (int n = 1; n <= maxOrder; ++n) {
                    if (n == 1) {
                        Assert.assertEquals((double)57.29577951308232, (double)dsX.toDegrees().getPartialDerivative(new int[]{1}), (double)epsilon);
                        continue;
                    }
                    Assert.assertEquals((double)0.0, (double)dsX.toDegrees().getPartialDerivative(new int[]{n}), (double)epsilon);
                }
            }
        }
    }

    @Test
    public void testToRadiansDefinition() {
        double epsilon = 3.0E-16;
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                Assert.assertEquals((double)FastMath.toRadians((double)x), (double)dsX.toRadians().getValue(), (double)epsilon);
                for (int n = 1; n <= maxOrder; ++n) {
                    if (n == 1) {
                        Assert.assertEquals((double)(Math.PI / 180), (double)dsX.toRadians().getPartialDerivative(new int[]{1}), (double)epsilon);
                        continue;
                    }
                    Assert.assertEquals((double)0.0, (double)dsX.toRadians().getPartialDerivative(new int[]{n}), (double)epsilon);
                }
            }
        }
    }

    @Test
    public void testDegRad() {
        double epsilon = 3.0E-16;
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure rebuiltX = dsX.toDegrees().toRadians();
                DerivativeStructure zero = rebuiltX.subtract(dsX);
                int n = 0;
                while (n <= maxOrder) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n++}), (double)epsilon);
                }
            }
        }
    }

    @Test(expected=DimensionMismatchException.class)
    public void testComposeMismatchedDimensions() {
        new DerivativeStructure(1, 3, 0, 1.2).compose(new double[3]);
    }

    @Test
    public void testCompose() {
        double[] epsilon = new double[]{1.0E-20, 5.0E-14, 2.0E-13, 3.0E-13, 2.0E-13, 1.0E-20};
        PolynomialFunction poly = new PolynomialFunction(new double[]{1.0, 2.0, 3.0, 4.0, 5.0, 6.0});
        for (int maxOrder = 0; maxOrder < 6; ++maxOrder) {
            PolynomialFunction[] p = new PolynomialFunction[maxOrder + 1];
            p[0] = poly;
            for (int i = 1; i <= maxOrder; ++i) {
                p[i] = p[i - 1].polynomialDerivative();
            }
            for (double x = 0.1; x < 1.2; x += 0.001) {
                DerivativeStructure dsX = new DerivativeStructure(1, maxOrder, 0, x);
                DerivativeStructure dsY1 = (DerivativeStructure)dsX.getField().getZero();
                for (int i = poly.degree(); i >= 0; --i) {
                    dsY1 = dsY1.multiply(dsX).add(poly.getCoefficients()[i]);
                }
                double[] f = new double[maxOrder + 1];
                for (int i = 0; i < f.length; ++i) {
                    f[i] = p[i].value(x);
                }
                DerivativeStructure dsY2 = dsX.compose(f);
                DerivativeStructure zero = dsY1.subtract(dsY2);
                for (int n = 0; n <= maxOrder; ++n) {
                    Assert.assertEquals((double)0.0, (double)zero.getPartialDerivative(new int[]{n}), (double)epsilon[n]);
                }
            }
        }
    }

    @Test
    public void testField() {
        for (int maxOrder = 1; maxOrder < 5; ++maxOrder) {
            DerivativeStructure x = new DerivativeStructure(3, maxOrder, 0, 1.0);
            this.checkF0F1((DerivativeStructure)x.getField().getZero(), 0.0, 0.0, 0.0, 0.0);
            this.checkF0F1((DerivativeStructure)x.getField().getOne(), 1.0, 0.0, 0.0, 0.0);
            Assert.assertEquals((long)maxOrder, (long)((DerivativeStructure)x.getField().getZero()).getOrder());
            Assert.assertEquals((long)3L, (long)((DerivativeStructure)x.getField().getZero()).getFreeParameters());
            Assert.assertEquals(DerivativeStructure.class, (Object)x.getField().getRuntimeClass());
        }
    }

    @Test
    public void testOneParameterConstructor() {
        double x = 1.2;
        double cos = FastMath.cos((double)x);
        double sin = FastMath.sin((double)x);
        DerivativeStructure yRef = new DerivativeStructure(1, 4, 0, x).cos();
        try {
            new DerivativeStructure(1, 4, new double[]{0.0, 0.0});
            Assert.fail((String)"an exception should have been thrown");
        }
        catch (DimensionMismatchException dimensionMismatchException) {
        }
        catch (Exception e) {
            Assert.fail((String)("wrong exceptionc caught " + e.getClass().getName()));
        }
        double[] derivatives = new double[]{cos, -sin, -cos, sin, cos};
        DerivativeStructure y = new DerivativeStructure(1, 4, derivatives);
        this.checkEquals(yRef, y, 1.0E-15);
        TestUtils.assertEquals(derivatives, y.getAllDerivatives(), 1.0E-15);
    }

    @Test
    public void testOneOrderConstructor() {
        double x = 1.2;
        double y = 2.4;
        double z = 12.5;
        DerivativeStructure xRef = new DerivativeStructure(3, 1, 0, x);
        DerivativeStructure yRef = new DerivativeStructure(3, 1, 1, y);
        DerivativeStructure zRef = new DerivativeStructure(3, 1, 2, z);
        try {
            new DerivativeStructure(3, 1, new double[]{x + y - z, 1.0, 1.0});
            Assert.fail((String)"an exception should have been thrown");
        }
        catch (DimensionMismatchException dimensionMismatchException) {
        }
        catch (Exception e) {
            Assert.fail((String)("wrong exceptionc caught " + e.getClass().getName()));
        }
        double[] derivatives = new double[]{x + y - z, 1.0, 1.0, -1.0};
        DerivativeStructure t = new DerivativeStructure(3, 1, derivatives);
        this.checkEquals(xRef.add(yRef.subtract(zRef)), t, 1.0E-15);
        TestUtils.assertEquals(derivatives, xRef.add(yRef.subtract(zRef)).getAllDerivatives(), 1.0E-15);
    }

    @Test
    public void testLinearCombination1DSDS() {
        DerivativeStructure[] a = new DerivativeStructure[]{new DerivativeStructure(6, 1, 0, -4921140.837095533), new DerivativeStructure(6, 1, 1, -2.1512094250440013E7), new DerivativeStructure(6, 1, 2, -890093.2794263769)};
        DerivativeStructure[] b = new DerivativeStructure[]{new DerivativeStructure(6, 1, 3, -2.7238580938724895E9), new DerivativeStructure(6, 1, 4, -2.1696649213418756E9), new DerivativeStructure(6, 1, 5, 6.7496887088853004E10)};
        DerivativeStructure abSumInline = a[0].linearCombination(a[0], b[0], a[1], b[1], a[2], b[2]);
        DerivativeStructure abSumArray = a[0].linearCombination(a, b);
        Assert.assertEquals((double)abSumInline.getValue(), (double)abSumArray.getValue(), (double)0.0);
        Assert.assertEquals((double)-1.8551294182586249, (double)abSumInline.getValue(), (double)1.0E-15);
        Assert.assertEquals((double)b[0].getValue(), (double)abSumInline.getPartialDerivative(new int[]{1, 0, 0, 0, 0, 0}), (double)1.0E-15);
        Assert.assertEquals((double)b[1].getValue(), (double)abSumInline.getPartialDerivative(new int[]{0, 1, 0, 0, 0, 0}), (double)1.0E-15);
        Assert.assertEquals((double)b[2].getValue(), (double)abSumInline.getPartialDerivative(new int[]{0, 0, 1, 0, 0, 0}), (double)1.0E-15);
        Assert.assertEquals((double)a[0].getValue(), (double)abSumInline.getPartialDerivative(new int[]{0, 0, 0, 1, 0, 0}), (double)1.0E-15);
        Assert.assertEquals((double)a[1].getValue(), (double)abSumInline.getPartialDerivative(new int[]{0, 0, 0, 0, 1, 0}), (double)1.0E-15);
        Assert.assertEquals((double)a[2].getValue(), (double)abSumInline.getPartialDerivative(new int[]{0, 0, 0, 0, 0, 1}), (double)1.0E-15);
    }

    @Test
    public void testLinearCombination1DoubleDS() {
        double[] a = new double[]{-4921140.837095533, -2.1512094250440013E7, -890093.2794263769};
        DerivativeStructure[] b = new DerivativeStructure[]{new DerivativeStructure(3, 1, 0, -2.7238580938724895E9), new DerivativeStructure(3, 1, 1, -2.1696649213418756E9), new DerivativeStructure(3, 1, 2, 6.7496887088853004E10)};
        DerivativeStructure abSumInline = b[0].linearCombination(a[0], b[0], a[1], b[1], a[2], b[2]);
        DerivativeStructure abSumArray = b[0].linearCombination(a, b);
        Assert.assertEquals((double)abSumInline.getValue(), (double)abSumArray.getValue(), (double)0.0);
        Assert.assertEquals((double)-1.8551294182586249, (double)abSumInline.getValue(), (double)1.0E-15);
        Assert.assertEquals((double)a[0], (double)abSumInline.getPartialDerivative(new int[]{1, 0, 0}), (double)1.0E-15);
        Assert.assertEquals((double)a[1], (double)abSumInline.getPartialDerivative(new int[]{0, 1, 0}), (double)1.0E-15);
        Assert.assertEquals((double)a[2], (double)abSumInline.getPartialDerivative(new int[]{0, 0, 1}), (double)1.0E-15);
    }

    @Test
    public void testLinearCombination2DSDS() {
        Well1024a random = new Well1024a(-4129932346759143663L);
        for (int i = 0; i < 10000; ++i) {
            DerivativeStructure[] u = new DerivativeStructure[4];
            DerivativeStructure[] v = new DerivativeStructure[4];
            for (int j = 0; j < u.length; ++j) {
                u[j] = new DerivativeStructure(u.length, 1, j, 1.0E17 * random.nextDouble());
                v[j] = new DerivativeStructure(u.length, 1, 1.0E17 * random.nextDouble());
            }
            DerivativeStructure lin = u[0].linearCombination(u[0], v[0], u[1], v[1]);
            double ref = u[0].getValue() * v[0].getValue() + u[1].getValue() * v[1].getValue();
            Assert.assertEquals((double)ref, (double)lin.getValue(), (double)(1.0E-15 * FastMath.abs((double)ref)));
            Assert.assertEquals((double)v[0].getValue(), (double)lin.getPartialDerivative(new int[]{1, 0, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[0].getValue())));
            Assert.assertEquals((double)v[1].getValue(), (double)lin.getPartialDerivative(new int[]{0, 1, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[1].getValue())));
            lin = u[0].linearCombination(u[0], v[0], u[1], v[1], u[2], v[2]);
            ref = u[0].getValue() * v[0].getValue() + u[1].getValue() * v[1].getValue() + u[2].getValue() * v[2].getValue();
            Assert.assertEquals((double)ref, (double)lin.getValue(), (double)(1.0E-15 * FastMath.abs((double)ref)));
            Assert.assertEquals((double)v[0].getValue(), (double)lin.getPartialDerivative(new int[]{1, 0, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[0].getValue())));
            Assert.assertEquals((double)v[1].getValue(), (double)lin.getPartialDerivative(new int[]{0, 1, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[1].getValue())));
            Assert.assertEquals((double)v[2].getValue(), (double)lin.getPartialDerivative(new int[]{0, 0, 1, 0}), (double)(1.0E-15 * FastMath.abs((double)v[2].getValue())));
            lin = u[0].linearCombination(u[0], v[0], u[1], v[1], u[2], v[2], u[3], v[3]);
            ref = u[0].getValue() * v[0].getValue() + u[1].getValue() * v[1].getValue() + u[2].getValue() * v[2].getValue() + u[3].getValue() * v[3].getValue();
            Assert.assertEquals((double)ref, (double)lin.getValue(), (double)(1.0E-15 * FastMath.abs((double)ref)));
            Assert.assertEquals((double)v[0].getValue(), (double)lin.getPartialDerivative(new int[]{1, 0, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[0].getValue())));
            Assert.assertEquals((double)v[1].getValue(), (double)lin.getPartialDerivative(new int[]{0, 1, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[1].getValue())));
            Assert.assertEquals((double)v[2].getValue(), (double)lin.getPartialDerivative(new int[]{0, 0, 1, 0}), (double)(1.0E-15 * FastMath.abs((double)v[2].getValue())));
            Assert.assertEquals((double)v[3].getValue(), (double)lin.getPartialDerivative(new int[]{0, 0, 0, 1}), (double)(1.0E-15 * FastMath.abs((double)v[3].getValue())));
        }
    }

    @Test
    public void testLinearCombination2DoubleDS() {
        Well1024a random = new Well1024a(-4129932346759143663L);
        for (int i = 0; i < 10000; ++i) {
            double[] u = new double[4];
            DerivativeStructure[] v = new DerivativeStructure[4];
            for (int j = 0; j < u.length; ++j) {
                u[j] = 1.0E17 * random.nextDouble();
                v[j] = new DerivativeStructure(u.length, 1, j, 1.0E17 * random.nextDouble());
            }
            DerivativeStructure lin = v[0].linearCombination(u[0], v[0], u[1], v[1]);
            double ref = u[0] * v[0].getValue() + u[1] * v[1].getValue();
            Assert.assertEquals((double)ref, (double)lin.getValue(), (double)(1.0E-15 * FastMath.abs((double)ref)));
            Assert.assertEquals((double)u[0], (double)lin.getPartialDerivative(new int[]{1, 0, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[0].getValue())));
            Assert.assertEquals((double)u[1], (double)lin.getPartialDerivative(new int[]{0, 1, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[1].getValue())));
            lin = v[0].linearCombination(u[0], v[0], u[1], v[1], u[2], v[2]);
            ref = u[0] * v[0].getValue() + u[1] * v[1].getValue() + u[2] * v[2].getValue();
            Assert.assertEquals((double)ref, (double)lin.getValue(), (double)(1.0E-15 * FastMath.abs((double)ref)));
            Assert.assertEquals((double)u[0], (double)lin.getPartialDerivative(new int[]{1, 0, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[0].getValue())));
            Assert.assertEquals((double)u[1], (double)lin.getPartialDerivative(new int[]{0, 1, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[1].getValue())));
            Assert.assertEquals((double)u[2], (double)lin.getPartialDerivative(new int[]{0, 0, 1, 0}), (double)(1.0E-15 * FastMath.abs((double)v[2].getValue())));
            lin = v[0].linearCombination(u[0], v[0], u[1], v[1], u[2], v[2], u[3], v[3]);
            ref = u[0] * v[0].getValue() + u[1] * v[1].getValue() + u[2] * v[2].getValue() + u[3] * v[3].getValue();
            Assert.assertEquals((double)ref, (double)lin.getValue(), (double)(1.0E-15 * FastMath.abs((double)ref)));
            Assert.assertEquals((double)u[0], (double)lin.getPartialDerivative(new int[]{1, 0, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[0].getValue())));
            Assert.assertEquals((double)u[1], (double)lin.getPartialDerivative(new int[]{0, 1, 0, 0}), (double)(1.0E-15 * FastMath.abs((double)v[1].getValue())));
            Assert.assertEquals((double)u[2], (double)lin.getPartialDerivative(new int[]{0, 0, 1, 0}), (double)(1.0E-15 * FastMath.abs((double)v[2].getValue())));
            Assert.assertEquals((double)u[3], (double)lin.getPartialDerivative(new int[]{0, 0, 0, 1}), (double)(1.0E-15 * FastMath.abs((double)v[3].getValue())));
        }
    }

    @Test
    public void testSerialization() {
        DerivativeStructure a = new DerivativeStructure(3, 2, 0, 1.3);
        DerivativeStructure b = (DerivativeStructure)TestUtils.serializeAndRecover(a);
        Assert.assertEquals((long)a.getFreeParameters(), (long)b.getFreeParameters());
        Assert.assertEquals((long)a.getOrder(), (long)b.getOrder());
        this.checkEquals(a, b, 1.0E-15);
    }

    private void checkF0F1(DerivativeStructure ds, double value, double ... derivatives) {
        Assert.assertEquals((long)derivatives.length, (long)ds.getFreeParameters());
        Assert.assertEquals((double)value, (double)ds.getValue(), (double)1.0E-15);
        Assert.assertEquals((double)value, (double)ds.getPartialDerivative(new int[ds.getFreeParameters()]), (double)1.0E-15);
        for (int i = 0; i < derivatives.length; ++i) {
            int[] orders = new int[derivatives.length];
            orders[i] = 1;
            Assert.assertEquals((double)derivatives[i], (double)ds.getPartialDerivative(orders), (double)1.0E-15);
        }
    }

    private void checkEquals(DerivativeStructure ds1, DerivativeStructure ds2, double epsilon) {
        boolean increment;
        Assert.assertEquals((long)ds1.getFreeParameters(), (long)ds2.getFreeParameters());
        Assert.assertEquals((long)ds1.getOrder(), (long)ds2.getOrder());
        int[] derivatives = new int[ds1.getFreeParameters()];
        int sum = 0;
        do {
            if (sum <= ds1.getOrder()) {
                Assert.assertEquals((double)ds1.getPartialDerivative(derivatives), (double)ds2.getPartialDerivative(derivatives), (double)epsilon);
            }
            increment = true;
            sum = 0;
            for (int i = derivatives.length - 1; i >= 0; --i) {
                if (increment) {
                    if (derivatives[i] == ds1.getOrder()) {
                        derivatives[i] = 0;
                    } else {
                        int n = i;
                        derivatives[n] = derivatives[n] + 1;
                        increment = false;
                    }
                }
                sum += derivatives[i];
            }
        } while (!increment);
    }
}

