/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.numbers.fraction;

import java.io.Serializable;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.Objects;
import org.apache.commons.numbers.core.NativeOperators;
import org.apache.commons.numbers.fraction.FractionException;

public final class BigFraction
extends Number
implements Comparable<BigFraction>,
NativeOperators<BigFraction>,
Serializable {
    public static final BigFraction ZERO = new BigFraction(BigInteger.ZERO);
    public static final BigFraction ONE = new BigFraction(BigInteger.ONE);
    private static final long serialVersionUID = 20190701L;
    private static final int DEFAULT_MAX_ITERATIONS = 100;
    private static final String NOT_FINITE = "Not finite: ";
    private static final long OVERFLOW = 0x80000000L;
    private final BigInteger numerator;
    private final BigInteger denominator;

    private BigFraction(BigInteger num, BigInteger den) {
        if (den.signum() == 0) {
            throw new FractionException("Denominator must be different from 0");
        }
        BigInteger gcd = num.gcd(den);
        if (BigInteger.ONE.compareTo(gcd) < 0) {
            this.numerator = num.divide(gcd);
            this.denominator = den.divide(gcd);
        } else {
            this.numerator = num;
            this.denominator = den;
        }
    }

    private BigFraction(BigInteger num) {
        this.numerator = num;
        this.denominator = BigInteger.ONE;
    }

    private static BigFraction from(double value, double epsilon, int maxDenominator, int maxIterations) {
        long den;
        long num;
        long q2;
        long p2;
        if (!Double.isFinite(value)) {
            throw new IllegalArgumentException(NOT_FINITE + value);
        }
        if (value == 0.0) {
            return ZERO;
        }
        double absValue = Math.abs(value);
        double r0 = absValue;
        long a0 = (long)Math.floor(r0);
        if (a0 > 0x80000000L) {
            throw new FractionException("Overflow trying to convert %s to fraction (%d/%d)", value, a0, 1);
        }
        if (r0 - (double)a0 <= epsilon) {
            if (value < 0.0) {
                a0 = -a0;
            }
            return new BigFraction(BigInteger.valueOf(a0));
        }
        long maxDen = Math.abs((long)maxDenominator);
        long p0 = 1L;
        long q0 = 0L;
        long p1 = a0;
        long q1 = 1L;
        int n = 0;
        boolean stop = false;
        do {
            ++n;
            double r1 = 1.0 / (r0 - (double)a0);
            long a1 = (long)Math.floor(r1);
            p2 = a1 * p1 + p0;
            q2 = a1 * q1 + q0;
            if (Long.compareUnsigned(p2, 0x80000000L) > 0 || Long.compareUnsigned(q2, 0x80000000L) > 0) {
                if (epsilon == 0.0) {
                    p2 = p1;
                    q2 = q1;
                    break;
                }
                throw new FractionException("Overflow trying to convert %s to fraction (%d/%d)", value, p2, q2);
            }
            double convergent = (double)p2 / (double)q2;
            if (n < maxIterations && Math.abs(convergent - absValue) > epsilon && q2 < maxDen) {
                p0 = p1;
                p1 = p2;
                q0 = q1;
                q1 = q2;
                a0 = a1;
                r0 = r1;
                continue;
            }
            stop = true;
        } while (!stop);
        if (n >= maxIterations) {
            throw new FractionException("Unable to convert %s to fraction after %d iterations", value, maxIterations);
        }
        if (q2 <= maxDen) {
            num = p2;
            den = q2;
        } else {
            num = p1;
            den = q1;
        }
        if ((double)(Math.signum(num) * Math.signum(den)) != Math.signum(value)) {
            num = -num;
        }
        return new BigFraction(BigInteger.valueOf(num), BigInteger.valueOf(den));
    }

    public static BigFraction from(double value) {
        int k;
        long m;
        if (!Double.isFinite(value)) {
            throw new IllegalArgumentException(NOT_FINITE + value);
        }
        if (value == 0.0) {
            return ZERO;
        }
        long bits = Double.doubleToLongBits(value);
        long sign = bits & Long.MIN_VALUE;
        long exponent = bits & 0x7FF0000000000000L;
        long mantissa = bits & 0xFFFFFFFFFFFFFL;
        if (exponent == 0L) {
            m = mantissa;
            k = -1074;
        } else {
            m = mantissa | 0x10000000000000L;
            k = (int)(exponent >> 52) - 1075;
        }
        if (sign != 0L) {
            m = -m;
        }
        while ((m & 0x1FFFFFFFFFFFFEL) != 0L && (m & 1L) == 0L) {
            m >>= 1;
            ++k;
        }
        return k < 0 ? new BigFraction(BigInteger.valueOf(m), BigInteger.ZERO.flipBit(-k)) : new BigFraction(BigInteger.valueOf(m).multiply(BigInteger.ZERO.flipBit(k)), BigInteger.ONE);
    }

    public static BigFraction from(double value, double epsilon, int maxIterations) {
        if (maxIterations < 1) {
            throw new IllegalArgumentException("Max iterations must be strictly positive: " + maxIterations);
        }
        if (epsilon >= 0.0) {
            return BigFraction.from(value, epsilon, Integer.MIN_VALUE, maxIterations);
        }
        throw new IllegalArgumentException("Epsilon must be positive: " + maxIterations);
    }

    public static BigFraction from(double value, int maxDenominator) {
        if (maxDenominator == 0) {
            throw new IllegalArgumentException("Denominator must be different from 0");
        }
        return BigFraction.from(value, 0.0, maxDenominator, 100);
    }

    public static BigFraction of(int num) {
        if (num == 0) {
            return ZERO;
        }
        return new BigFraction(BigInteger.valueOf(num));
    }

    public static BigFraction of(long num) {
        if (num == 0L) {
            return ZERO;
        }
        return new BigFraction(BigInteger.valueOf(num));
    }

    public static BigFraction of(BigInteger num) {
        Objects.requireNonNull(num, "numerator");
        if (num.signum() == 0) {
            return ZERO;
        }
        return new BigFraction(num);
    }

    public static BigFraction of(int num, int den) {
        if (num == 0) {
            return ZERO;
        }
        return new BigFraction(BigInteger.valueOf(num), BigInteger.valueOf(den));
    }

    public static BigFraction of(long num, long den) {
        if (num == 0L) {
            return ZERO;
        }
        return new BigFraction(BigInteger.valueOf(num), BigInteger.valueOf(den));
    }

    public static BigFraction of(BigInteger num, BigInteger den) {
        if (num.signum() == 0) {
            return ZERO;
        }
        return new BigFraction(num, den);
    }

    public static BigFraction parse(String s) {
        String stripped = s.replace(",", "");
        int slashLoc = stripped.indexOf(47);
        if (slashLoc == -1) {
            return BigFraction.of(new BigInteger(stripped.trim()));
        }
        BigInteger num = new BigInteger(stripped.substring(0, slashLoc).trim());
        BigInteger denom = new BigInteger(stripped.substring(slashLoc + 1).trim());
        return BigFraction.of(num, denom);
    }

    public BigFraction zero() {
        return ZERO;
    }

    public boolean isZero() {
        return this.numerator.signum() == 0;
    }

    public BigFraction one() {
        return ONE;
    }

    public boolean isOne() {
        return this.numerator.equals(this.denominator);
    }

    public BigInteger getNumerator() {
        return this.numerator;
    }

    public int getNumeratorAsInt() {
        return this.numerator.intValue();
    }

    public long getNumeratorAsLong() {
        return this.numerator.longValue();
    }

    public BigInteger getDenominator() {
        return this.denominator;
    }

    public int getDenominatorAsInt() {
        return this.denominator.intValue();
    }

    public long getDenominatorAsLong() {
        return this.denominator.longValue();
    }

    public int signum() {
        return this.numerator.signum() * this.denominator.signum();
    }

    public BigFraction abs() {
        return this.signum() >= 0 ? this : this.negate();
    }

    public BigFraction negate() {
        return new BigFraction(this.numerator.negate(), this.denominator);
    }

    public BigFraction reciprocal() {
        return new BigFraction(this.denominator, this.numerator);
    }

    @Override
    public double doubleValue() {
        return Double.longBitsToDouble(this.toFloatingPointBits(11, 52));
    }

    @Override
    public float floatValue() {
        return Float.intBitsToFloat((int)this.toFloatingPointBits(8, 23));
    }

    @Override
    public int intValue() {
        return this.numerator.divide(this.denominator).intValue();
    }

    @Override
    public long longValue() {
        return this.numerator.divide(this.denominator).longValue();
    }

    public BigDecimal bigDecimalValue() {
        return new BigDecimal(this.numerator).divide(new BigDecimal(this.denominator));
    }

    public BigDecimal bigDecimalValue(RoundingMode roundingMode) {
        return new BigDecimal(this.numerator).divide(new BigDecimal(this.denominator), roundingMode);
    }

    public BigDecimal bigDecimalValue(int scale, RoundingMode roundingMode) {
        return new BigDecimal(this.numerator).divide(new BigDecimal(this.denominator), scale, roundingMode);
    }

    public BigFraction add(int value) {
        return this.add(BigInteger.valueOf(value));
    }

    public BigFraction add(long value) {
        return this.add(BigInteger.valueOf(value));
    }

    public BigFraction add(BigInteger value) {
        if (value.signum() == 0) {
            return this;
        }
        if (this.isZero()) {
            return BigFraction.of(value);
        }
        return BigFraction.of(this.numerator.add(this.denominator.multiply(value)), this.denominator);
    }

    public BigFraction add(BigFraction value) {
        BigInteger den;
        BigInteger num;
        if (value.isZero()) {
            return this;
        }
        if (this.isZero()) {
            return value;
        }
        if (this.denominator.equals(value.denominator)) {
            num = this.numerator.add(value.numerator);
            den = this.denominator;
        } else {
            num = this.numerator.multiply(value.denominator).add(value.numerator.multiply(this.denominator));
            den = this.denominator.multiply(value.denominator);
        }
        if (num.signum() == 0) {
            return ZERO;
        }
        return new BigFraction(num, den);
    }

    public BigFraction subtract(int value) {
        return this.subtract(BigInteger.valueOf(value));
    }

    public BigFraction subtract(long value) {
        return this.subtract(BigInteger.valueOf(value));
    }

    public BigFraction subtract(BigInteger value) {
        if (value.signum() == 0) {
            return this;
        }
        if (this.isZero()) {
            return BigFraction.of(value.negate());
        }
        return BigFraction.of(this.numerator.subtract(this.denominator.multiply(value)), this.denominator);
    }

    public BigFraction subtract(BigFraction value) {
        BigInteger den;
        BigInteger num;
        if (value.isZero()) {
            return this;
        }
        if (this.isZero()) {
            return value.negate();
        }
        if (this.denominator.equals(value.denominator)) {
            num = this.numerator.subtract(value.numerator);
            den = this.denominator;
        } else {
            num = this.numerator.multiply(value.denominator).subtract(value.numerator.multiply(this.denominator));
            den = this.denominator.multiply(value.denominator);
        }
        if (num.signum() == 0) {
            return ZERO;
        }
        return new BigFraction(num, den);
    }

    public BigFraction multiply(int value) {
        if (value == 0 || this.isZero()) {
            return ZERO;
        }
        return this.multiply(BigInteger.valueOf(value));
    }

    public BigFraction multiply(long value) {
        if (value == 0L || this.isZero()) {
            return ZERO;
        }
        return this.multiply(BigInteger.valueOf(value));
    }

    public BigFraction multiply(BigInteger value) {
        if (value.signum() == 0 || this.isZero()) {
            return ZERO;
        }
        return new BigFraction(value.multiply(this.numerator), this.denominator);
    }

    public BigFraction multiply(BigFraction value) {
        if (value.isZero() || this.isZero()) {
            return ZERO;
        }
        return new BigFraction(this.numerator.multiply(value.numerator), this.denominator.multiply(value.denominator));
    }

    public BigFraction divide(int value) {
        return this.divide(BigInteger.valueOf(value));
    }

    public BigFraction divide(long value) {
        return this.divide(BigInteger.valueOf(value));
    }

    public BigFraction divide(BigInteger value) {
        if (value.signum() == 0) {
            throw new FractionException("The value to divide by must not be zero");
        }
        if (this.isZero()) {
            return ZERO;
        }
        return new BigFraction(this.numerator, this.denominator.multiply(value));
    }

    public BigFraction divide(BigFraction value) {
        if (value.isZero()) {
            throw new FractionException("The value to divide by must not be zero");
        }
        if (this.isZero()) {
            return ZERO;
        }
        return new BigFraction(this.numerator.multiply(value.denominator), this.denominator.multiply(value.numerator));
    }

    public BigFraction pow(int exponent) {
        if (exponent == 1) {
            return this;
        }
        if (exponent == 0) {
            return ONE;
        }
        if (this.isZero()) {
            if (exponent < 0) {
                throw new FractionException("Denominator must be different from 0");
            }
            return ZERO;
        }
        if (exponent > 0) {
            return new BigFraction(this.numerator.pow(exponent), this.denominator.pow(exponent));
        }
        if (exponent == -1) {
            return this.reciprocal();
        }
        if (exponent == Integer.MIN_VALUE) {
            return new BigFraction(this.denominator.pow(Integer.MAX_VALUE).multiply(this.denominator), this.numerator.pow(Integer.MAX_VALUE).multiply(this.numerator));
        }
        return new BigFraction(this.denominator.pow(-exponent), this.numerator.pow(-exponent));
    }

    public String toString() {
        String str = this.isZero() ? "0" : (BigInteger.ONE.equals(this.denominator) ? this.numerator.toString() : this.numerator + " / " + this.denominator);
        return str;
    }

    @Override
    public int compareTo(BigFraction other) {
        int rhsSigNum;
        int lhsSigNum = this.signum();
        if (lhsSigNum != (rhsSigNum = other.signum())) {
            return lhsSigNum > rhsSigNum ? 1 : -1;
        }
        if (lhsSigNum == 0) {
            return 0;
        }
        BigInteger nOd = this.numerator.abs().multiply(other.denominator.abs());
        BigInteger dOn = this.denominator.abs().multiply(other.numerator.abs());
        return nOd.compareTo(dOn);
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other instanceof BigFraction) {
            BigFraction rhs = (BigFraction)other;
            if (this.signum() == rhs.signum()) {
                return this.numerator.abs().equals(rhs.numerator.abs()) && this.denominator.abs().equals(rhs.denominator.abs());
            }
        }
        return false;
    }

    public int hashCode() {
        int numS = this.numerator.signum();
        int denS = this.denominator.signum();
        return (31 * (31 + this.numerator.hashCode() * numS) + this.denominator.hashCode() * denS) * numS * denS;
    }

    private long toFloatingPointBits(int exponentLength, int significandLength) {
        long maxExponent;
        int exponentBias;
        long exponent;
        int quotRightShift;
        BigInteger dividend;
        BigInteger quotient;
        long significand;
        if (this.isZero()) {
            return 0L;
        }
        long sign = this.numerator.signum() * this.denominator.signum() == -1 ? 1L : 0L;
        BigInteger positiveNumerator = this.numerator.abs();
        BigInteger positiveDenominator = this.denominator.abs();
        int denRightShift = positiveDenominator.getLowestSetBit();
        BigInteger divisor = positiveDenominator.shiftRight(denRightShift);
        int numRightShift = positiveNumerator.bitLength() - divisor.bitLength() - (significandLength + 2);
        if (numRightShift > 0 && divisor.equals(BigInteger.ONE)) {
            numRightShift = Math.min(numRightShift, positiveNumerator.getLowestSetBit());
        }
        if (((significand = BigFraction.roundAndRightShift(quotient = (dividend = positiveNumerator.shiftRight(numRightShift)).divide(divisor), quotRightShift = quotient.bitLength() - (significandLength + 1), !divisor.equals(BigInteger.ONE)).longValue()) & 1L << significandLength + 1) != 0L) {
            significand >>= 1;
            ++quotRightShift;
        }
        if ((exponent = (long)numRightShift - (long)denRightShift + (long)quotRightShift + (long)significandLength + (long)(exponentBias = (1 << exponentLength - 1) - 1)) >= (maxExponent = (1L << exponentLength) - 1L)) {
            exponent = maxExponent;
            significand = 0L;
        } else if (exponent > 0L) {
            significand &= -1L >>> 64 - significandLength;
        } else {
            significand = BigFraction.roundAndRightShift(quotient, 1 - exponentBias - significandLength - (numRightShift - denRightShift), !divisor.equals(BigInteger.ONE)).longValue();
            exponent = 0L;
        }
        return sign << significandLength + exponentLength | exponent << significandLength | significand;
    }

    private static BigInteger roundAndRightShift(BigInteger value, int bits, boolean hasFractionalBits) {
        BigInteger result = value.shiftRight(bits);
        if (value.testBit(bits - 1) && (hasFractionalBits || value.getLowestSetBit() < bits - 1 || value.testBit(bits))) {
            result = result.add(BigInteger.ONE);
        }
        return result;
    }
}

