// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.analyzer;

import org.apache.doris.catalog.InfoSchemaDb;
import org.apache.doris.catalog.SchemaTable;
import org.apache.doris.nereids.exceptions.UnboundException;
import org.apache.doris.nereids.memo.GroupExpression;
import org.apache.doris.nereids.properties.LogicalProperties;
import org.apache.doris.nereids.trees.expressions.NamedExpression;
import org.apache.doris.nereids.trees.expressions.Slot;
import org.apache.doris.nereids.trees.plans.BlockFuncDepsPropagation;
import org.apache.doris.nereids.trees.plans.Plan;
import org.apache.doris.nereids.trees.plans.PlanType;
import org.apache.doris.nereids.trees.plans.algebra.Sink;
import org.apache.doris.nereids.trees.plans.commands.info.DMLCommandType;
import org.apache.doris.nereids.trees.plans.logical.UnboundLogicalSink;
import org.apache.doris.nereids.trees.plans.visitor.PlanVisitor;
import org.apache.doris.nereids.util.Utils;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Unbound black hole sink.
 * The blackhole sink is currently used in "warm up select" SQL statements to preload file block caches into tables.
 * It is planned as a terminal sink (like /dev/null),
 * meaning a "black hole" at the end of the execution plan that discards all incoming data.
 */
public class UnboundBlackholeSink<CHILD_TYPE extends Plan> extends UnboundLogicalSink<CHILD_TYPE>
        implements Unbound, Sink, BlockFuncDepsPropagation {

    /**
     * UnboundBlackholeSink Context
     */
    public static class UnboundBlackholeSinkContext {
        private boolean isForWarmUp = false;

        public UnboundBlackholeSinkContext(boolean isForWarmUp) {
            this.isForWarmUp = isForWarmUp;
        }

        public boolean isForWarmUp() {
            return isForWarmUp;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            UnboundBlackholeSinkContext that = (UnboundBlackholeSinkContext) o;
            return isForWarmUp == that.isForWarmUp;
        }

        @Override
        public int hashCode() {
            return Objects.hash(isForWarmUp);
        }
    }

    private UnboundBlackholeSinkContext context;

    /**
     * create unbound sink for blackhole sink
     */
    public UnboundBlackholeSink(CHILD_TYPE child, UnboundBlackholeSinkContext context) {
        super(ImmutableList.of(InfoSchemaDb.DATABASE_NAME, SchemaTable.BLACKHOLE_TABLE_NAME),
                PlanType.LOGICAL_UNBOUND_BLACKHOLE_SINK,
                ImmutableList.of(),
                Optional.empty(),
                Optional.empty(),
                ImmutableList.of(),
                DMLCommandType.INSERT,
                child);
        this.context = context;
    }

    /**
     * create unbound sink for blackhole sink
     */
    public UnboundBlackholeSink(Optional<GroupExpression> groupExpression,
            Optional<LogicalProperties> logicalProperties, CHILD_TYPE child, UnboundBlackholeSinkContext context) {
        super(ImmutableList.of(InfoSchemaDb.DATABASE_NAME, SchemaTable.BLACKHOLE_TABLE_NAME),
                PlanType.LOGICAL_UNBOUND_BLACKHOLE_SINK,
                ImmutableList.of(),
                groupExpression,
                logicalProperties,
                ImmutableList.of(),
                DMLCommandType.INSERT,
                child);
        this.context = context;
    }

    public UnboundBlackholeSinkContext getContext() {
        return context;
    }

    @Override
    public Plan withChildren(List<Plan> children) {
        Preconditions.checkArgument(children.size() == 1, "UnboundBlackholeSink only accepts one child");
        return new UnboundBlackholeSink<>(groupExpression, Optional.empty(), children.get(0), context);
    }

    @Override
    public <R, C> R accept(PlanVisitor<R, C> visitor, C context) {
        return visitor.visitUnboundBlackholeSink(this, context);
    }

    @Override
    public Plan withGroupExpression(Optional<GroupExpression> groupExpression) {
        return new UnboundBlackholeSink<>(groupExpression, Optional.of(getLogicalProperties()), child(), context);
    }

    @Override
    public Plan withGroupExprLogicalPropChildren(Optional<GroupExpression> groupExpression,
            Optional<LogicalProperties> logicalProperties, List<Plan> children) {
        Preconditions.checkArgument(children.size() == 1, "UnboundBlackholeSink only accepts one child");
        return new UnboundBlackholeSink<>(groupExpression, logicalProperties, children.get(0), context);
    }

    @Override
    public UnboundBlackholeSink<CHILD_TYPE> withOutputExprs(List<NamedExpression> outputExprs) {
        throw new UnboundException("could not call withOutputExprs on UnboundBlackholeSink");
    }

    @Override
    public List<Slot> computeOutput() {
        throw new UnboundException("output");
    }

    @Override
    public String toString() {
        return Utils.toSqlString("UnboundBlackholeSink[" + id.asInt() + "]");
    }
}
