/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.formats.csv;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.sql.Date;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import org.apache.flink.annotation.Internal;
import org.apache.flink.formats.common.Converter;
import org.apache.flink.formats.common.TimeFormats;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.JsonNode;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.node.ArrayNode;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.GenericArrayData;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.StringData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.ArrayType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.logical.TimeType;
import org.apache.flink.table.types.logical.utils.LogicalTypeUtils;

@Internal
public class CsvToRowDataConverters
implements Serializable {
    private static final long serialVersionUID = 1L;
    private final boolean ignoreParseErrors;

    public CsvToRowDataConverters(boolean ignoreParseErrors) {
        this.ignoreParseErrors = ignoreParseErrors;
    }

    public CsvToRowDataConverter createRowConverter(RowType rowType, boolean isTopLevel) {
        CsvToRowDataConverter[] fieldConverters = (CsvToRowDataConverter[])rowType.getFields().stream().map(RowType.RowField::getType).map(this::createNullableConverter).toArray(CsvToRowDataConverter[]::new);
        String[] fieldNames = rowType.getFieldNames().toArray(new String[0]);
        int arity = fieldNames.length;
        return jsonNode -> {
            int nodeSize = jsonNode.size();
            if (nodeSize == 0) {
                return null;
            }
            CsvToRowDataConverters.validateArity(arity, nodeSize, this.ignoreParseErrors);
            GenericRowData row = new GenericRowData(arity);
            for (int i = 0; i < arity; ++i) {
                JsonNode field = isTopLevel ? jsonNode.get(fieldNames[i]) : jsonNode.get(i);
                try {
                    if (field == null) {
                        row.setField(i, null);
                        continue;
                    }
                    row.setField(i, fieldConverters[i].convert(field));
                    continue;
                }
                catch (Throwable t) {
                    throw new RuntimeException(String.format("Fail to deserialize at field: %s.", fieldNames[i]), t);
                }
            }
            return row;
        };
    }

    private CsvToRowDataConverter createNullableConverter(LogicalType type) {
        CsvToRowDataConverter converter = this.createConverter(type);
        return jsonNode -> {
            if (jsonNode == null || jsonNode.isNull()) {
                return null;
            }
            try {
                return converter.convert(jsonNode);
            }
            catch (Throwable t) {
                if (!this.ignoreParseErrors) {
                    throw t;
                }
                return null;
            }
        };
    }

    private CsvToRowDataConverter createConverter(LogicalType type) {
        switch (type.getTypeRoot()) {
            case NULL: {
                return jsonNode -> null;
            }
            case BOOLEAN: {
                return this::convertToBoolean;
            }
            case TINYINT: {
                return jsonNode -> Byte.parseByte(jsonNode.asText().trim());
            }
            case SMALLINT: {
                return jsonNode -> Short.parseShort(jsonNode.asText().trim());
            }
            case INTEGER: 
            case INTERVAL_YEAR_MONTH: {
                return this::convertToInt;
            }
            case BIGINT: 
            case INTERVAL_DAY_TIME: {
                return this::convertToLong;
            }
            case DATE: {
                return this::convertToDate;
            }
            case TIME_WITHOUT_TIME_ZONE: {
                return this.convertToTime((TimeType)type);
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: {
                return jsonNode -> this.convertToTimestamp(jsonNode, TimeFormats.SQL_TIMESTAMP_FORMAT);
            }
            case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                return jsonNode -> this.convertToTimestamp(jsonNode, TimeFormats.SQL_TIMESTAMP_WITH_LOCAL_TIMEZONE_FORMAT);
            }
            case FLOAT: {
                return this::convertToFloat;
            }
            case DOUBLE: {
                return this::convertToDouble;
            }
            case CHAR: 
            case VARCHAR: {
                return this::convertToString;
            }
            case BINARY: 
            case VARBINARY: {
                return this::convertToBytes;
            }
            case DECIMAL: {
                return this.createDecimalConverter((DecimalType)type);
            }
            case ARRAY: {
                return this.createArrayConverter((ArrayType)type);
            }
            case ROW: {
                return this.createRowConverter((RowType)type, false);
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + type);
    }

    private boolean convertToBoolean(JsonNode jsonNode) {
        if (jsonNode.isBoolean()) {
            return jsonNode.asBoolean();
        }
        return Boolean.parseBoolean(jsonNode.asText().trim());
    }

    private int convertToInt(JsonNode jsonNode) {
        if (jsonNode.canConvertToInt()) {
            return jsonNode.asInt();
        }
        return Integer.parseInt(jsonNode.asText().trim());
    }

    private long convertToLong(JsonNode jsonNode) {
        if (jsonNode.canConvertToLong()) {
            return jsonNode.asLong();
        }
        return Long.parseLong(jsonNode.asText().trim());
    }

    private double convertToDouble(JsonNode jsonNode) {
        if (jsonNode.isDouble()) {
            return jsonNode.asDouble();
        }
        return Double.parseDouble(jsonNode.asText().trim());
    }

    private float convertToFloat(JsonNode jsonNode) {
        if (jsonNode.isDouble()) {
            return (float)jsonNode.asDouble();
        }
        return Float.parseFloat(jsonNode.asText().trim());
    }

    private int convertToDate(JsonNode jsonNode) {
        return (int)Date.valueOf(jsonNode.asText()).toLocalDate().toEpochDay();
    }

    private CsvToRowDataConverter convertToTime(TimeType timeType) {
        int precision = timeType.getPrecision();
        if (precision > 3) {
            throw new IllegalArgumentException("Csv does not support TIME type with precision: " + precision + ", it only supports precision 0 ~ 3.");
        }
        return jsonNode -> {
            LocalTime localTime = LocalTime.parse(jsonNode.asText());
            int mills = (int)(localTime.toNanoOfDay() / 1000000L);
            if (precision == 2) {
                mills = mills / 10 * 10;
            } else if (precision == 1) {
                mills = mills / 100 * 100;
            } else if (precision == 0) {
                mills = mills / 1000 * 1000;
            }
            return mills;
        };
    }

    private TimestampData convertToTimestamp(JsonNode jsonNode, DateTimeFormatter dateTimeFormatter) {
        return TimestampData.fromLocalDateTime((LocalDateTime)LocalDateTime.parse(jsonNode.asText().trim(), dateTimeFormatter));
    }

    private StringData convertToString(JsonNode jsonNode) {
        return StringData.fromString((String)jsonNode.asText());
    }

    private byte[] convertToBytes(JsonNode jsonNode) {
        try {
            return jsonNode.binaryValue();
        }
        catch (IOException e) {
            throw new CsvParseException("Unable to deserialize byte array.", e);
        }
    }

    private CsvToRowDataConverter createDecimalConverter(DecimalType decimalType) {
        int precision = decimalType.getPrecision();
        int scale = decimalType.getScale();
        return jsonNode -> {
            BigDecimal bigDecimal = jsonNode.isBigDecimal() ? jsonNode.decimalValue() : new BigDecimal(jsonNode.asText());
            return DecimalData.fromBigDecimal((BigDecimal)bigDecimal, (int)precision, (int)scale);
        };
    }

    private CsvToRowDataConverter createArrayConverter(ArrayType arrayType) {
        CsvToRowDataConverter elementConverter = this.createNullableConverter(arrayType.getElementType());
        Class elementClass = LogicalTypeUtils.toInternalConversionClass((LogicalType)arrayType.getElementType());
        return jsonNode -> {
            ArrayNode node = (ArrayNode)jsonNode;
            Object[] array = (Object[])Array.newInstance(elementClass, node.size());
            for (int i = 0; i < node.size(); ++i) {
                JsonNode innerNode = node.get(i);
                array[i] = elementConverter.convert(innerNode);
            }
            return new GenericArrayData(array);
        };
    }

    private static void validateArity(int expected, int actual, boolean ignoreParseErrors) {
        if (expected > actual && !ignoreParseErrors) {
            throw new RuntimeException("Row length mismatch. " + expected + " fields expected but was " + actual + ".");
        }
    }

    private static final class CsvParseException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        public CsvParseException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    @FunctionalInterface
    static interface CsvToRowDataConverter
    extends Converter<JsonNode, Object, Void> {
        @Override
        default public Object convert(JsonNode source, Void context) {
            return this.convert(source);
        }

        public Object convert(JsonNode var1);
    }
}

