/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.monitoring;

import com.google.common.annotations.VisibleForTesting;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.db.monitoring.Monitorable;
import org.apache.cassandra.utils.MonotonicClock;
import org.apache.cassandra.utils.NoSpamLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class MonitoringTask {
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    private static final Logger logger = LoggerFactory.getLogger(MonitoringTask.class);
    private static final NoSpamLogger noSpamLogger = NoSpamLogger.getLogger(logger, 5L, TimeUnit.MINUTES);
    private static final int REPORT_INTERVAL_MS = Math.max(0, Integer.parseInt(System.getProperty("cassandra.monitoring_report_interval_ms", "5000")));
    private static final int MAX_OPERATIONS = Integer.parseInt(System.getProperty("cassandra.monitoring_max_operations", "50"));
    @VisibleForTesting
    static MonitoringTask instance = MonitoringTask.make(REPORT_INTERVAL_MS, MAX_OPERATIONS);
    private final ScheduledFuture<?> reportingTask;
    private final OperationsQueue failedOperationsQueue;
    private final OperationsQueue slowOperationsQueue;
    private long approxLastLogTimeNanos;

    @VisibleForTesting
    static MonitoringTask make(int reportIntervalMillis, int maxTimedoutOperations) {
        if (instance != null) {
            instance.cancel();
            instance = null;
        }
        return new MonitoringTask(reportIntervalMillis, maxTimedoutOperations);
    }

    private MonitoringTask(int reportIntervalMillis, int maxOperations) {
        this.failedOperationsQueue = new OperationsQueue(maxOperations);
        this.slowOperationsQueue = new OperationsQueue(maxOperations);
        this.approxLastLogTimeNanos = MonotonicClock.approxTime.now();
        logger.info("Scheduling monitoring task with report interval of {} ms, max operations {}", (Object)reportIntervalMillis, (Object)maxOperations);
        this.reportingTask = ScheduledExecutors.scheduledTasks.scheduleWithFixedDelay(() -> this.logOperations(MonotonicClock.approxTime.now()), reportIntervalMillis, reportIntervalMillis, TimeUnit.MILLISECONDS);
    }

    public void cancel() {
        this.reportingTask.cancel(false);
    }

    static void addFailedOperation(Monitorable operation, long nowNanos) {
        MonitoringTask.instance.failedOperationsQueue.offer(new FailedOperation(operation, nowNanos));
    }

    static void addSlowOperation(Monitorable operation, long nowNanos) {
        MonitoringTask.instance.slowOperationsQueue.offer(new SlowOperation(operation, nowNanos));
    }

    @VisibleForTesting
    List<String> getFailedOperations() {
        return this.getLogMessages(this.failedOperationsQueue.popOperations());
    }

    @VisibleForTesting
    List<String> getSlowOperations() {
        return this.getLogMessages(this.slowOperationsQueue.popOperations());
    }

    private List<String> getLogMessages(AggregatedOperations operations) {
        String ret = operations.getLogMessage();
        return ret.isEmpty() ? Collections.emptyList() : Arrays.asList(ret.split("\n"));
    }

    @VisibleForTesting
    private void logOperations(long approxCurrentTimeNanos) {
        this.logSlowOperations(approxCurrentTimeNanos);
        this.logFailedOperations(approxCurrentTimeNanos);
        this.approxLastLogTimeNanos = approxCurrentTimeNanos;
    }

    @VisibleForTesting
    boolean logFailedOperations(long nowNanos) {
        AggregatedOperations failedOperations = this.failedOperationsQueue.popOperations();
        if (!failedOperations.isEmpty()) {
            long elapsedNanos = nowNanos - this.approxLastLogTimeNanos;
            noSpamLogger.warn("Some operations timed out, details available at debug level (debug.log)", new Object[0]);
            if (logger.isDebugEnabled()) {
                logger.debug("{} operations timed out in the last {} msecs:{}{}", new Object[]{failedOperations.num(), TimeUnit.NANOSECONDS.toMillis(elapsedNanos), LINE_SEPARATOR, failedOperations.getLogMessage()});
            }
            return true;
        }
        return false;
    }

    @VisibleForTesting
    boolean logSlowOperations(long approxCurrentTimeNanos) {
        AggregatedOperations slowOperations = this.slowOperationsQueue.popOperations();
        if (!slowOperations.isEmpty()) {
            long approxElapsedNanos = approxCurrentTimeNanos - this.approxLastLogTimeNanos;
            noSpamLogger.info("Some operations were slow, details available at debug level (debug.log)", new Object[0]);
            if (logger.isDebugEnabled()) {
                logger.debug("{} operations were slow in the last {} msecs:{}{}", new Object[]{slowOperations.num(), TimeUnit.NANOSECONDS.toMillis(approxElapsedNanos), LINE_SEPARATOR, slowOperations.getLogMessage()});
            }
            return true;
        }
        return false;
    }

    private static final class SlowOperation
    extends Operation {
        SlowOperation(Monitorable operation, long failedAt) {
            super(operation, failedAt);
        }

        @Override
        public String getLogMessage() {
            if (this.numTimesReported == 1) {
                return String.format("<%s>, time %d msec - slow timeout %d %s", this.name(), TimeUnit.NANOSECONDS.toMillis(this.totalTimeNanos), TimeUnit.NANOSECONDS.toMillis(this.operation.slowTimeoutNanos()), this.operation.isCrossNode() ? "msec/cross-node" : "msec");
            }
            return String.format("<%s>, was slow %d times: avg/min/max %d/%d/%d msec - slow timeout %d %s", this.name(), this.numTimesReported, TimeUnit.NANOSECONDS.toMillis(this.totalTimeNanos / (long)this.numTimesReported), TimeUnit.NANOSECONDS.toMillis(this.minTime), TimeUnit.NANOSECONDS.toMillis(this.maxTime), TimeUnit.NANOSECONDS.toMillis(this.operation.slowTimeoutNanos()), this.operation.isCrossNode() ? "msec/cross-node" : "msec");
        }
    }

    private static final class FailedOperation
    extends Operation {
        FailedOperation(Monitorable operation, long failedAtNanos) {
            super(operation, failedAtNanos);
        }

        @Override
        public String getLogMessage() {
            if (this.numTimesReported == 1) {
                return String.format("<%s>, total time %d msec, timeout %d %s", this.name(), TimeUnit.NANOSECONDS.toMillis(this.totalTimeNanos), TimeUnit.NANOSECONDS.toMillis(this.operation.timeoutNanos()), this.operation.isCrossNode() ? "msec/cross-node" : "msec");
            }
            return String.format("<%s> timed out %d times, avg/min/max %d/%d/%d msec, timeout %d %s", this.name(), this.numTimesReported, TimeUnit.NANOSECONDS.toMillis(this.totalTimeNanos / (long)this.numTimesReported), TimeUnit.NANOSECONDS.toMillis(this.minTime), TimeUnit.NANOSECONDS.toMillis(this.maxTime), TimeUnit.NANOSECONDS.toMillis(this.operation.timeoutNanos()), this.operation.isCrossNode() ? "msec/cross-node" : "msec");
        }
    }

    protected static abstract class Operation {
        final Monitorable operation;
        int numTimesReported;
        long totalTimeNanos;
        long maxTime;
        long minTime;
        private String name;

        Operation(Monitorable operation, long failedAtNanos) {
            this.operation = operation;
            this.numTimesReported = 1;
            this.minTime = this.totalTimeNanos = failedAtNanos - operation.creationTimeNanos();
            this.maxTime = this.totalTimeNanos;
        }

        public String name() {
            if (this.name == null) {
                this.name = this.operation.name();
            }
            return this.name;
        }

        void add(Operation operation) {
            ++this.numTimesReported;
            this.totalTimeNanos += operation.totalTimeNanos;
            this.maxTime = Math.max(this.maxTime, operation.maxTime);
            this.minTime = Math.min(this.minTime, operation.minTime);
        }

        public abstract String getLogMessage();
    }

    private static final class AggregatedOperations {
        private final Map<String, Operation> operations;
        private final long numDropped;

        AggregatedOperations(Map<String, Operation> operations, long numDropped) {
            this.operations = operations;
            this.numDropped = numDropped;
        }

        public boolean isEmpty() {
            return this.operations.isEmpty() && this.numDropped == 0L;
        }

        public long num() {
            return (long)this.operations.size() + this.numDropped;
        }

        String getLogMessage() {
            if (this.isEmpty()) {
                return "";
            }
            StringBuilder ret = new StringBuilder();
            this.operations.values().forEach(o -> AggregatedOperations.addOperation(ret, o));
            if (this.numDropped > 0L) {
                ret.append(LINE_SEPARATOR).append("... (").append(this.numDropped).append(" were dropped)");
            }
            return ret.toString();
        }

        private static void addOperation(StringBuilder ret, Operation operation) {
            if (ret.length() > 0) {
                ret.append(LINE_SEPARATOR);
            }
            ret.append(operation.getLogMessage());
        }
    }

    private static final class OperationsQueue {
        private final int maxOperations;
        private final BlockingQueue<Operation> queue;
        private final AtomicLong numDroppedOperations;

        OperationsQueue(int maxOperations) {
            this.maxOperations = maxOperations;
            this.queue = maxOperations > 0 ? new ArrayBlockingQueue(maxOperations) : new LinkedBlockingQueue();
            this.numDroppedOperations = new AtomicLong();
        }

        private void offer(Operation operation) {
            if (this.maxOperations == 0) {
                return;
            }
            if (!this.queue.offer(operation)) {
                this.numDroppedOperations.incrementAndGet();
            }
        }

        private AggregatedOperations popOperations() {
            Operation operation;
            HashMap<String, Operation> operations = new HashMap<String, Operation>();
            while ((operation = (Operation)this.queue.poll()) != null) {
                Operation existing = (Operation)operations.get(operation.name());
                if (existing != null) {
                    existing.add(operation);
                    continue;
                }
                operations.put(operation.name(), operation);
            }
            return new AggregatedOperations(operations, this.numDroppedOperations.getAndSet(0L));
        }
    }
}

