/**
 * Copyright (c) 2023 Kiel University and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.elk.alg.vertiflex.options;

import java.util.EnumSet;
import org.eclipse.elk.alg.vertiflex.EdgeRoutingStrategy;
import org.eclipse.elk.alg.vertiflex.VertiFlexLayoutProvider;
import org.eclipse.elk.core.AbstractLayoutProvider;
import org.eclipse.elk.core.data.ILayoutMetaDataProvider;
import org.eclipse.elk.core.data.LayoutAlgorithmData;
import org.eclipse.elk.core.math.ElkMargin;
import org.eclipse.elk.core.math.ElkPadding;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.core.options.PortConstraints;
import org.eclipse.elk.graph.properties.GraphFeature;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;

/**
 * @vertiflex.md
 */
@SuppressWarnings("all")
public class VertiFlexOptions implements ILayoutMetaDataProvider {
  /**
   * The id of the ELK VertiFlex algorithm.
   */
  public static final String ALGORITHM_ID = "org.eclipse.elk.vertiflex";

  /**
   * Default value for {@link #SPACING_NODE_NODE} with algorithm "ELK VertiFlex".
   */
  private static final double SPACING_NODE_NODE_DEFAULT = 20;

  /**
   * The minimal distance to be preserved between each two nodes.
   */
  public static final IProperty<Double> SPACING_NODE_NODE = new Property<Double>(
                                CoreOptions.SPACING_NODE_NODE,
                                SPACING_NODE_NODE_DEFAULT);

  /**
   * Default value for {@link #PADDING} with algorithm "ELK VertiFlex".
   */
  private static final ElkPadding PADDING_DEFAULT = new ElkPadding(5);

  /**
   * The padding to be left to a parent element's border when placing child elements. This can
   * also serve as an output option of a layout algorithm if node size calculation is setup
   * appropriately.
   */
  public static final IProperty<ElkPadding> PADDING = new Property<ElkPadding>(
                                CoreOptions.PADDING,
                                PADDING_DEFAULT);

  /**
   * Whether the algorithm should be run in interactive mode for the content of a parent node.
   * What this means exactly depends on how the specific algorithm interprets this option.
   * Usually in the interactive mode algorithms try to modify the current layout as little as
   * possible.
   */
  public static final IProperty<Boolean> INTERACTIVE = CoreOptions.INTERACTIVE;

  /**
   * Defines constraints of the position of the ports of a node.
   */
  public static final IProperty<PortConstraints> PORT_CONSTRAINTS = CoreOptions.PORT_CONSTRAINTS;

  /**
   * Default value for {@link #EDGE_LABELS_INLINE} with algorithm "ELK VertiFlex".
   */
  private static final boolean EDGE_LABELS_INLINE_DEFAULT = false;

  /**
   * If true, an edge label is placed directly on its edge. May only apply to center edge labels.
   * This kind of label placement is only advisable if the label's rendering is such that it is not
   * crossed by its edge and thus stays legible.
   */
  public static final IProperty<Boolean> EDGE_LABELS_INLINE = new Property<Boolean>(
                                CoreOptions.EDGE_LABELS_INLINE,
                                EDGE_LABELS_INLINE_DEFAULT);

  /**
   * Node micro layout comprises the computation of node dimensions (if requested), the placement of ports
   * and their labels, and the placement of node labels.
   * The functionality is implemented independent of any specific layout algorithm and shouldn't have any
   * negative impact on the layout algorithm's performance itself. Yet, if any unforeseen behavior occurs,
   * this option allows to deactivate the micro layout.
   */
  public static final IProperty<Boolean> OMIT_NODE_MICRO_LAYOUT = CoreOptions.OMIT_NODE_MICRO_LAYOUT;

  /**
   * Margins define additional space around the actual bounds of a graph element. For instance,
   * ports or labels being placed on the outside of a node's border might introduce such a
   * margin. The margin is used to guarantee non-overlap of other graph elements with those
   * ports or labels.
   */
  public static final IProperty<ElkMargin> MARGINS = CoreOptions.MARGINS;

  /**
   * The Y position that the node should be fixed at.
   */
  public static final IProperty<Double> VERTICAL_CONSTRAINT = VertiFlexMetaDataProvider.VERTICAL_CONSTRAINT;

  /**
   * Strategy for the layout of the children. 'straight' for straight line drawings, 'bend' for a possible bend.
   * When straight edges are prioritized the nodes will be reordered in order to guarantee that straight edges are
   * possible. If bend points are enabled on the other hand, the given model order of the nodes is maintained and
   * bend points are introduced to prevent edge node overlaps.
   */
  public static final IProperty<EdgeRoutingStrategy> LAYOUT_STRATEGY = VertiFlexMetaDataProvider.LAYOUT_STRATEGY;

  /**
   * The distance to use between nodes of different layers if no vertical constraints are set.
   */
  public static final IProperty<Double> LAYER_DISTANCE = VertiFlexMetaDataProvider.LAYER_DISTANCE;

  /**
   * Consider node model as a secondary criterion when using straight line routing.
   */
  public static final IProperty<Boolean> CONSIDER_NODE_MODEL_ORDER = VertiFlexMetaDataProvider.CONSIDER_NODE_MODEL_ORDER;

  /**
   * Layouter-specific algorithm factory.
   */
  public static class VertiflexFactory implements org.eclipse.elk.core.util.IFactory<AbstractLayoutProvider> {
    public AbstractLayoutProvider create() {
      AbstractLayoutProvider provider = new VertiFlexLayoutProvider();
      provider.initialize("");
      return provider;
    }

    public void destroy(final AbstractLayoutProvider obj) {
      obj.dispose();
    }
  }

  public void apply(final org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry registry) {
    registry.register(new LayoutAlgorithmData.Builder()
        .id("org.eclipse.elk.vertiflex")
        .name("ELK VertiFlex")
        .description("Tree layout algorithm that allows defining set vertical positions for nodes rather than automatically placing nodes on levels according to their topology.")
        .providerFactory(new VertiflexFactory())
        .category("org.eclipse.elk.tree")
        .melkBundleName(null)
        .definingBundleId("org.eclipse.elk.alg.vertiflex")
        .supportedFeatures(EnumSet.of(GraphFeature.MULTI_EDGES, GraphFeature.EDGE_LABELS))
        .create()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.spacing.nodeNode",
        SPACING_NODE_NODE_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.padding",
        PADDING_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.interactive",
        INTERACTIVE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.portConstraints",
        PORT_CONSTRAINTS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.edgeLabels.inline",
        EDGE_LABELS_INLINE_DEFAULT
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.omitNodeMicroLayout",
        OMIT_NODE_MICRO_LAYOUT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.margins",
        MARGINS.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.vertiflex.verticalConstraint",
        VERTICAL_CONSTRAINT.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.vertiflex.layoutStrategy",
        LAYOUT_STRATEGY.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.vertiflex.layerDistance",
        LAYER_DISTANCE.getDefault()
    );
    registry.addOptionSupport(
        "org.eclipse.elk.vertiflex",
        "org.eclipse.elk.vertiflex.considerNodeModelOrder",
        CONSIDER_NODE_MODEL_ORDER.getDefault()
    );
  }
}
