/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.executionframework.debugger;

import com.google.common.base.Objects;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.impl.EObjectImpl;
import org.eclipse.gemoc.executionframework.debugger.IMutableFieldExtractor;
import org.eclipse.gemoc.executionframework.debugger.MutableField;
import org.eclipse.gemoc.executionframework.engine.commons.DslHelper;
import org.eclipse.gemoc.executionframework.engine.commons.K3DslHelper;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.osgi.framework.Bundle;

@SuppressWarnings("all")
public class IntrospectiveMutableFieldExtractor implements IMutableFieldExtractor {
  private String languageName;
  
  private Map<EObject, List<MutableField>> eObjects = new HashMap<EObject, List<MutableField>>();
  
  private Map<EClass, List<Pair<Class<?>, Class<?>>>> aspectClasses = new HashMap<EClass, List<Pair<Class<?>, Class<?>>>>();
  
  public IntrospectiveMutableFieldExtractor(final String languageName) {
    this.languageName = languageName;
  }
  
  private String decapitalize(final String string) {
    final char[] c = string.toCharArray();
    c[0] = Character.toLowerCase(c[0]);
    return new String(c);
  }
  
  private String findName(final Class<?> cls, final EObject eObject) {
    try {
      final Function1<Field, Boolean> _function = new Function1<Field, Boolean>() {
        @Override
        public Boolean apply(final Field f) {
          return Boolean.valueOf(f.getName().equals("name"));
        }
      };
      final Iterable<Field> name = IterableExtensions.<Field>filter(((Iterable<Field>)Conversions.doWrapArray(cls.getDeclaredFields())), _function);
      boolean _isEmpty = IterableExtensions.isEmpty(name);
      if (_isEmpty) {
        Class<?> _superclass = cls.getSuperclass();
        boolean _notEquals = (!Objects.equal(_superclass, EObjectImpl.class));
        if (_notEquals) {
          return this.findName(cls.getSuperclass(), eObject);
        }
        return null;
      } else {
        final Field f = ((Field[])Conversions.unwrapArray(name, Field.class))[0];
        f.setAccessible(true);
        return f.get(eObject).toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private String findId(final Class<?> cls, final EObject eObject) {
    try {
      final Function1<Field, Boolean> _function = new Function1<Field, Boolean>() {
        @Override
        public Boolean apply(final Field f) {
          return Boolean.valueOf(f.getName().equals("id"));
        }
      };
      final Iterable<Field> id = IterableExtensions.<Field>filter(((Iterable<Field>)Conversions.doWrapArray(cls.getDeclaredFields())), _function);
      boolean _isEmpty = IterableExtensions.isEmpty(id);
      if (_isEmpty) {
        Class<?> _superclass = cls.getSuperclass();
        boolean _notEquals = (!Objects.equal(_superclass, EObjectImpl.class));
        if (_notEquals) {
          return this.findId(cls.getSuperclass(), eObject);
        }
        return null;
      } else {
        final Field f = ((Field[])Conversions.unwrapArray(id, Field.class))[0];
        f.setAccessible(true);
        return f.get(eObject).toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private String findDataName(final EObject eObject) {
    final String name = this.findName(eObject.getClass(), eObject);
    if ((name == null)) {
      final String id = this.findId(eObject.getClass(), eObject);
      if ((id == null)) {
        return eObject.toString();
      } else {
        String _decapitalize = this.decapitalize(eObject.eClass().getName());
        String _plus = (_decapitalize + " ");
        return (_plus + id);
      }
    } else {
      return name;
    }
  }
  
  private List<MutableField> getMutableFieldsFromAspect(final EObject eObject, final Class<?> properties, final Class<?> aspect) {
    final ArrayList<MutableField> result = new ArrayList<MutableField>();
    final Field[] fields = properties.getFields();
    boolean _isEmpty = ((List<Field>)Conversions.doWrapArray(fields)).isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Consumer<Field> _function = new Consumer<Field>() {
        @Override
        public void accept(final Field f) {
          final Function1<Method, Boolean> _function = new Function1<Method, Boolean>() {
            @Override
            public Boolean apply(final Method m) {
              return Boolean.valueOf(m.getName().equals(f.getName()));
            }
          };
          final Iterable<Method> methods = IterableExtensions.<Method>filter(((Iterable<Method>)Conversions.doWrapArray(aspect.getMethods())), _function);
          final Function1<Method, Boolean> _function_1 = new Function1<Method, Boolean>() {
            @Override
            public Boolean apply(final Method m) {
              int _parameterCount = m.getParameterCount();
              return Boolean.valueOf((_parameterCount == 1));
            }
          };
          final Method getter = IterableExtensions.<Method>findFirst(methods, _function_1);
          final Function1<Method, Boolean> _function_2 = new Function1<Method, Boolean>() {
            @Override
            public Boolean apply(final Method m) {
              int _parameterCount = m.getParameterCount();
              return Boolean.valueOf((_parameterCount == 2));
            }
          };
          final Method setter = IterableExtensions.<Method>findFirst(methods, _function_2);
          if (((getter != null) && (setter != null))) {
            String _findDataName = IntrospectiveMutableFieldExtractor.this.findDataName(eObject);
            final Supplier<Object> _function_3 = new Supplier<Object>() {
              @Override
              public Object get() {
                try {
                  return getter.invoke(null, eObject);
                } catch (Throwable _e) {
                  throw Exceptions.sneakyThrow(_e);
                }
              }
            };
            final Consumer<Object> _function_4 = new Consumer<Object>() {
              @Override
              public void accept(final Object t) {
                try {
                  setter.invoke(null, eObject, t);
                } catch (Throwable _e) {
                  throw Exceptions.sneakyThrow(_e);
                }
              }
            };
            final MutableField data = new MutableField(_findDataName, eObject, _function_3, _function_4);
            result.add(data);
          }
        }
      };
      ((List<Field>)Conversions.doWrapArray(fields)).forEach(_function);
    }
    return result;
  }
  
  @Override
  public List<MutableField> extractMutableField(final EObject eObject) {
    boolean _containsKey = this.eObjects.containsKey(eObject);
    boolean _not = (!_containsKey);
    if (_not) {
      final ArrayList<MutableField> datas = new ArrayList<MutableField>();
      boolean _containsKey_1 = this.aspectClasses.containsKey(eObject.eClass());
      boolean _not_1 = (!_containsKey_1);
      if (_not_1) {
        final Map<Class<?>, List<Class<?>>> classes = this.getStaticHelperClasses(eObject);
        if ((classes != null)) {
          final ArrayList<Pair<Class<?>, Class<?>>> list = new ArrayList<Pair<Class<?>, Class<?>>>();
          final BiConsumer<Class<?>, List<Class<?>>> _function = new BiConsumer<Class<?>, List<Class<?>>>() {
            @Override
            public void accept(final Class<?> i, final List<Class<?>> l) {
              final Consumer<Class<?>> _function = new Consumer<Class<?>>() {
                @Override
                public void accept(final Class<?> c) {
                  try {
                    Bundle _dslBundle = DslHelper.getDslBundle(IntrospectiveMutableFieldExtractor.this.languageName);
                    String _name = c.getName();
                    String _simpleName = i.getSimpleName();
                    String _plus = (_name + _simpleName);
                    String _plus_1 = (_plus + "AspectProperties");
                    final Class<?> properties = _dslBundle.loadClass(_plus_1);
                    final Pair<Class<?>, Class<?>> pair = new Pair<Class<?>, Class<?>>(c, properties);
                    list.add(pair);
                    datas.addAll(IntrospectiveMutableFieldExtractor.this.getMutableFieldsFromAspect(eObject, properties, c));
                  } catch (final Throwable _t) {
                    if (_t instanceof ClassNotFoundException) {
                    } else {
                      throw Exceptions.sneakyThrow(_t);
                    }
                  }
                }
              };
              l.forEach(_function);
            }
          };
          classes.forEach(_function);
          this.aspectClasses.put(eObject.eClass(), list);
        } else {
          this.aspectClasses.put(eObject.eClass(), Collections.EMPTY_LIST);
        }
      } else {
        final List<Pair<Class<?>, Class<?>>> list_1 = this.aspectClasses.get(eObject.eClass());
        final Consumer<Pair<Class<?>, Class<?>>> _function_1 = new Consumer<Pair<Class<?>, Class<?>>>() {
          @Override
          public void accept(final Pair<Class<?>, Class<?>> p) {
            datas.addAll(IntrospectiveMutableFieldExtractor.this.getMutableFieldsFromAspect(eObject, p.getValue(), p.getKey()));
          }
        };
        list_1.forEach(_function_1);
      }
      this.eObjects.put(eObject, datas);
      return datas;
    } else {
      return this.eObjects.get(eObject);
    }
  }
  
  private void getSuperInterfacesOfInterface(final Class<?> c, final HashSet<Class<?>> set) {
    final Function1<Class<?>, Boolean> _function = new Function1<Class<?>, Boolean>() {
      @Override
      public Boolean apply(final Class<?> i) {
        boolean _equals = i.equals(EObject.class);
        return Boolean.valueOf((!_equals));
      }
    };
    final Iterable<Class<?>> possibleSuperInterfaces = IterableExtensions.<Class<?>>filter(Arrays.<Class<?>>asList(c.getInterfaces()), _function);
    final Consumer<Class<?>> _function_1 = new Consumer<Class<?>>() {
      @Override
      public void accept(final Class<?> i) {
        boolean _add = set.add(i);
        if (_add) {
          IntrospectiveMutableFieldExtractor.this.getSuperInterfacesOfInterface(i, set);
        }
      }
    };
    possibleSuperInterfaces.forEach(_function_1);
  }
  
  private List<Class<?>> getSuperInterfacesOfInterface(final Class<?> c) {
    if ((c == null)) {
      return Collections.EMPTY_LIST;
    }
    final LinkedHashSet<Class<?>> interfacesFound = new LinkedHashSet<Class<?>>();
    this.getSuperInterfacesOfInterface(c, interfacesFound);
    return new ArrayList<Class<?>>(interfacesFound);
  }
  
  private List<Class<?>> getInterfacesOfEObject(final EObject o) {
    final List<Class<?>> possibleInterfaces = new ArrayList<Class<?>>();
    final List<Class<?>> interfaces = this.getAllInterfaces(o.getClass());
    final Function1<Class<?>, Boolean> _function = new Function1<Class<?>, Boolean>() {
      @Override
      public Boolean apply(final Class<?> i) {
        return Boolean.valueOf(i.getSimpleName().equals(o.eClass().getName()));
      }
    };
    final Class<?> baseInterface = IterableExtensions.<Class<?>>findFirst(interfaces, _function);
    if ((baseInterface != null)) {
      possibleInterfaces.add(baseInterface);
      possibleInterfaces.addAll(this.getSuperInterfacesOfInterface(baseInterface));
    }
    InputOutput.<List<Class<?>>>println(possibleInterfaces);
    return possibleInterfaces;
  }
  
  private List<Class<?>> getAllInterfaces(final Class<? extends EObject> cls) {
    if ((cls == null)) {
      return Collections.EMPTY_LIST;
    }
    final LinkedHashSet<Class<?>> interfacesFound = new LinkedHashSet<Class<?>>();
    this.getAllInterfaces(cls, interfacesFound);
    final ArrayList<Class<?>> res = new ArrayList<Class<?>>(interfacesFound);
    return res;
  }
  
  private void getAllInterfaces(final Class<?> cls, final HashSet<Class<?>> interfacesFound) {
    Class<?> currCls = cls;
    while ((currCls != null)) {
      {
        final Consumer<Class<?>> _function = new Consumer<Class<?>>() {
          @Override
          public void accept(final Class<?> i) {
            boolean _add = interfacesFound.add(i);
            if (_add) {
              IntrospectiveMutableFieldExtractor.this.getAllInterfaces(i, interfacesFound);
            }
          }
        };
        ((List<Class<?>>)Conversions.doWrapArray(currCls.getInterfaces())).forEach(_function);
        currCls = currCls.getSuperclass();
      }
    }
  }
  
  private Map<Class<?>, List<Class<?>>> getStaticHelperClasses(final EObject target) {
    final List<Class<?>> allPossibleInterfaces = this.getInterfacesOfEObject(target);
    final Map<Class<?>, List<Class<?>>> res = new HashMap<Class<?>, List<Class<?>>>();
    final Set<Class<?>> allAspects = K3DslHelper.getAspects(this.languageName);
    final Consumer<Class<?>> _function = new Consumer<Class<?>>() {
      @Override
      public void accept(final Class<?> i) {
        final Function1<Class<?>, Boolean> _function = new Function1<Class<?>, Boolean>() {
          @Override
          public Boolean apply(final Class<?> asp) {
            Class<?> _target = K3DslHelper.getTarget(asp);
            return Boolean.valueOf(Objects.equal(_target, i));
          }
        };
        final Iterable<Class<?>> appliedAspects = IterableExtensions.<Class<?>>filter(allAspects, _function);
        res.put(i, IterableExtensions.<Class<?>>toList(appliedAspects));
      }
    };
    allPossibleInterfaces.forEach(_function);
    return res;
  }
}
