/**
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Mark Czotter - initial API and implementation
 */
package org.eclipse.incquery.patternlanguage.emf.jvmmodel;

import com.google.inject.Inject;
import org.eclipse.emf.common.util.EList;
import org.eclipse.incquery.patternlanguage.emf.util.EMFPatternLanguageJvmModelInferrerUtil;
import org.eclipse.incquery.patternlanguage.patternLanguage.Pattern;
import org.eclipse.incquery.patternlanguage.patternLanguage.PatternModel;
import org.eclipse.incquery.patternlanguage.patternLanguage.Variable;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.lib.Extension;

@SuppressWarnings("all")
public class JavadocInferrer {
  @Inject
  @Extension
  private EMFPatternLanguageJvmModelInferrerUtil _eMFPatternLanguageJvmModelInferrerUtil;
  
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;
  
  /**
   * Infers javadoc for Match class based on the input 'pattern'.
   */
  public CharSequence javadocMatchClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Pattern-specific match representation of the ");
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(pattern);
    _builder.append(_fullyQualifiedName, "");
    _builder.append(" pattern,");
    _builder.newLineIfNotEmpty();
    _builder.append("to be used in conjunction with {@link ");
    String _matcherClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matcherClassName(pattern);
    _builder.append(_matcherClassName, "");
    _builder.append("}.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("<p>Class fields correspond to parameters of the pattern. Fields with value null are considered unassigned.");
    _builder.newLine();
    _builder.append("Each instance is a (possibly partial) substitution of pattern parameters,");
    _builder.newLine();
    _builder.append("usable to represent a match of the pattern in the result of a query,");
    _builder.newLine();
    _builder.append("or to specify the bound (fixed) input parameters when issuing a query.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@see ");
    String _matcherClassName_1 = this._eMFPatternLanguageJvmModelInferrerUtil.matcherClassName(pattern);
    _builder.append(_matcherClassName_1, "");
    _builder.newLineIfNotEmpty();
    _builder.append("@see ");
    String _processorClassName = this._eMFPatternLanguageJvmModelInferrerUtil.processorClassName(pattern);
    _builder.append(_processorClassName, "");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocMatcherClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Generated pattern matcher API of the ");
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(pattern);
    _builder.append(_fullyQualifiedName, "");
    _builder.append(" pattern,");
    _builder.newLineIfNotEmpty();
    _builder.append("providing pattern-specific query methods.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("<p>Use the pattern matcher on a given model via {@link #on(IncQueryEngine)},");
    _builder.newLine();
    _builder.append("e.g. in conjunction with {@link IncQueryEngine#on(Notifier)}.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("<p>Matches of the pattern will be represented as {@link ");
    String _matchClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matchClassName(pattern);
    _builder.append(_matchClassName, "");
    _builder.append("}.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("<p>Original source:");
    _builder.newLine();
    _builder.append("<code><pre>");
    _builder.newLine();
    String _serializeToJavadoc = this._eMFPatternLanguageJvmModelInferrerUtil.serializeToJavadoc(pattern);
    _builder.append(_serializeToJavadoc, "");
    _builder.newLineIfNotEmpty();
    _builder.append("</pre></code>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@see ");
    String _matchClassName_1 = this._eMFPatternLanguageJvmModelInferrerUtil.matchClassName(pattern);
    _builder.append(_matchClassName_1, "");
    _builder.newLineIfNotEmpty();
    _builder.append("@see ");
    String _processorClassName = this._eMFPatternLanguageJvmModelInferrerUtil.processorClassName(pattern);
    _builder.append(_processorClassName, "");
    _builder.newLineIfNotEmpty();
    _builder.append("@see ");
    String _querySpecificationClassName = this._eMFPatternLanguageJvmModelInferrerUtil.querySpecificationClassName(pattern);
    _builder.append(_querySpecificationClassName, "");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocQuerySpecificationClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A pattern-specific query specification that can instantiate ");
    String _matcherClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matcherClassName(pattern);
    _builder.append(_matcherClassName, "");
    _builder.append(" in a type-safe way.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("@see ");
    String _matcherClassName_1 = this._eMFPatternLanguageJvmModelInferrerUtil.matcherClassName(pattern);
    _builder.append(_matcherClassName_1, "");
    _builder.newLineIfNotEmpty();
    _builder.append("@see ");
    String _matchClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matchClassName(pattern);
    _builder.append(_matchClassName, "");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocProcessorClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A match processor tailored for the ");
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(pattern);
    _builder.append(_fullyQualifiedName, "");
    _builder.append(" pattern.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("Clients should derive an (anonymous) class that implements the abstract process().");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocEvaluatorClass(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A xbase xexpression evaluator tailored for the ");
    QualifiedName _fullyQualifiedName = this._iQualifiedNameProvider.getFullyQualifiedName(pattern);
    _builder.append(_fullyQualifiedName, "");
    _builder.append(" pattern.");
    return _builder;
  }
  
  public CharSequence javadocEvaluatorClassWrapperMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A wrapper method for calling the generated java method with the correct attributes.");
    return _builder;
  }
  
  public CharSequence javadocMatcherConstructorNotifier(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Initializes the pattern matcher over a given EMF model root (recommended: Resource or ResourceSet).");
    _builder.newLine();
    _builder.append("If a pattern matcher is already constructed with the same root, only a light-weight reference is returned.");
    _builder.newLine();
    _builder.append("The scope of pattern matching will be the given EMF model root and below (see FAQ for more precise definition).");
    _builder.newLine();
    _builder.append("The match set will be incrementally refreshed upon updates from this scope.");
    _builder.newLine();
    _builder.append("<p>The matcher will be created within the managed {@link IncQueryEngine} belonging to the EMF model root, so");
    _builder.newLine();
    _builder.append("multiple matchers will reuse the same engine and benefit from increased performance and reduced memory footprint.");
    _builder.newLine();
    _builder.append("@param emfRoot the root of the EMF containment hierarchy where the pattern matcher will operate. Recommended: Resource or ResourceSet.");
    _builder.newLine();
    _builder.append("@throws IncQueryException if an error occurs during pattern matcher creation");
    _builder.newLine();
    _builder.append("@deprecated use {@link #on(IncQueryEngine)} instead, e.g. in conjunction with {@link IncQueryEngine#on(Notifier)}");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocMatcherConstructorEngine(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Initializes the pattern matcher within an existing EMF-IncQuery engine.");
    _builder.newLine();
    _builder.append("If the pattern matcher is already constructed in the engine, only a light-weight reference is returned.");
    _builder.newLine();
    _builder.append("The match set will be incrementally refreshed upon updates.");
    _builder.newLine();
    _builder.append("@param engine the existing EMF-IncQuery engine in which this matcher will be created.");
    _builder.newLine();
    _builder.append("@throws IncQueryException if an error occurs during pattern matcher creation");
    _builder.newLine();
    _builder.append("@deprecated use {@link #on(IncQueryEngine)} instead");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocMatcherStaticOnEngine(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Initializes the pattern matcher within an existing EMF-IncQuery engine.");
    _builder.newLine();
    _builder.append("If the pattern matcher is already constructed in the engine, only a light-weight reference is returned.");
    _builder.newLine();
    _builder.append("The match set will be incrementally refreshed upon updates.");
    _builder.newLine();
    _builder.append("@param engine the existing EMF-IncQuery engine in which this matcher will be created.");
    _builder.newLine();
    _builder.append("@throws IncQueryException if an error occurs during pattern matcher creation");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGetAllMatchesMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns the set of all matches of the pattern that conform to the given fixed values of some parameters.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return matches represented as a ");
    String _matchClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matchClassName(pattern);
    _builder.append(_matchClassName, "");
    _builder.append(" object.");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocGetOneArbitraryMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns an arbitrarily chosen match of the pattern that conforms to the given fixed values of some parameters.");
    _builder.newLine();
    _builder.append("Neither determinism nor randomness of selection is guaranteed.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return a match represented as a ");
    String _matchClassName = this._eMFPatternLanguageJvmModelInferrerUtil.matchClassName(pattern);
    _builder.append(_matchClassName, "");
    _builder.append(" object, or null if no match is found.");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence javadocHasMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Indicates whether the given combination of specified pattern parameters constitute a valid pattern match,");
    _builder.newLine();
    _builder.append("under any possible substitution of the unspecified parameters (if any).");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return true if the input is a valid (partial) match of the pattern.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocHasMatchMethodNoParameter(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Indicates whether the (parameterless) pattern matches or not.");
    _builder.newLine();
    _builder.append("@return true if the pattern has a valid match.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocCountMatchesMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns the number of all matches of the pattern that conform to the given fixed values of some parameters.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return the number of pattern matches found.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocForEachMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Executes the given processor on each match of the pattern that conforms to the given fixed values of some parameters.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@param processor the action that will process each pattern match.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocForOneArbitraryMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Executes the given processor on an arbitrarily chosen match of the pattern that conforms to the given fixed values of some parameters.");
    _builder.newLine();
    _builder.append("Neither determinism nor randomness of selection is guaranteed.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@param processor the action that will process the selected match.");
    _builder.newLine();
    _builder.append("@return true if the pattern has at least one match with the given parameter values, false if the processor was not invoked");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocProcessMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Defines the action that is to be executed on each match.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(" in the currently processed match");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
  
  public CharSequence javadocNewFilteredDeltaMonitorMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Registers a new filtered delta monitor on this pattern matcher.");
    _builder.newLine();
    _builder.append("The DeltaMonitor can be used to track changes (delta) in the set of filtered pattern matches from now on, considering those matches only that conform to the given fixed values of some parameters.");
    _builder.newLine();
    _builder.append("It can also be reset to track changes from a later point in time,");
    _builder.newLine();
    _builder.append("and changes can even be acknowledged on an individual basis.");
    _builder.newLine();
    _builder.append("See {@link DeltaMonitor} for details.");
    _builder.newLine();
    _builder.append("@param fillAtStart if true, all current matches are reported as new match events; if false, the delta monitor starts empty.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return the delta monitor.");
    _builder.newLine();
    _builder.append("@deprecated use the IncQuery Databinding API (IncQueryObservables) instead.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocNewMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns a new (partial) match.");
    _builder.newLine();
    _builder.append("This can be used e.g. to call the matcher with a partial match.");
    _builder.newLine();
    _builder.append("<p>The returned match will be immutable. Use {@link #newEmptyMatch()} to obtain a mutable match object.");
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return the (partial) match object.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocNewMutableMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns a mutable (partial) match.");
    _builder.newLine();
    _builder.append("Fields of the mutable match can be filled to create a partial match, usable as matcher input.");
    _builder.newLine();
    _builder.newLine();
    {
      EList<Variable> _parameters = pattern.getParameters();
      for(final Variable p : _parameters) {
        _builder.append("@param ");
        String _parameterName = this._eMFPatternLanguageJvmModelInferrerUtil.parameterName(p);
        _builder.append(_parameterName, "");
        _builder.append(" the fixed value of pattern parameter ");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append(", or null if not bound.");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("@return the new, mutable (partial) match object.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocNewEmptyMatchMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Returns an empty, mutable match.");
    _builder.newLine();
    _builder.append("Fields of the mutable match can be filled to create a partial match, usable as matcher input.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@return the empty match.");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGetAllValuesOfMethod(final Variable parameter) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Retrieve the set of values that occur in matches for ");
    String _name = parameter.getName();
    _builder.append(_name, "");
    _builder.append(".");
    _builder.newLineIfNotEmpty();
    _builder.append("@return the Set of all values, null if no parameter with the given name exists, empty set if there are no matches");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocQuerySpecificationMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@return the singleton instance of the query specification of this pattern");
    _builder.newLine();
    _builder.append("@throws IncQueryException if the pattern definition could not be loaded");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocQuerySpecificationInstanceMethod(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@return the singleton instance of the query specification");
    _builder.newLine();
    _builder.append("@throws IncQueryException if the pattern definition could not be loaded");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGroupClass(final PatternModel model) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("A pattern group formed of all patterns defined in ");
    String _modelFileName = this._eMFPatternLanguageJvmModelInferrerUtil.modelFileName(model);
    _builder.append(_modelFileName, "");
    _builder.append(".eiq.");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("<p>Use the static instance as any {@link org.eclipse.incquery.runtime.api.IPatternGroup}, to conveniently prepare");
    _builder.newLine();
    _builder.append("an EMF-IncQuery engine for matching all patterns originally defined in file ");
    String _modelFileName_1 = this._eMFPatternLanguageJvmModelInferrerUtil.modelFileName(model);
    _builder.append(_modelFileName_1, "");
    _builder.append(".eiq,");
    _builder.newLineIfNotEmpty();
    _builder.append("in order to achieve better performance than one-by-one on-demand matcher initialization.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("<p> From package ");
    String _packageName = model.getPackageName();
    _builder.append(_packageName, "");
    _builder.append(", the group contains the definition of the following patterns: <ul>");
    _builder.newLineIfNotEmpty();
    {
      EList<Pattern> _patterns = model.getPatterns();
      for(final Pattern p : _patterns) {
        _builder.append("<li>");
        String _name = p.getName();
        _builder.append(_name, "");
        _builder.append("</li>");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("</ul>");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@see IPatternGroup");
    _builder.newLine();
    return _builder;
  }
  
  public CharSequence javadocGroupClassInstanceMethod(final PatternModel model) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Access the pattern group.");
    _builder.newLine();
    _builder.newLine();
    _builder.append("@return the singleton instance of the group");
    _builder.newLine();
    _builder.append("@throws IncQueryException if there was an error loading the generated code of pattern specifications");
    _builder.newLine();
    return _builder;
  }
}
