/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.commons.eventlistprofiler;

import java.io.OutputStream;
import java.io.PrintWriter;
import java.lang.ref.WeakReference;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * This class is Thread safe
 */
public final class EventListenerProfiler {

  private static EventListenerProfiler instance = new EventListenerProfiler();

  public static EventListenerProfiler getInstance() {
    return instance;
  }

  private boolean m_enabled = false;
  private Object m_sourcesLock = new Object();
  private List<WeakReference<IEventListenerSource>> m_sources = new ArrayList<WeakReference<IEventListenerSource>>();

  private EventListenerProfiler() {
    m_sourcesLock = new Object();
    m_sources = new ArrayList<WeakReference<IEventListenerSource>>();
  }

  public boolean isEnabled() {
    return m_enabled;
  }

  public void setEnabled(boolean enabled) {
    m_enabled = enabled;
  }

  /**
   * Add a weak reference to a source provider NOTE: the passed argument MUST be
   * referenced by the source type, otherwise it is garbage collected
   * immediately after adding
   */
  public void registerSourceAsWeakReference(IEventListenerSource source) {
    if (!m_enabled) return;
    synchronized (m_sourcesLock) {
      m_sources.add(new WeakReference<IEventListenerSource>(source));
    }
  }

  public void dump() {
    dump(System.out);
  }

  public void dump(OutputStream o) {
    /**
     * this call to gc is intended
     */
    System.gc();
    PrintWriter out = new PrintWriter(o, true);
    if (!m_enabled) return;
    try {
      EventListenerSnapshot snapshot = new EventListenerSnapshot();
      synchronized (m_sourcesLock) {
        manageNoLock();
        NumberFormat fmt = NumberFormat.getIntegerInstance();
        out.println("Max memory:   " + fmt.format(Runtime.getRuntime().maxMemory()));
        out.println("Total memory: " + fmt.format(Runtime.getRuntime().totalMemory()));
        out.println("Free memory:  " + fmt.format(Runtime.getRuntime().freeMemory()));
        out.println("(Used memory):" + fmt.format(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()));
        out.println();
        for (WeakReference<IEventListenerSource> ref : m_sources) {
          IEventListenerSource p = ref.get();
          if (p != null) {
            p.dumpListenerList(snapshot);
          }
        }
      }
      snapshot.dump(out);
    }
    catch (Throwable t) {
      t.printStackTrace();
    }
    finally {
      if (o != System.out) {
        out.close();
      }
    }
  }

  private void manageNoLock() {
    for (Iterator<WeakReference<IEventListenerSource>> it = m_sources.iterator(); it.hasNext();) {
      WeakReference<IEventListenerSource> ref = it.next();
      if (ref.get() == null) {
        it.remove();
      }
    }
  }

}
