package org.eclipse.smarthome.model.rule.jvmmodel;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.smarthome.core.items.Item;
import org.eclipse.smarthome.core.items.ItemRegistry;
import org.eclipse.smarthome.core.types.Command;
import org.eclipse.smarthome.core.types.State;
import org.eclipse.smarthome.core.types.Type;
import org.eclipse.smarthome.model.rule.rules.ChangedEventTrigger;
import org.eclipse.smarthome.model.rule.rules.CommandEventTrigger;
import org.eclipse.smarthome.model.rule.rules.EventTrigger;
import org.eclipse.smarthome.model.rule.rules.Rule;
import org.eclipse.smarthome.model.rule.rules.RuleModel;
import org.eclipse.smarthome.model.rule.rules.VariableDeclaration;
import org.eclipse.smarthome.model.script.engine.IItemRegistryProvider;
import org.eclipse.smarthome.model.script.jvmmodel.ScriptJvmModelInferrer;
import org.eclipse.smarthome.model.script.scoping.StateAndCommandProvider;
import org.eclipse.smarthome.model.script.script.Script;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.xbase.XBlockExpression;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>Infers a JVM model from the source model.</p>
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code
 * which is generated from the source model. Other models link against the JVM model rather than the source model.</p>
 * 
 * @author Oliver Libutzki - Xtext 2.5.0 migration
 */
@SuppressWarnings("all")
public class RulesJvmModelInferrer extends ScriptJvmModelInferrer {
  private final static Logger logger = LoggerFactory.getLogger(RulesJvmModelInferrer.class);
  
  /**
   * Variable name for the previous state of an item in a "changed state triggered" rule
   */
  public final static String VAR_PREVIOUS_STATE = "previousState";
  
  /**
   * Variable name for the received command in a "command triggered" rule
   */
  public final static String VAR_RECEIVED_COMMAND = "receivedCommand";
  
  /**
   * conveninence API to build and initialize JvmTypes and their members.
   */
  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;
  
  @Inject
  @Extension
  private IQualifiedNameProvider _iQualifiedNameProvider;
  
  @Inject
  private IItemRegistryProvider itemRegistryProvider;
  
  @Inject
  private StateAndCommandProvider stateAndCommandProvider;
  
  /**
   * Is called for each instance of the first argument's type contained in a resource.
   * 
   * @param element - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if iPrelinkingPhase is <code>true</code>
   */
  protected void _infer(final RuleModel ruleModel, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    Resource _eResource = ruleModel.eResource();
    URI _uRI = _eResource.getURI();
    String _lastSegment = _uRI.lastSegment();
    String[] _split = _lastSegment.split("\\.");
    String _head = IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(_split)));
    String _firstUpper = StringExtensions.toFirstUpper(_head);
    final String className = (_firstUpper + "Rules");
    JvmGenericType _class = this._jvmTypesBuilder.toClass(ruleModel, className);
    IJvmDeclaredTypeAcceptor.IPostIndexingInitializing<JvmGenericType> _accept = acceptor.<JvmGenericType>accept(_class);
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      public void apply(final JvmGenericType it) {
        EList<JvmMember> _members = it.getMembers();
        EList<VariableDeclaration> _variables = ruleModel.getVariables();
        final Function1<VariableDeclaration, JvmField> _function = new Function1<VariableDeclaration, JvmField>() {
          public JvmField apply(final VariableDeclaration it) {
            String _name = it.getName();
            JvmTypeReference _type = it.getType();
            JvmTypeReference _cloneWithProxies = null;
            if (_type!=null) {
              _cloneWithProxies=RulesJvmModelInferrer.this._jvmTypesBuilder.cloneWithProxies(_type);
            }
            JvmField _field = RulesJvmModelInferrer.this._jvmTypesBuilder.toField(it, _name, _cloneWithProxies);
            final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
              public void apply(final JvmField field) {
                field.setStatic(true);
                boolean _isWriteable = it.isWriteable();
                boolean _not = (!_isWriteable);
                field.setFinal(_not);
                XExpression _right = it.getRight();
                RulesJvmModelInferrer.this._jvmTypesBuilder.setInitializer(field, _right);
              }
            };
            return ObjectExtensions.<JvmField>operator_doubleArrow(_field, _function);
          }
        };
        List<JvmMember> _map = ListExtensions.<VariableDeclaration, JvmMember>map(_variables, _function);
        RulesJvmModelInferrer.this._jvmTypesBuilder.<JvmMember>operator_add(_members, _map);
        final Set<String> fieldNames = CollectionLiterals.<String>newHashSet();
        final Iterable<Type> types = RulesJvmModelInferrer.this.stateAndCommandProvider.getAllTypes();
        final Procedure1<Type> _function_1 = new Procedure1<Type>() {
          public void apply(final Type type) {
            final String name = type.toString();
            boolean _add = fieldNames.add(name);
            if (_add) {
              EList<JvmMember> _members = it.getMembers();
              Class<? extends Type> _class = type.getClass();
              JvmTypeReference _newTypeRef = RulesJvmModelInferrer.this._jvmTypesBuilder.newTypeRef(ruleModel, _class);
              final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
                public void apply(final JvmField it) {
                  it.setStatic(true);
                }
              };
              JvmField _field = RulesJvmModelInferrer.this._jvmTypesBuilder.toField(ruleModel, name, _newTypeRef, _function);
              RulesJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
            } else {
              Class<? extends Type> _class_1 = type.getClass();
              String _name = _class_1.getName();
              RulesJvmModelInferrer.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", name, _name);
            }
          }
        };
        IterableExtensions.<Type>forEach(types, _function_1);
        final ItemRegistry itemRegistry = RulesJvmModelInferrer.this.itemRegistryProvider.get();
        Collection<Item> _items = null;
        if (itemRegistry!=null) {
          _items=itemRegistry.getItems();
        }
        if (_items!=null) {
          final Procedure1<Item> _function_2 = new Procedure1<Item>() {
            public void apply(final Item item) {
              final String name = item.getName();
              boolean _add = fieldNames.add(name);
              if (_add) {
                EList<JvmMember> _members = it.getMembers();
                String _name = item.getName();
                Class<? extends Item> _class = item.getClass();
                JvmTypeReference _newTypeRef = RulesJvmModelInferrer.this._jvmTypesBuilder.newTypeRef(ruleModel, _class);
                final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
                  public void apply(final JvmField it) {
                    it.setStatic(true);
                  }
                };
                JvmField _field = RulesJvmModelInferrer.this._jvmTypesBuilder.toField(ruleModel, _name, _newTypeRef, _function);
                RulesJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
              } else {
                String _name_1 = item.getName();
                Class<? extends Item> _class_1 = item.getClass();
                String _name_2 = _class_1.getName();
                RulesJvmModelInferrer.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", _name_1, _name_2);
              }
            }
          };
          IterableExtensions.<Item>forEach(_items, _function_2);
        }
        EList<JvmMember> _members_1 = it.getMembers();
        EList<Rule> _rules = ruleModel.getRules();
        final Function1<Rule, JvmOperation> _function_3 = new Function1<Rule, JvmOperation>() {
          public JvmOperation apply(final Rule rule) {
            String _name = rule.getName();
            String _plus = ("_" + _name);
            JvmTypeReference _newTypeRef = RulesJvmModelInferrer.this._jvmTypesBuilder.newTypeRef(ruleModel, Void.TYPE);
            final Procedure1<JvmOperation> _function = new Procedure1<JvmOperation>() {
              public void apply(final JvmOperation it) {
                it.setStatic(true);
                boolean _containsCommandTrigger = RulesJvmModelInferrer.this.containsCommandTrigger(rule);
                if (_containsCommandTrigger) {
                  final JvmTypeReference commandTypeRef = RulesJvmModelInferrer.this._jvmTypesBuilder.newTypeRef(ruleModel, Command.class);
                  EList<JvmFormalParameter> _parameters = it.getParameters();
                  JvmFormalParameter _parameter = RulesJvmModelInferrer.this._jvmTypesBuilder.toParameter(rule, RulesJvmModelInferrer.VAR_RECEIVED_COMMAND, commandTypeRef);
                  RulesJvmModelInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
                }
                boolean _containsStateChangeTrigger = RulesJvmModelInferrer.this.containsStateChangeTrigger(rule);
                if (_containsStateChangeTrigger) {
                  final JvmTypeReference stateTypeRef = RulesJvmModelInferrer.this._jvmTypesBuilder.newTypeRef(ruleModel, State.class);
                  EList<JvmFormalParameter> _parameters_1 = it.getParameters();
                  JvmFormalParameter _parameter_1 = RulesJvmModelInferrer.this._jvmTypesBuilder.toParameter(rule, RulesJvmModelInferrer.VAR_PREVIOUS_STATE, stateTypeRef);
                  RulesJvmModelInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters_1, _parameter_1);
                }
                XBlockExpression _script = rule.getScript();
                RulesJvmModelInferrer.this._jvmTypesBuilder.setBody(it, _script);
              }
            };
            return RulesJvmModelInferrer.this._jvmTypesBuilder.toMethod(rule, _plus, _newTypeRef, _function);
          }
        };
        List<JvmMember> _map_1 = ListExtensions.<Rule, JvmMember>map(_rules, _function_3);
        RulesJvmModelInferrer.this._jvmTypesBuilder.<JvmMember>operator_add(_members_1, _map_1);
      }
    };
    _accept.initializeLater(_function);
  }
  
  private boolean containsCommandTrigger(final Rule rule) {
    EList<EventTrigger> _eventtrigger = rule.getEventtrigger();
    for (final EventTrigger trigger : _eventtrigger) {
      if ((trigger instanceof CommandEventTrigger)) {
        return true;
      }
    }
    return false;
  }
  
  private boolean containsStateChangeTrigger(final Rule rule) {
    EList<EventTrigger> _eventtrigger = rule.getEventtrigger();
    for (final EventTrigger trigger : _eventtrigger) {
      if ((trigger instanceof ChangedEventTrigger)) {
        return true;
      }
    }
    return false;
  }
  
  public void infer(final EObject ruleModel, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    if (ruleModel instanceof Script) {
      _infer((Script)ruleModel, acceptor, isPreIndexingPhase);
      return;
    } else if (ruleModel instanceof RuleModel) {
      _infer((RuleModel)ruleModel, acceptor, isPreIndexingPhase);
      return;
    } else if (ruleModel != null) {
      _infer(ruleModel, acceptor, isPreIndexingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(ruleModel, acceptor, isPreIndexingPhase).toString());
    }
  }
}
