"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = void 0;
const volar_service_html_1 = require("volar-service-html");
const language_service_1 = require("@volar/language-service");
const pug = require("./languageService");
function create() {
    return (context, modules) => {
        const htmlService = (0, volar_service_html_1.create)()(context, modules);
        if (!context) {
            return htmlService;
        }
        const pugDocuments = new WeakMap();
        const pugLs = pug.getLanguageService(htmlService.provide['html/languageService']());
        return {
            ...htmlService,
            provide: {
                'pug/pugDocument': getPugDocument,
                'pug/languageService': () => pugLs,
                'pug/updateCustomData': htmlService.provide['html/updateCustomData'],
            },
            provideCompletionItems(document, position, _) {
                return worker(document, (pugDocument) => {
                    return pugLs.doComplete(pugDocument, position, htmlService.provide['html/documentContext']());
                });
            },
            provideDiagnostics(document) {
                return worker(document, (pugDocument) => {
                    if (pugDocument.error) {
                        return [{
                                source: 'pug',
                                severity: 1,
                                code: pugDocument.error.code,
                                message: pugDocument.error.msg,
                                range: {
                                    start: { line: pugDocument.error.line, character: pugDocument.error.column },
                                    end: { line: pugDocument.error.line, character: pugDocument.error.column },
                                },
                            }];
                    }
                    return [];
                });
            },
            provideHover(document, position) {
                return worker(document, async (pugDocument) => {
                    const hoverSettings = await context.env.getConfiguration?.('html.hover');
                    return pugLs.doHover(pugDocument, position, hoverSettings);
                });
            },
            provideDocumentHighlights(document, position) {
                return worker(document, (pugDocument) => {
                    return pugLs.findDocumentHighlights(pugDocument, position);
                });
            },
            provideDocumentLinks(document) {
                return worker(document, (pugDocument) => {
                    return pugLs.findDocumentLinks(pugDocument, htmlService.provide['html/documentContext']());
                });
            },
            provideDocumentSymbols(document, token) {
                return worker(document, async (pugDoc) => {
                    const htmlResult = await htmlService.provideDocumentSymbols?.(pugDoc.map.virtualFileDocument, token) ?? [];
                    const pugResult = htmlResult.map(htmlSymbol => language_service_1.transformer.asDocumentSymbol(htmlSymbol, range => pugDoc.map.toSourceRange(range))).filter((symbol) => symbol !== undefined);
                    return pugResult;
                });
            },
            provideFoldingRanges(document) {
                return worker(document, (pugDocument) => {
                    return pugLs.getFoldingRanges(pugDocument);
                });
            },
            provideSelectionRanges(document, positions) {
                return worker(document, (pugDocument) => {
                    return pugLs.getSelectionRanges(pugDocument, positions);
                });
            },
            async provideAutoInsertionEdit(document, position, insertContext) {
                return worker(document, async (pugDocument) => {
                    const lastCharacter = insertContext.lastChange.text[insertContext.lastChange.text.length - 1];
                    if (insertContext.lastChange.rangeLength === 0 && lastCharacter === '=') {
                        const enabled = (await context.env.getConfiguration?.('html.autoCreateQuotes')) ?? true;
                        if (enabled) {
                            const text = pugLs.doQuoteComplete(pugDocument, position, await context.env.getConfiguration?.('html.completion'));
                            if (text) {
                                return text;
                            }
                        }
                    }
                });
            },
        };
        function worker(document, callback) {
            const pugDocument = getPugDocument(document);
            if (!pugDocument)
                return;
            return callback(pugDocument);
        }
        function getPugDocument(document) {
            if (document.languageId !== 'jade')
                return;
            const cache = pugDocuments.get(document);
            if (cache) {
                const [cacheVersion, cacheDoc] = cache;
                if (cacheVersion === document.version) {
                    return cacheDoc;
                }
            }
            const doc = pugLs.parsePugDocument(document.getText());
            pugDocuments.set(document, [document.version, doc]);
            return doc;
        }
    };
}
exports.create = create;
exports.default = create;
//# sourceMappingURL=index.js.map