/***************************************************************************
*   Copyright (C) 2010 by Peter Hatina                                    *
*   email: phatina (at) gmail.com                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU Lesser General Public License           *
*   version 2.1 as published by the Free Software Foundation              *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU Lesser General Public License for more details.                   *
*   http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.               *
***************************************************************************/

#include "powerscheme.h"
#include <iostream>
#include <QMessageBox>

Scheme::Scheme()
{
}

Scheme::Scheme(GUID g, const QString &name):
    m_guid(g),
    m_name(name)
{
}

Scheme::~Scheme()
{
}

void Scheme::setGUID(GUID g)
{
    m_guid = g;
}

void Scheme::setName(const QString &name)
{
    m_name = name;
}

GUID Scheme::guid() const
{
    return m_guid;
}

QString Scheme::name() const
{
    return m_name;
}

//------------------------------------------------------------------------------

#if (defined (Q_OS_WIN32) && (WINVER >= 0x0600))
const GUID GUID_PROCESSOR_SETTINGS_SUBGROUP = {
    0x54533251,
    0x82be,
    0x4824,
    {0x96, 0xc1, 0x47, 0xb60, 0xb74, 0x0d, 0x00}};

const QString LIB_POWRPROF = "C:\\Windows\\System32\\powrprof.dll";

PowerScheme::PowerScheme():
    m_power_set_active_scheme(NULL),
    m_power_get_active_scheme(NULL)
{
    m_lib_powrprof.setFileName(LIB_POWRPROF);
    if (!m_lib_powrprof.load())
        return;

    // load necessary functions
    m_lib_powrprof.resolve("PowerReadFriendlyName");
    m_power_set_active_scheme = (power_set_active_scheme_t)
            m_lib_powrprof.resolve("PowerSetActiveScheme");
    m_power_get_active_scheme = (power_get_active_scheme_t)
            m_lib_powrprof.resolve("PowerGetActiveScheme");
}

PowerScheme::~PowerScheme()
{
    if (m_lib_powrprof.isLoaded())
        m_lib_powrprof.unload();
}

bool PowerScheme::powrProfLoaded() const
{
    return m_lib_powrprof.isLoaded();
}

void PowerScheme::setMinPower()
{
    if (!m_power_set_active_scheme)
        return;

    m_power_set_active_scheme(NULL, &GUID_MAX_POWER_SAVINGS);
}

void PowerScheme::setMaxPower()
{
    if (!m_power_set_active_scheme)
        return;

    m_power_set_active_scheme(NULL, &GUID_MIN_POWER_SAVINGS);
}

void PowerScheme::setBalanced()
{
    if (!m_power_set_active_scheme)
        return;

    m_power_set_active_scheme(NULL, &GUID_TYPICAL_POWER_SAVINGS);
}

bool PowerScheme::isMinPower() const
{
    if (!m_power_get_active_scheme)
        return false;

    GUID act_scheme;
    GUID *p_act_scheme = &act_scheme;

    if (m_power_get_active_scheme(NULL, &p_act_scheme) == ERROR_SUCCESS)
    {
        return *p_act_scheme == GUID_MAX_POWER_SAVINGS;
    }

    return false;
}

bool PowerScheme::isMaxPower() const
{
    if (!m_power_get_active_scheme)
        return false;

    GUID act_scheme;
    GUID *p_act_scheme = &act_scheme;

    if (m_power_get_active_scheme(NULL, &p_act_scheme) == ERROR_SUCCESS)
    {
        return *p_act_scheme == GUID_MIN_POWER_SAVINGS;
    }

    return false;
}

bool PowerScheme::isBalanced() const
{
    if (!m_power_get_active_scheme)
        return false;

    GUID act_scheme;
    GUID *p_act_scheme = &act_scheme;

    if (m_power_get_active_scheme(NULL, &p_act_scheme) == ERROR_SUCCESS)
    {
        return *p_act_scheme == GUID_TYPICAL_POWER_SAVINGS;
    }

    return false;
}
#endif
