// Copyright 2023 The Bazel Authors. All rights reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package python

import (
	"github.com/bazelbuild/bazel-gazelle/rule"
)

const (
	pyBinaryKind  = "py_binary"
	pyLibraryKind = "py_library"
	pyTestKind    = "py_test"
)

// Kinds returns a map that maps rule names (kinds) and information on how to
// match and merge attributes that may be found in rules of those kinds.
func (*Python) Kinds() map[string]rule.KindInfo {
	return pyKinds
}

var pyKinds = map[string]rule.KindInfo{
	pyBinaryKind: {
		MatchAny: true,
		NonEmptyAttrs: map[string]bool{
			"deps":    true,
			"main":    true,
			"srcs":    true,
			"imports": true,
		},
		SubstituteAttrs: map[string]bool{},
		MergeableAttrs: map[string]bool{
			"srcs": true,
		},
		ResolveAttrs: map[string]bool{
			"deps": true,
		},
	},
	pyLibraryKind: {
		MatchAny:   false,
		MatchAttrs: []string{"srcs"},
		NonEmptyAttrs: map[string]bool{
			"deps":    true,
			"srcs":    true,
			"imports": true,
		},
		SubstituteAttrs: map[string]bool{},
		MergeableAttrs: map[string]bool{
			"srcs": true,
		},
		ResolveAttrs: map[string]bool{
			"deps": true,
		},
	},
	pyTestKind: {
		MatchAny: false,
		NonEmptyAttrs: map[string]bool{
			"deps":    true,
			"main":    true,
			"srcs":    true,
			"imports": true,
		},
		SubstituteAttrs: map[string]bool{},
		MergeableAttrs: map[string]bool{
			"srcs": true,
		},
		ResolveAttrs: map[string]bool{
			"deps": true,
		},
	},
}

// Loads returns .bzl files and symbols they define. Every rule generated by
// GenerateRules, now or in the past, should be loadable from one of these
// files.
func (py *Python) Loads() []rule.LoadInfo {
	return pyLoads
}

var pyLoads = []rule.LoadInfo{
	{
		Name: "@rules_python//python:defs.bzl",
		Symbols: []string{
			pyBinaryKind,
			pyLibraryKind,
			pyTestKind,
		},
	},
}
