/* ps1.c -- higher level functions */

/*  Copyright (C) 2001 Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <math.h>
#include "dap_make.h"
#include "externs.h"
#include "ps.h"

#define LEFT 144.0
#define BOTTOM 144.0
#define PORTWIDTH 324.0
#define PORTHEIGHT 324.0
#define LANDWIDTH 324.0
#define LANDHEIGHT 324.0

extern dataobs dap_obs[];
extern FILE *dap_err;

extern char *pict_newstr(char *str);

static double arint(double x)
{
double i;

if (fabs(i = rint(x)) == 0.0)
	return 0.0;
else
	return i;
}

void pict_maketick(tick *t, double num, char *label, double len)
{
t->tick_num = num;
t->tick_lab = pict_newstr(label);
t->tick_len = len;
}

static void yaxis(pict *p, tick ytick[], int nyticks, double xpos, double ypos,
			double side, int marks)
{
int ny;
char tpos[3];
char npos[3];
double lablen;
double txtang;
int lab1len;
int labslen;

if (side > 0.0)
	{
	strcpy(tpos, "cb");
	strcpy(npos, "rm");
	}
else
	{
	strcpy(tpos, "ct");
	strcpy(npos, "lm");
	}
for (labslen = 0, ny = 0; ny < nyticks - 1; ny++)
	{
	lab1len = strlen(ytick[ny].tick_lab);
	if (lab1len > labslen)
		labslen = lab1len;
	}
ny = nyticks - 1;
lablen = (1.5 * side - 0.5) * ytick[ny].tick_len;
txtang = 0.0;
if (ny >= 1 && ytick[0].tick_num <= ytick[ny].tick_num &&
		ytick[ny].tick_num <= ytick[ny - 1].tick_num)
	{
	lablen += side * 3.5 * ((double) labslen) * ytick[ny].tick_len;
	txtang = 90.0;
	}
if (side > 0.0 || marks)
	pict_text(p, ytick[ny].tick_lab, ypos - lablen, ytick[ny].tick_num, txtang, tpos);
while (--ny >= 0)
	{
	pict_point(p, ypos, ytick[ny].tick_num);
	pict_point(p, ypos + side * ytick[ny].tick_len, ytick[ny].tick_num);
	pict_point(p, ypos, ytick[ny].tick_num);
	if (side > 0.0 || marks)
		pict_text(p, ytick[ny].tick_lab,
			ypos - (1.5 * side - 0.5) * ytick[ny].tick_len,
			ytick[ny].tick_num, 0.0, npos);
	}
pict_point(p, ypos, xpos);
}

static void xaxis(pict *p, tick xtick[], int nxticks, double xpos, double ypos,
			double side, int marks)
{
int nx;
char tpos[3];
double lablen;

strcpy(tpos, "lt");
nx = nxticks - 1;
lablen = side * xtick[nx].tick_len;
if (nx >= 1 && xtick[0].tick_num <= xtick[nx].tick_num &&
                xtick[nx].tick_num <= xtick[nx - 1].tick_num)
	{
	tpos[0] = 'c';
	lablen += 10.0 * xtick[nx].tick_len;
	}
if (side > 0.0)
	pict_text(p, xtick[nx].tick_lab, xtick[nx].tick_num, xpos - lablen, 0.0, tpos);
while (--nx >= 0)
	{
	pict_point(p, xtick[nx].tick_num, xpos);
	if (side > 0.0 || marks)
		pict_point(p, xtick[nx].tick_num, xpos + side * xtick[nx].tick_len);
	pict_point(p, xtick[nx].tick_num, xpos);
	if (side > 0.0)
		pict_text(p, xtick[nx].tick_lab, xtick[nx].tick_num,
			xpos - side * xtick[nx].tick_len, 0.0, "ct");
	}
pict_point(p, ypos, xpos);
}

void pict_axes(pict *p, tick xtick[], int nxticks, tick ytick[], int nyticks,
		char style[], double bpos, double lpos, double tpos, double rpos)
{
double xpos, ypos;
int rmarks, tmarks;

xpos = 0.0;
ypos = 0.0;
rmarks = 0;
tmarks = 0;
if (style[0] == '#' || style[0] == 'B')
	tmarks = 1;
if (style[1] == '#' || style[1] == 'B')
	rmarks = 1;
if (style[0] != 'b' && style[1] != 'b' &&
		style[0] != 'B' && style[1] != 'B')
	{
	switch (style[0])
		{
	case '-':
	case '0':
		xpos = bpos;
		break;
	case '#':
	case '+':
		xpos = tpos;
		break;
	case 'n':
		xpos = 0.0;
		break;
		}
	switch (style[1])
		{
	case '-':
	case '0':
		ypos = lpos;
		break;
	case '#':
	case '+':
		ypos = rpos;
		break;
	case 'n':
		ypos = 0.0;
		break;
		}
	switch (style[0])
		{
	case '-':
	case '0':
		xaxis(p, xtick, nxticks, bpos, ypos, 1.0, tmarks);
		break;
	case '+':
	case '#':
		xaxis(p, xtick, nxticks, tpos, ypos, -1.0, tmarks);
		break;
	case 'n':
		break;
		}
	switch (style[1])
		{
	case '-':
	case '0':
		yaxis(p, ytick, nyticks, xpos, lpos, 1.0, rmarks);
		break;
	case '+':
	case '#':
		yaxis(p, ytick, nyticks, xpos, rpos, -1.0, rmarks);
		break;
	case 'n':
		break;
		}
	}
else if (style[0] != 'b' && style[0] != 'B')
	{
	switch (style[0])
		{
	case '-':
	case '0':
		yaxis(p, ytick, nyticks, bpos, lpos, 1.0, rmarks);
		xaxis(p, xtick, nxticks, bpos, lpos, 1.0, tmarks);
		xaxis(p, xtick, nxticks, bpos, rpos, 1.0, tmarks);
		yaxis(p, ytick, nyticks, bpos, rpos, -1.0, rmarks);
		break;
	case '+':
	case '#':
		yaxis(p, ytick, nyticks, tpos, lpos, 1.0, rmarks);
		xaxis(p, xtick, nxticks, tpos, lpos, -1.0, tmarks);
		xaxis(p, xtick, nxticks, tpos, rpos, -1.0, tmarks);
		yaxis(p, ytick, nyticks, tpos, rpos, -1.0, rmarks);
		break;
	case 'n':
		fputs("(axes) Can't have double y-axes and no x-axis.\n", dap_err);
		exit(1);
		}
	}
else if (style[1] != 'b' && style[1] != 'B')
	{
	switch (style[1])
		{
	case '-':
	case '0':
		xaxis(p, xtick, nxticks, bpos, lpos, 1.0, tmarks);
		yaxis(p, ytick, nyticks, bpos, lpos, 1.0, rmarks);
		yaxis(p, ytick, nyticks, tpos, lpos, 1.0, rmarks);
		xaxis(p, xtick, nxticks, tpos, lpos, -1.0, tmarks);
		break;
	case '+':
	case '#':
		xaxis(p, xtick, nxticks, bpos, rpos, 1.0, tmarks);
		yaxis(p, ytick, nyticks, bpos, rpos, -1.0, rmarks);
		yaxis(p, ytick, nyticks, tpos, rpos, -1.0, rmarks);
		xaxis(p, xtick, nxticks, tpos, rpos, -1.0, tmarks);
		break;
	case 'n':
		fputs("(axes) Can't have double x-axes and no y-axis.\n", dap_err);
		exit(1);
		}
	}
else
	{
	yaxis(p, ytick, nyticks, bpos, lpos, 1.0, rmarks);
	xaxis(p, xtick, nxticks, bpos, rpos, 1.0, tmarks);
	yaxis(p, ytick, nyticks, tpos, rpos, -1.0, rmarks);
	xaxis(p, xtick, nxticks, tpos, lpos, -1.0, tmarks);
	}
}

static void ticks(tick ticks[], double min, double space, double ticklen,
		char *form, int nticks, double labpos, char *alab, double (*tfunct)())
{
int n;
static char *lab;
double coord;

if (!form)
	{
	fputs("(ticks) No format.\n", dap_err);
	exit(1);
	}
if (!lab)
	lab = dap_malloc(dap_maxtxt + 1, "dap_maxtxt");
for (n = 0; n < nticks - 1; n++)
	{
	coord = min + space * (double) n;
	if (tfunct)
		sprintf(lab, form, (*tfunct)(coord));
	else
		sprintf(lab, form, coord);
	pict_maketick(ticks + n, coord, lab, ticklen);
	}
pict_maketick(ticks + n, labpos, alab, ticklen);
}

double pict_autoaxes(pict *p, char *xlab, char *ylab, char *axspec,
	double (*xfunct)(), double (*yfunct)(), char *caption, int autopos)
{
pict *pp;
int totpts;
double xspace, yspace;
double xlen, ylen;
double minx, maxx, miny, maxy;
double minx1, maxx1;
int nxticks, nyticks;
tick *xticks;
tick *yticks;
double xticklen;
double yticklen;
char xform[10], yform[10];
char as[3];
double lpos, rpos, bpos, tpos;
double xlabpos, ylabpos;
double captoff;
double width, height;
int lexpand, rexpand;
double specxmax, specxmin;
double specymax, specymin;
int specxticks, specyticks;
int a, w;
char *word;
double digs, places, sign;
int nt;

minx = 0.0;
maxx = 0.0;
miny = 0.0;
maxy = 0.0;
lpos = 0.0;
rpos = 0.0;
bpos = 0.0;
tpos = 0.0;
xlabpos = 0.0;
ylabpos = 0.0;
captoff = 0.0;
xticks = (tick *) dap_malloc(sizeof(tick) * dap_maxntxt, "dap_maxntxt");
yticks = (tick *) dap_malloc(sizeof(tick) * dap_maxntxt, "dap_maxntxt");
specxmin = 0.0 / 0.0;
specxmax = 0.0 / 0.0;
specymin = 0.0 / 0.0;
specymax = 0.0 / 0.0;
specxticks = -1;
specyticks = -1;
word = dap_malloc(dap_namelen + 1, "");
if (axspec && axspec[0])
	{
	as[0] = axspec[0];
	if (axspec[1] && axspec[1] != ' ')
		as[1] = axspec[1];
	else
		as[1] = '0';
	as[2] = '\0';
	for (a = 2; axspec[a] == ' '; a++)
		;
	while (axspec[a])
		{
		for (w = 0; axspec[a] && axspec[a] != ' '; )
			{
			if (w < dap_namelen)
				word[w++] = axspec[a++];
			else
				{
				word[w] = '\0';
				fprintf(dap_err,
			"pict_autoaxes: word in axspec too long: %s\n",
					word);
				exit(1);
				}
			}
		word[w] = '\0';
		if (!strncmp(word, "MAXX", 4) || !strncmp(word, "MINX", 4) ||
				!strncmp(word, "MAXY", 4) || !strncmp(word, "MINY", 4))
			{
			w = 4;
			sign = 1.0;
			if (word[w] == '-')
				{
				sign = -1.0;
				w++;
				}
			for (digs = 0.0, places = 0.0;
					('0' <= word[w] && word[w] <= '9') ||
					word[w] == '.'; w++)
				{
				if (word[w] == '.')
					{
					if (places > 0.0)
						{
						fprintf(dap_err,
				"pict_autoaxes: bad number for MIN or MAX: %s\n",
							word + 4);
						exit(1);
						}
					places = 1.0;
					}
				else
					{
					if (places > 0.0)
						places *= 10.0;
					digs = 10.0 * digs +
						(double) (word[w] - '0');
					}
				}
			digs *= sign;
			if (places > 0.0)
				digs /= places;
			if (word[w] && word[w] != ' ')
				{
				fprintf(dap_err,
				"pict_autoaxes: bad number for MIN or MAX: %s\n",
					word + 3);
				exit(1);
				}
			if (word[1] == 'A')
				{
				if (word[3] == 'X')
					specxmax = digs;
				else
					specymax = digs;
				}
			else
				{
				if (word[3] == 'X')
					specxmin = digs;
				else
					specymin = digs;
				}
			}
		else if (!strncmp(word, "NXTICKS", 7) ||
				!strncmp(word, "NYTICKS", 7))
			{
			for (nt = 0, w = 7; '0' <= word[w] && word[w] <= '9'; w++)
				nt = 10 * nt + word[w] - '0';
			if (word[w] && word[w] != ' ' || !nt)
				{
				fprintf(dap_err,
				"pict_autoaxes: bad number of ticks: %s\n",
					word + 7);
				exit(1);
				}
			if (word[1] == 'X')
				specxticks = nt + 1;
			else
				specyticks = nt - 1;
			}
		else
			{
			fprintf(dap_err,
			"pict_autoaxes: bad axes specification: %s\n",
				word);
			exit(1);
			}
		while (axspec[a] == ' ')
			a++;
		}
	}
else
	strcpy(as, "00");
for (pp = p, totpts = 0; pp && pp->pict_next; pp = pp->pict_next)
	{
	if (pp->pict_npts)
		{
		if (pp == p || minx > pp->pict_minx)
			minx = pp->pict_minx;
		if (pp == p || miny > pp->pict_miny)
			miny = pp->pict_miny;
		if (pp == p || maxx < pp->pict_maxx)
			maxx = pp->pict_maxx;
		if (pp == p || maxy < pp->pict_maxy)
			maxy = pp->pict_maxy;
		totpts += pp->pict_npts;
		}
	}
if (!totpts)
	{
	fprintf(dap_err, "pict_autoaxes: no points.\n");
	exit(1);
	}
if (finite(specxmin))
	minx = specxmin;
if (finite(specxmax))
	maxx = specxmax;
if (finite(specymin))
	miny = specymin;
if (finite(specymax))
	maxy = specymax;
minx1 = minx;
maxx1 = maxx;
if (minx > 0.0 && !finite(specxmin))
	minx = 0.0;
if (miny > 0.0 && !finite(specymin))
	miny = 0.0;
if (maxx < 0.0 && !finite(specxmax))
	maxx = 0.0;
if (maxy < 0.0 && !finite(specymax))
	maxy = 0.0;
xlen = 1e5 / (maxx - minx);
if (!finite(specxmin))
	minx = arint(xlen * minx) / xlen;
if (!finite(specxmax))
	maxx = arint(xlen * maxx) / xlen;
ylen = 1e5 / (maxy - miny);
if (!finite(specymin))
	miny = arint(ylen * miny) / ylen;
if (!finite(specymax))
	maxy = arint(ylen * maxy) / ylen;
xlen = maxx - minx;
if (xlen >= 1.0)
        {
        for (xspace = 1.0; 10.0 * xspace < xlen; xspace *= 10.0)
                ; 
        xspace *= ceil(xlen / xspace) / 10.0;
        }
else            
        {
        for (xspace = 0.1; 0.1 * xspace > xlen; xspace *= 0.1)
                ;
        xspace *= ceil(xlen / xspace) / 10.0;
        }
ylen = maxy - miny;
if (ylen >= 1.0)
        {
        for (yspace = 1.0; 10.0 * yspace < ylen; yspace *= 10.0)
                ;
        yspace *= ceil(ylen / yspace) / 10.0;
        }
else
        {
        for (yspace = 0.1; 0.1 * yspace > ylen; yspace *= 0.1)
                ;
        yspace *= ceil(ylen / yspace) / 10.0;
        }
if (xspace == rint(xspace))
	strcpy(xform, "%.0f");
else if (10.0 * xspace == rint(10.0 * xspace))
	strcpy(xform, "%.1f");
else
	strcpy(xform, "%.2f");
if (yspace == rint(yspace))
	strcpy(yform, "%.0f");
else if (10.0 * yspace == rint(10.0 * yspace))
	strcpy(yform, "%.1f");
else
	strcpy(yform, "%.2f");
if (minx < minx1)
	minx = xspace * floor(minx / xspace);
else
	minx = xspace * (floor(minx / xspace) - 1.0);
lexpand = (minx + xspace <= minx1);
	miny = yspace * floor(miny / yspace);
if (maxx > maxx1)
	maxx = xspace * ceil(maxx / xspace);
else
	maxx = xspace * (ceil(maxx / xspace) + 1.0);
rexpand = (maxx - xspace >= maxx1);
if (!finite(specymax))
	maxy = yspace * ceil(maxy / yspace);
xticklen = (maxy - miny) / 100.0;
yticklen = (maxx - minx) / 100.0;
if (specxticks > 0)
	{
	nxticks = 2 + specxticks - lexpand - rexpand;
	xspace = (maxx - minx) / ((double) specxticks);
	}
else
	nxticks = 2 + (int) rint((maxx - minx) / xspace) - lexpand - rexpand;
if (specyticks > 0)
	{
	nyticks = 2 + specyticks;
	yspace = (maxy - miny) / ((double) specyticks);
	}
else
	nyticks = 2 + (int) rint((maxy - miny) / yspace);
if (as[0] == 'b' || as[0] == 'B')
	{
	bpos = miny;
	tpos = maxy;
	xlabpos = 0.5 * (minx + maxx);
	captoff = (maxy - miny) / 5.0;
	}
else if (as[0] != 'n')
	{
	switch (as[0])
		{
	case '-':
		bpos = miny;
		xlabpos = 0.5 * (minx + maxx);
		captoff = (maxy - miny) / 5.0;
		break;
	case '+':
	case '#':
		tpos = maxy;
		xlabpos = 0.5 * (minx + maxx);
		captoff = (maxy - miny) / 5.0;
		break;
	case '0':
		bpos = 0.0;
		xlabpos = maxx + xspace;
		captoff = (maxy - miny) / 10.0;
		break;
	default:
		fprintf(dap_err, "(autoaxes) Bad axis specification: %s\n", axspec);
		exit(1);
		}
	}
if (as[1] == 'b' || as[1] == 'B')
	{
	lpos = minx;
	rpos = maxx;
	ylabpos = 0.5 * (miny + maxy);
	}
else if (as[1] != 'n')
	{
	switch (as[1])
		{
	case '-':
		lpos = minx;
		ylabpos = 0.5 * (miny + maxy);
		break;
	case '+':
	case '#':
		rpos = maxx;
		ylabpos = 0.5 * (miny + maxy);
		break;
	case '0':
		lpos = 0.0;
		ylabpos = maxy + yspace;
		break;
	default:
		fprintf(dap_err, "(autoaxes) Bad axis specification: %s\n", axspec);
		exit(1);
		}
	}
if (lexpand)
	ticks(xticks, minx + xspace, xspace, xticklen, xform, nxticks, xlabpos, xlab, xfunct);
else
	ticks(xticks, minx, xspace, xticklen, xform, nxticks, xlabpos, xlab, xfunct);
ticks(yticks, miny, yspace, yticklen, yform, nyticks, ylabpos, ylab, yfunct);
pict_axes(pp, xticks, nxticks, yticks, nyticks, as, bpos, lpos, tpos, rpos);
pict_text(pp, caption, 0.5 * (minx + maxx), miny - captoff, 0.0, "ct ");
if (autopos)
	{
	if (autopos == PORTRAIT)
		{
		width = PORTWIDTH;
		height = PORTHEIGHT;
		}
	else
		{
		width = LANDWIDTH;
		height = LANDHEIGHT;
		}
	pict_scale(p, 0.5 * (minx + maxx), 0.5 * (miny + maxy),
			width / (maxx - minx), height / (maxy - miny));
	pict_translate(p, LEFT + 0.5 * (width - (minx + maxx)),
			BOTTOM + 0.5 * (height - (miny + maxy)));
	xspace *= width / (maxx - minx);
	yspace *= width / (maxy - miny);
	}
dap_free(xticks);
dap_free(yticks);
dap_free(word);
return sqrt(xspace * xspace + yspace * yspace);
}

#if 0
int pict_npts;				/* number of points in pict, if any */
char pict_type[5];			/* "LINE" = connected lines */
double pict_dash;			/* dash length for lines if > 0.0 */
double (*pict_pt)[2];			/* the points */
double pict_minx, pict_maxx;		/* bounds */
double pict_miny, pict_maxy;		/* bounds */
int pict_ntxt;				/* number of texts */
char **pict_txt;			/* text to display */ 
char *pict_font;			/* font for displayed text, if any */
double pict_fs;				/* font size */
double **pict_tpt;			/* location of text */
double *pict_tang;			/* angle for text */
char **pict_pos;			/* text position */
double pict_lw;				/* line width */
double pict_r;				/* radius for circles */
double pict_lgray;              	/* gray level for lines */ 
double pict_fgray;              	/* gray level for fill: if >= 0, fill then stroke */
struct _pict *pict_patt;        	/* pict to use for fill or patterned points */
struct _pict *pict_next;        	/* for linking in list */
#endif

#define MAXPICTSAVE 1000

void pict_save(pict *p, int npicts, char *dataset)
{
pict *firstp;	/* start of pict list */
int pn;		/* picture number */
char strspec[15];
int len;
int maxlen;
int pict_npts;				/* number of points in pict, if any */
char pict_type[5];			/* "LINE" = connected lines */
double pict_dash;			/* dash length for lines if > 0.0 */
double pict_minx, pict_maxx;		/* bounds */
double pict_miny, pict_maxy;		/* bounds */
int pict_ntxt;				/* number of texts */
char *pict_font;			/* font for displayed text, if any */
double pict_fs;				/* font size */
double pict_lw;				/* line width */
double pict_r;				/* radius for circles */
double pict_lgray;              	/* gray level for lines */ 
double pict_fgray;              	/* gray level for fill: if >= 0, fill then stroke */
int pict_next;				/* next pict in list */
double pict_pt[2];			/* the points */
int ptn;
char *pict_txt;
double pict_tpt[2];			/* location of text */
double pict_tang;			/* angle for text */
char pict_pos[4];			/* text position */
char *outname;
int pict_patt;				/* pattern number */

firstp = p;

for (p = firstp, maxlen = 0; p; )
	{
	len = strlen(p->pict_font);
	if (len > maxlen)
		maxlen = len;
	for (ptn = 0; ptn < p->pict_ntxt; ptn++)
		{
		len = strlen(p->pict_txt[ptn]);
		if (len > maxlen)
			maxlen = len;
		}
	if (npicts)
		{
		if (++p >= firstp + npicts)
			break;
		}
	else
		p = p->pict_next;
	}
if (maxlen > 9998)
	{
	fprintf(dap_err, "(pict_save) maximum string length too long: %d\n", maxlen);
	exit(1);
	}
pict_font = dap_malloc(maxlen + 1, "");
pict_txt = dap_malloc(maxlen + 1, "");

/* first write out basic structure */
infile(NULL, NULL);
dap_vd("pict_npts 0", 0); dap_il("pict_npts", &pict_npts);
dap_vd("pict_type 5", 0); dap_sl("pict_type", pict_type);
dap_vd("pict_dash -1", 0); dap_dl("pict_dash", &pict_dash);
dap_vd("pict_minx -1", 0); dap_dl("pict_minx", &pict_minx);
dap_vd("pict_maxx -1", 0); dap_dl("pict_maxx", &pict_maxx);
dap_vd("pict_miny -1", 0); dap_dl("pict_miny", &pict_miny);
dap_vd("pict_maxy -1", 0); dap_dl("pict_maxy", &pict_maxy);
dap_vd("pict_ntxt 0", 0); dap_il("pict_ntxt", &pict_ntxt);
sprintf(strspec, "pict_font %d", maxlen);
dap_vd(strspec, 0); dap_sl("pict_font", pict_font);
dap_vd("pict_fs -1", 0); dap_dl("pict_fs", &pict_fs);
dap_vd("pict_lw -1", 0); dap_dl("pict_lw", &pict_lw);
dap_vd("pict_r -1", 0); dap_dl("pict_r", &pict_r);
dap_vd("pict_lgray -1", 0); dap_dl("pict_lgray", &pict_lgray);
dap_vd("pict_fgray -1", 0); dap_dl("pict_fgray", &pict_fgray);
dap_vd("pict_next 0", 0); dap_il("pict_next", &pict_next);
dap_vd("pict_patt 0", 0); dap_il("pict_patt", &pict_patt);
outname = dap_malloc(strlen(dataset) + 9, ""); /* max 10000 picts */
for (pn = 0, p = firstp; p; pn++)
	{
	if (pn < MAXPICTSAVE)
		{
		sprintf(outname, "%s.pic%04d", dataset, pn);
		outset(outname, "");
		pict_npts = p->pict_npts;
		strcpy(pict_type, p->pict_type);
		pict_dash = p->pict_dash;
		pict_minx = p->pict_minx;
		pict_maxx = p->pict_maxx;
		pict_miny = p->pict_miny;
		pict_maxy = p->pict_maxy;
		pict_ntxt = p->pict_ntxt;
		strcpy(pict_font, p->pict_font);
		pict_fs = p->pict_fs;
		pict_lw = p->pict_lw;
		pict_r = p->pict_r;
		pict_lgray = p->pict_lgray;
		pict_fgray = p->pict_fgray;
		if (npicts)
			{
			if (pn < npicts - 1)
				{
				if (p->pict_next)
					pict_next = -(p->pict_next - firstp);
				else
					pict_next = -pn;
				}
			else
				pict_next = 0;
			}
		else
			{
			if (p->pict_next)
				pict_next = pn + 1;
			else
				pict_next = 0;
			}
		if (p->pict_patt)
			pict_patt = 1;
		else
			pict_patt = 0;
		output();
		}
	else
		{
		fputs("(pict_save) too many picts.\n", dap_err);
		exit(1);
		}
	if (npicts)
		{
		if (++p >= firstp + npicts)
			break;
		}
	else
		p = p->pict_next;
	}
infile(NULL, NULL);
dap_vd("pict_pt[0] -1 pict_pt[1] - 1", 0);
dap_dl("pict_pt", pict_pt);
for (pn = 0, p = firstp; p; pn++)
	{
	sprintf(outname, "%s.pts%04d", dataset, pn);
	outset(outname, "");
	for (ptn = 0; ptn < p->pict_npts; ptn++)
		{
		pict_pt[0] = p->pict_pt[ptn][0];
		pict_pt[1] = p->pict_pt[ptn][1];
		output();
		}
	if (npicts)
		{
		if (++p >= firstp + npicts)
			break;
		}
	else
		p = p->pict_next;
	}
infile(NULL, NULL);
sprintf(strspec, "pict_txt %d", maxlen);
dap_vd(strspec, 0); dap_sl("pict_txt", pict_txt);
dap_vd("pict_tpt[0] -1 pict_tpt[1] -1", 0);
dap_dl("pict_tpt", pict_tpt);
dap_vd("pict_tang -1", 0); dap_dl("pict_tang", &pict_tang);
dap_vd("pict_pos 3", 0); dap_sl("pict_pos", pict_pos);
for (pn = 0, p = firstp; p; pn++)
	{
	sprintf(outname, "%s.txt%04d", dataset, pn);
	outset(outname, "");
	for (ptn = 0; ptn < p->pict_ntxt; ptn++)
		{
		strcpy(pict_txt, p->pict_txt[ptn]);
		pict_tpt[0] = p->pict_tpt[ptn][0];
		pict_tpt[1] = p->pict_tpt[ptn][1];
		pict_tang = p->pict_tang[ptn];
		strcpy(pict_pos, p->pict_pos[ptn]);
		output();
		}
	if (npicts)
		{
		if (++p >= firstp + npicts)
			break;
		}
	else
		p = p->pict_next;
	}
for (pn = 0, p = firstp; p; pn++)
	{
	if (p->pict_patt)
		{
		sprintf(outname, "%s.pat%04d", dataset, pn++);
		pict_save(p->pict_patt, 0, outname);
		}
	if (npicts)
		{
		if (++p >= firstp + npicts)
			break;
		}
	else
		p = p->pict_next;
	}
dap_free(outname);
dap_free(pict_font);
dap_free(pict_txt);
}

pict *pict_rest(char *dataset)
{
int npict;	/* total number of picts, including patterns */
pict *p;
char *inname;
int pn;
int ptn;
int pnext;
int npicts;
/* all these are indices to variables */
int pict_npts;				/* number of points in pict, if any */
int pict_type;				/* "LINE" = connected lines */
int pict_dash;				/* dash length for lines if > 0.0 */
int pict_minx, pict_maxx;		/* bounds */
int pict_miny, pict_maxy;		/* bounds */
int pict_ntxt;				/* number of texts */
int pict_font;				/* font for displayed text, if any */
int pict_fs;				/* font size */
int pict_lw;				/* line width */
int pict_r;				/* radius for circles */
int pict_lgray;              		/* gray level for lines */ 
int pict_fgray;              		/* gray level for fill: if >= 0, fill then stroke */
int pict_pt;				/* the points */
int pict_txt;
int pict_tpt;				/* location of text */
int pict_tang;				/* angle for text */
int pict_pos;				/* text position */
int pict_next;				/* link to next */
int pict_patt;				/* pattern number */
int dim;
int npts;
double *dblmem;
char *charmem;

inname = dap_malloc(strlen(dataset) + 9, "");
for (pn = 0, npicts = 0; ; npicts++)
	{
	sprintf(inname, "%s.pic%04d", dataset, pn);
	inset(inname);
	step();
	if ((pict_next = dap_varnum("pict_next")) < 0)
		{
		fprintf(dap_err, "(pict_rest) no pict_next in %s\n", inname);
		exit(1);
		}
	pnext = dap_obs[0].do_int[pict_next];
	if (!pnext)
		break;
	if (pnext > 0)
		pn = pnext;
	else if (pnext < 0)
		++pn;
	}
npicts++;
p = (pict *) dap_malloc(sizeof(pict) * npicts, "");
for (pn = 0; pn < npicts; pn++)
	{
	sprintf(inname, "%s.pic%04d", dataset, pn);
	inset(inname);
	step();
	if ((pict_npts = dap_varnum("pict_npts")) < 0)
		{
		fputs("(pict_rest) missing pict_npts\n", dap_err);
		exit(1);
		}
	p[pn].pict_npts = dap_obs[0].do_int[pict_npts];
	if ((pict_type = dap_varnum("pict_type")) < 0)
		{
		fputs("(pict_rest) missing pict_type\n", dap_err);
		exit(1);
		}
	strcpy(p[pn].pict_type, dap_obs[0].do_str[pict_type]);
	if ((pict_dash = dap_varnum("pict_dash")) < 0)
		{
		fputs("(pict_rest) missing pict_dash\n", dap_err);
		exit(1);
		}
	p[pn].pict_dash = dap_obs[0].do_dbl[pict_dash];
	if ((pict_minx = dap_varnum("pict_minx")) < 0)
		{
		fputs("(pict_rest) missing pict_minx\n", dap_err);
		exit(1);
		}
	p[pn].pict_minx = dap_obs[0].do_dbl[pict_minx];
	if ((pict_maxx = dap_varnum("pict_maxx")) < 0)
		{
		fputs("(pict_rest) missing pict_maxx\n", dap_err);
		exit(1);
		}
	p[pn].pict_maxx = dap_obs[0].do_dbl[pict_maxx];
	if ((pict_miny = dap_varnum("pict_miny")) < 0)
		{
		fputs("(pict_rest) missing pict_miny\n", dap_err);
		exit(1);
		}
	p[pn].pict_miny = dap_obs[0].do_dbl[pict_miny];
	if ((pict_maxy = dap_varnum("pict_maxy")) < 0)
		{
		fputs("(pict_rest) missing pict_maxy\n", dap_err);
		exit(1);
		}
	p[pn].pict_maxy = dap_obs[0].do_dbl[pict_maxy];
	if ((pict_ntxt = dap_varnum("pict_ntxt")) < 0)
		{
		fputs("(pict_rest) missing pict_ntxt\n", dap_err);
		exit(1);
		}
	p[pn].pict_ntxt = dap_obs[0].do_int[pict_ntxt];
	if ((pict_font = dap_varnum("pict_font")) < 0)
		{
		fputs("(pict_rest) missing pict_font\n", dap_err);
		exit(1);
		}
	p[pn].pict_font = dap_malloc(strlen(dap_obs[0].do_str[pict_font]) + 1, "");
	strcpy(p[pn].pict_font, dap_obs[0].do_str[pict_font]);
	if ((pict_fs = dap_varnum("pict_fs")) < 0)
		{
		fputs("(pict_rest) missing pict_fs\n", dap_err);
		exit(1);
		}
	p[pn].pict_fs = dap_obs[0].do_dbl[pict_fs];
	if ((pict_lw = dap_varnum("pict_lw")) < 0)
		{
		fputs("(pict_rest) missing pict_lw\n", dap_err);
		exit(1);
		}
	p[pn].pict_lw = dap_obs[0].do_dbl[pict_lw];
	if ((pict_r = dap_varnum("pict_r")) < 0)
		{
		fputs("(pict_rest) missing pict_r\n", dap_err);
		exit(1);
		}
	p[pn].pict_r = dap_obs[0].do_dbl[pict_r];
	if ((pict_lgray = dap_varnum("pict_lgray")) < 0)
		{
		fputs("(pict_rest) missing pict_lgray\n", dap_err);
		exit(1);
		}
	p[pn].pict_lgray = dap_obs[0].do_dbl[pict_lgray];
	if ((pict_fgray = dap_varnum("pict_fgray")) < 0)
		{
		fputs("(pict_rest) missing pict_fgray\n", dap_err);
		exit(1);
		}
	p[pn].pict_fgray = dap_obs[0].do_dbl[pict_fgray];
	if ((pict_next = dap_varnum("pict_next")) < 0)
		{
		fputs("(pict_rest) missing pict_next\n", dap_err);
		exit(1);
		}
	pnext = dap_obs[0].do_int[pict_next];
	if (pnext < 0)
		{
		if (pnext == -pn)
			p[pn].pict_next = NULL;
		else
			p[pn].pict_next = p - pnext;
		}
	else if (pnext)
		p[pn].pict_next = p + pn + 1;
	else
		p[pn].pict_next = NULL;
	if ((pict_patt = dap_varnum("pict_patt")) < 0)
		{
		fputs("(pict_rest) missing pict_patt\n", dap_err);
		exit(1);
		}
	if (dap_obs[0].do_int[pict_patt])
		{
		sprintf(inname, "%s.pat%04d", dataset, pn);
		p[pn].pict_patt = pict_rest(inname);
		}
	else
		p[pn].pict_patt = NULL;
	}
for (pn = 0; pn < npicts; pn++)
	{
	sprintf(inname, "%s.pts%04d", dataset, pn);
	inset(inname);
	if ((pict_pt = dap_arrnum("pict_pt", &dim)) < 0)
		{
		fputs("(pict_rest) missing pict_pt\n", dap_err);
		exit(1);
		}
	if (dim != 2)
		{
		fprintf(dap_err, "(pict_rest) bad dimension for pict_pt: %d\n", dim);
		exit(1);
		}
	npts = p[pn].pict_npts;
	for (ptn = 0, p[pn].pict_npts = 0; ptn < npts; ptn++)
		{
		step();
		pict_point(p + pn, dap_obs[0].do_dbl[pict_pt],
					dap_obs[0].do_dbl[pict_pt + 1]);
		}
	}
for (pn = 0; pn < npicts; pn++)
	{
	sprintf(inname, "%s.txt%04d", dataset, pn);
	inset(inname);
	npts = p[pn].pict_ntxt;
	p[pn].pict_txt =
		(char **) dap_malloc(sizeof(char *) * dap_maxntxt, "dap_maxntxt");
	dblmem =
		(double *) dap_malloc(sizeof(double) * 2 * dap_maxntxt, "dap_maxntxt");
	p[pn].pict_tpt =
		(double **) dap_malloc(sizeof(double *) * dap_maxntxt, "dap_maxntxt");
	for (ptn = 0; ptn < dap_maxntxt; ptn++)
		p[pn].pict_tpt[ptn] = dblmem + 2 * ptn;
	p[pn].pict_tang =
		(double *) dap_malloc(sizeof(double) * dap_maxntxt, "dap_maxntxt");
	charmem = dap_malloc(3 * dap_maxntxt, "dap_maxntxt"); 
	p[pn].pict_pos =
		(char **) dap_malloc(sizeof(char *) * dap_maxntxt, "dap_maxntxt");
	for (ptn = 0; ptn < dap_maxntxt; ptn++)
		p[pn].pict_pos[ptn] = charmem + 3 * ptn;
	if ((pict_txt = dap_varnum("pict_txt")) < 0)
		{
		fputs("(pict_rest) missing pict_txt\n", dap_err);
		exit(1);
		}
	if ((pict_tpt = dap_arrnum("pict_tpt", &dim)) < 0)
		{
		fputs("(pict_rest) missing pict_tpt\n", dap_err);
		exit(1);
		}
	if (dim != 2)
		{
		fprintf(dap_err, "(pict_rest) bad dimension for pict_tpt: %d\n", dim);
		exit(1);
		}
	if ((pict_tang = dap_varnum("pict_tang")) < 0)
		{
		fputs("(pict_rest) missing pict_tang\n", dap_err);
		exit(1);
		}
	if ((pict_pos = dap_varnum("pict_pos")) < 0)
		{
		fputs("(pict_rest) missing pict_pos\n", dap_err);
		exit(1);
		}
	for (ptn = 0, p[pn].pict_ntxt = 0; ptn < npts; ptn++)
		{
		step();
		pict_text(p + pn, dap_obs[0].do_str[pict_txt],
			dap_obs[0].do_dbl[pict_tpt], dap_obs[0].do_dbl[pict_tpt + 1],
			dap_obs[0].do_dbl[pict_tang], dap_obs[0].do_str[pict_pos]);
		}
	}
dap_free(inname);
return p;
}
