/*$Id: Firekeeper.cpp,v 1.31 2008/06/01 12:21:01 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006, 2007, 2008 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "nsCOMPtr.h"
#include "nsXPCOM.h"
#include "nsMemory.h"
#include "nsEmbedString.h"
#include "plstr.h"

#include "nsIServiceManager.h"

#include "nsIURI.h"
#include "nsIChannel.h"
#include "nsIObserverService.h"
#include "nsIComponentManager.h"
#include "nsIComponentRegistrar.h"
#include "nsIStreamListenerTee.h"

#include "Firekeeper.h"
#include "Error.h"
#include "Common.h"

#include "fkRequestExaminator.h"
#include "fkStreamListenerTeeFactory.h"
#include "fkSentence.h"
#include "fkRule.h"
#include "ActiveRuleSet.h"

#define NS_CATEGORYMANAGER_CONTRACTID "@mozilla.org/categorymanager;1"

static NS_DEFINE_CID(kStreamListenerTeeCID, NS_STREAMLISTENERTEE_CID);
static NS_DEFINE_CID(kOrigStreamListenerTeeCID, NS_ORIGSTREAMLISTENERTEE_CID);


nsDataHashtable<nsISupportsHashKey, fkRequestExaminator *> *examinatorHashTable;

/*Loaded rules*/
Rules *rules = nsnull;

nsCOMPtr<fkIJudge> fkJudge = nsnull;

/*TODO: this should be configurable*/
const char *rulesOrder[]={"pass", "drop", "alert"};

Firekeeper::Firekeeper() : enabled(false)
{
	TRACE("constructor %08x", this);
}

Firekeeper::~Firekeeper()
{
	TRACE("destructor %08x", this);
	FK_ASSERT(false);
}

NS_IMPL_ISUPPORTS2(Firekeeper, fkIFirekeeper,
		   nsIObserver)

/*
  Return global Firekeeper object (singleton)
 */
Firekeeper*
Firekeeper::GetInstance()
{
	static Firekeeper *firekeeper;

	TRACE("");
	if (firekeeper == nsnull){
		firekeeper = new Firekeeper();
		if (!firekeeper)
			return nsnull;

		/*don't destroy firekeeper*/
		NS_ADDREF(firekeeper);

		nsresult rv = firekeeper->Init();
		if (NS_FAILED(rv)){
			//TODO: delete firekeeper; ...
			return nsnull;
		}
	}

	NS_ADDREF(firekeeper);
	return firekeeper;
}



NS_IMETHODIMP
Firekeeper::ReplaceTeeListener() {
	TRACE("");
	nsCOMPtr<nsIComponentManager> compmgr;
	nsresult rv = NS_GetComponentManager(getter_AddRefs(compmgr));
	if (NS_FAILED(rv)){
		TRACE("Can't get ComponentManager");
		return rv;
	}
	nsCOMPtr<nsIComponentRegistrar> compreg = 
		do_QueryInterface(compmgr, &rv);
	if (NS_FAILED(rv)){
		TRACE("Can't get ComponentRegistrar");
		return rv;
	}

	nsCOMPtr<nsIStreamListenerTee> tee;
	nsCOMPtr<nsIFactory> factory;
	rv = compmgr->GetClassObjectByContractID(NS_STREAMLISTENERTEE_CONTRACTID,
						 NS_GET_IID(nsIFactory),
						 getter_AddRefs(factory));
	FK_ASSERT(!NS_FAILED(rv));
	FK_ASSERT(factory != nsnull);

	if (NS_FAILED(rv)){
		return rv;
	}

	rv = factory->CreateInstance(nsnull, NS_GET_IID(nsIStreamListenerTee), 
				     getter_AddRefs(tee));
	if (NS_FAILED(rv)){
		return rv;
	}
	FK_ASSERT(!NS_FAILED(rv));
	
	rv = compreg->UnregisterFactory(kStreamListenerTeeCID,
					factory);
	FK_ASSERT(!NS_FAILED(rv));
	if (NS_FAILED(rv)){
		return rv;
	}
	rv = compreg->RegisterFactory(kOrigStreamListenerTeeCID, "nsStreamListenerTee",
				      NS_ORIGSTREAMLISTENERTEE_CONTRACTID,
				      factory);
	FK_ASSERT(!NS_FAILED(rv));
	if (NS_FAILED(rv)){
		return rv;
	}

	fkStreamListenerTeeFactory *fktee_factory_raw = new fkStreamListenerTeeFactory();
	if (!fktee_factory_raw)
		return NS_ERROR_OUT_OF_MEMORY;

	//TODO use separate var.
	nsCOMPtr<nsIFactory> fktee_factory = do_QueryInterface(fktee_factory_raw, &rv);
	FK_ASSERT(!NS_FAILED(rv));
	
	rv = compreg->RegisterFactory(kStreamListenerTeeCID, "fkStreamListenerTee",
				      NS_STREAMLISTENERTEE_CONTRACTID,
				      fktee_factory);
	FK_ASSERT(!NS_FAILED(rv));
	if (NS_FAILED(rv)){
		return rv;
	}

	TRACE("nsStreamListenerTee replaced");
	return NS_OK;
}

/*
  Init Firekeeper. Replace oryginal Mozilla's HTTP and HTTPS handlers
  with new ones, able to trace and check traffic.
 */
NS_IMETHODIMP
Firekeeper::Init()
{
	nsresult rv;
	TRACE("");	
	
	rules = new Rules();
	if (!rules || rules->init() < 0)
		return NS_ERROR_OUT_OF_MEMORY;

	if (!examinatorHashTable){
		rv = NS_ERROR_OUT_OF_MEMORY;
		examinatorHashTable = new nsDataHashtable<nsISupportsHashKey, 
			fkRequestExaminator *>;
		if (!examinatorHashTable || NS_FAILED(rv = examinatorHashTable->Init()))
			return rv;
	}

	
	nsCOMPtr<nsIServiceManager> servman;
	rv = NS_GetServiceManager(getter_AddRefs(servman));
	if (NS_FAILED(rv)){
		TRACE("Can't get ServiceManager %08x", rv);
		return rv;
	}

	nsCOMPtr<nsIObserverService> observerService;
	rv = servman->
		GetServiceByContractID("@mozilla.org/observer-service;1",
				       NS_GET_IID(nsIObserverService),
				       getter_AddRefs(observerService));
	if (NS_FAILED(rv)){
		TRACE("Can't get observer-service");
		return rv;
	}
	
#if 0
	rv = observerService->AddObserver(this, "http-on-examine-response", false);
	if (NS_FAILED(rv)){
		TRACE("Can't add http-on-examine-response observer");
		return rv;
	}
	TRACE("http-on-examine-response registered");
#endif
	
	rv = observerService->AddObserver(this, "http-on-modify-request", false);
	if (NS_FAILED(rv)){
		TRACE("Can't add http-on-modify-request observer");
		return rv;
	}
	TRACE("http-on-modify-request registered");

	TRACE("replacing tee listener");
	rv = ReplaceTeeListener();
	if (NS_FAILED(rv)){
		TRACE("failed to replace tee listener");
		return rv;
	}

	return NS_OK;
}

//-----------------------------------------------------------------------------
// Firekeeper::fkIFirekeeper
//-----------------------------------------------------------------------------
NS_IMETHODIMP
Firekeeper::AddRule(const char *rule, char **errmsg)
{
	char *error = nsnull;
	TRACE("");
	if (!errmsg)
		return NS_ERROR_NULL_POINTER;

	/*TODO: don't create parser here*/
	RulesParser p;

	if (p.init(&error) < 0){
		err_msg(error);
		return NS_ERROR_FAILURE;
	}
	Rule *newRule =  p.parseRule(rule, &error);

	if (error){
		TRACE("error %s", error);
		*errmsg = PL_strdup(error);
		if (!*errmsg)
			return NS_ERROR_OUT_OF_MEMORY;

		/*Can't return NS_ERROR_FAILURE here, it would
		  make errmsg variable inaccessible in javascript*/
		return NS_OK;
	}

	rules->addRule(newRule);
	searchAPI->search_compile_all();

	return NS_OK;
}

NS_IMETHODIMP
Firekeeper::AddRules(const char *rulesbuf, char **errmsg)
{
	char *error;
	TRACE("");
	if (!errmsg)
		return NS_ERROR_NULL_POINTER;

	RulesParser p;
	if (p.init(&error)){
		err_msg(error);
		return NS_ERROR_FAILURE;
	}
	p.parseRules(rules, rulesbuf, &error);
	searchAPI->search_compile_all();

	if (error){
		TRACE("error %s", error);
		*errmsg = PL_strdup(error);
		if (!*errmsg)
			return NS_ERROR_OUT_OF_MEMORY;

		/*even when there were parsing errors return NS_OK*/
	}

	return NS_OK;
}

NS_IMETHODIMP
Firekeeper::SetJudge(fkIJudge *judge)
{
	TRACE("");
	fkJudge = judge;
	return NS_OK;
}

NS_IMETHODIMP
Firekeeper::Enable(PRBool tracingEnabled)
{
	TRACE("");
	enabled = tracingEnabled;
	return NS_OK;
}

NS_IMETHODIMP
Firekeeper::IsEnabled(PRBool *tracingEnabled)
{
	TRACE("");
	if (!tracingEnabled)
		return NS_ERROR_NULL_POINTER;
	*tracingEnabled = enabled;
	return NS_OK;
}

NS_IMETHODIMP
Firekeeper::ResetRules()
{
	TRACE("");
	if (rules == nsnull)
		return NS_ERROR_FAILURE;

	rules->resetRules();
	TRACE("done");
	return NS_OK;
}

NS_IMETHODIMP
Firekeeper::examineRequest(nsIChannel *channel) {
	nsCOMPtr<nsIURI> uri;
	nsEmbedCString url;
	channel->GetURI(getter_AddRefs(uri));
	uri->GetSpec(url);

	fkRequestExaminator *examinator = new fkRequestExaminator();
	if (!examinator || NS_FAILED(examinator->Init()))
		return NS_ERROR_OUT_OF_MEMORY;

	const fkSentence *sentence = examinator->CheckUrl(url.get());

	if (sentence && sentence->action == sentence->BLOCK){
		channel->Cancel(NS_ERROR_ABORT);
		delete examinator;
		return NS_ERROR_ABORT;
	}
	
	FK_ASSERT(!examinatorHashTable->Get(channel, nsnull));
	examinatorHashTable->Put(channel, examinator);
		
	return NS_OK;
}


//-----------------------------------------------------------------------------
// Firekeeper::nsIObserver
//-----------------------------------------------------------------------------
NS_IMETHODIMP
Firekeeper::Observe(nsISupports *subject, const char* topic,
		    const PRUnichar* dat)
{
	nsresult rv;

	if (!topic || !enabled || strcmp(topic, "app-startup") == 0)
		return NS_OK;
	
	nsCOMPtr<nsIChannel> channel = do_QueryInterface(subject, &rv);
	if (NS_FAILED(rv)){
		TRACE("can't get nsIChannel interface");
		return rv;
	}

#ifdef DEBUG
	nsEmbedCString url;
	nsCOMPtr<nsIURI> uri;
	channel->GetURI(getter_AddRefs(uri));
	uri->GetSpec(url);
	TRACE("%s http-on-modify-request", url.get());
#endif

	return examineRequest(channel);
}
