/*****************************************************************************
 * Authors: Louis Bavoil <louis@bavoil.net>
 *          Fredrik Hbinette <hubbe@hubbe.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include "mozplugger.h"

#define THIS (*(data_t **) &(instance->pdata))
#define FEND(X,Y) (strchr(X,Y)?strchr(X,Y):(X)+strlen(X))

/*****************************************************************************
 * Type declarations
 *****************************************************************************/
typedef struct argument
{
  char *var;
  char *value;
} argument_t;

typedef struct data
{
     Display *display;
     char *displayname;
     NPWindow windata;
     int pid;
     int repeats;
     int flags;
     char *command;
     char *winname;
     uint16 mode;
     char *mimetype;
     char *href;
     char autostart;
     int num_arguments;
     struct argument *args;
} data_t;

typedef struct mimetype
{
     struct mimetype *next;
     char *line;
} mimetype_t;

typedef struct command
{
     struct command *next;
     int flags;
     char *cmd;
     char *winname;
} command_t;

typedef struct handle
{
     struct handle *next;
     struct mimetype *types;
     struct command *commands;
} handle_t;

typedef struct
{
     char *name;
     int flag;
} options_t;

/*****************************************************************************
 * Global variables
 *****************************************************************************/
static char *helper_fname;
static char *controller_fname;
static struct handle *first_handle;
static struct handle **prev_handlep;
static struct handle **handlep;
static char tempfile;

/*****************************************************************************
 * Wrapper for fork() which handles some magic needed to prevent netscape
 * from freaking out. It also prevents interferrance from signals.
 *****************************************************************************/
static int my_fork(NPP instance)
{
     int pid;
     sigset_t set,oset;

     /* Mask all the signals to avoid being interrupted by a signal */
     sigfillset(&set);
     sigprocmask(SIG_SETMASK, &set, &oset);

     D(">>>>>>>>Forking<<<<<<<<,\n");

     if ((pid = fork()) == -1)
	  return -1;

     if (!pid)
     {
	  int signum;
	  alarm(0);

	  if (!(THIS->flags & H_DAEMON))
	       setsid();

	  for (signum=0; signum < NSIG; signum++)
	       signal(signum, SIG_DFL);

#ifdef DEBUG
	  close_debug();
	  D("Child re-opened ndebug\n");
#endif
	  THIS->display=0;
     } else {
	  D("Child running with pid=%d\n", pid);
     }

     /* Restore the signal mask */
     sigprocmask(SIG_SETMASK, &oset, &set);

     return pid;
}

/*****************************************************************************
 * Wrapper for putenv().
 *****************************************************************************/
#define ENV_BUFFER 16348
static void my_putenv(char *buffer, int *offset, const char *var, const char *value)
{
     int l = strlen(var) + strlen(value) + 2;
     if (!(*offset + l < ENV_BUFFER))
     {
	  D("Buffer overflow in putenv(%s=%s)\n", var, value);
	  return;
     }
     
     sprintf(buffer+*offset, "%s=%s", var, value);
     putenv(buffer+*offset);
     *offset += l;
}

/*****************************************************************************
 * Wrapper for execlp() that calls the helper.
 *****************************************************************************/
static void run(NPP instance, const char *file)
{
     char buffer[ENV_BUFFER];
     char foo[64];
     int offset = 0;
     int i;

     if ((THIS->flags & H_CONTROLS) && (THIS->mode == NP_FULL))
	  THIS->flags &= ~(H_CONTROLS | H_SWALLOW | H_FILL);

     snprintf(buffer, sizeof(buffer), "%d,%d,%lu,%d,%d,%d,%d",
	     THIS->flags,
	     THIS->repeats,
	     (unsigned long int) THIS->windata.window,
	     (int) THIS->windata.x,
	      (int) THIS->windata.y,
	     (int) THIS->windata.width,
	     (int) THIS->windata.height);

     D("Executing helper: %s %s %s %s %s %s\n",
       helper_fname,
       buffer,
       file,
       THIS->displayname,
       THIS->command,
       THIS->mimetype);

     offset = strlen(buffer)+1;

     snprintf(foo, sizeof(foo), "%lu", (long unsigned)THIS->windata.window);
     my_putenv(buffer, &offset, "window", foo);

     snprintf(foo, sizeof(foo), "0x%lx", (long unsigned)THIS->windata.window);
     my_putenv(buffer, &offset, "hexwindow", foo);

     snprintf(foo, sizeof(foo), "%ld", (long)THIS->repeats);
     my_putenv(buffer, &offset, "repeat", foo);

     snprintf(foo, sizeof(foo), "%ld", (long)THIS->windata.width);
     my_putenv(buffer, &offset, "width", foo);

     snprintf(foo, sizeof(foo), "%ld", (long)THIS->windata.height);
     my_putenv(buffer, &offset, "height", foo);

     my_putenv(buffer, &offset, "mimetype", THIS->mimetype);
     
     my_putenv(buffer, &offset, "file", file);

     my_putenv(buffer, &offset, "autostart", THIS->autostart ? "1" : "0");
      
     if (THIS->winname)
	  my_putenv(buffer, &offset, "winname", THIS->winname);
     
     if (THIS->displayname)
	  my_putenv(buffer, &offset, "DISPLAY", THIS->displayname);

     if (controller_fname)
	  my_putenv(buffer, &offset, "controller", controller_fname);

     for (i = 0; i < THIS->num_arguments; i++)
	  if (THIS->args[i].value)
	       my_putenv(buffer, &offset, THIS->args[i].var, THIS->args[i].value);

     execlp(helper_fname, helper_fname, buffer, THIS->command, NULL);

     D("EXECLP FAILED!\n");
     
     _exit(EX_UNAVAILABLE);
}

/*****************************************************************************
 * Check if 'file' is somewhere to be found in the user PATH.
 *****************************************************************************/
static int inpath(char *path, char *file)
{
     static struct stat filestat;
     static char buf[1024];
     int i;
     int count = 1;

     for (i = strlen(path)-1; i > 0; i--)
	  if (path[i] == ':')
	  {
	       path[i] = 0;
	       count++;
	  }

     for (i = 0; i < count; i++)
     {
	  snprintf(buf, sizeof(buf), "%s/%s", path, file);
	  D("stat(%s)\n", buf);
	  if (!stat(buf, &filestat))
	  {
	       D("yes\n");
	       return 1;
	  }
	  D("no\n");
	  path += (strlen(path) + 1);
     }

     return 0;
}

/*****************************************************************************
 * Check if 'file' exists.
 *****************************************************************************/
static int find(char *file)
{
     static struct stat filestat;
     static char path[1024];
     char *env_path;

     D("find(%s)\n", file);

     if (file[0] == '/')
	  return !stat(file, &filestat);
     
     /* Get the environment variable PATH */
     if (!(env_path = getenv("PATH")))
     {
	  D("No PATH !\n");
	  return 0;
     }
     
     /* Duplicate to avoid corrupting Mozilla's PATH */
     strncpy(path, env_path, sizeof(path));

     return inpath(path, file);
}

/*****************************************************************************
 * Delete a mime handler if no command has been found for it.
 *****************************************************************************/
static void filter_previous_handler()
{
     if (prev_handlep && (*prev_handlep) && !(*prev_handlep)->commands)
     {
	  handlep = prev_handlep;
	  free((*prev_handlep)->types->line);
	  free((*prev_handlep)->types);
	  free((*prev_handlep));
	  *prev_handlep = NULL;
	  prev_handlep = NULL;
     }
}

/*****************************************************************************
 * Match a word to an option of the type swallow(name).
 *****************************************************************************/
static char *gobble_winname(char *x, char *option, command_t *c)
{
     char *end;
 
     x += strlen(option);
     while (isspace(*x)) x++;
     if (*x != '(')
     {
	  fprintf(stderr,"Expected '(' after '%s'\n", option);
	  return NULL;
     }
     x++;
     end = strchr(x,')');
     if (end)
     {
	  if (!(c->winname = malloc(end - x + 1)))
	       return NULL;
	  memcpy(c->winname, x, end-x);
	  c->winname[end-x]=0;
	  x = end+1;
     }
     return x;
}

/*****************************************************************************
 * Match a word to a single word option.
 *****************************************************************************/
static int gobble(char *line, char *kw)
{
     return !strncasecmp(line,kw,strlen(kw)) && !isalnum(line[strlen(kw)]);
}

/*****************************************************************************
 * Scan a line for all the possible options.
 *****************************************************************************/
static int parse_options(char **x, command_t *commandp)
{
     static options_t options[] = {
	  { "repeat", 		H_REPEATCOUNT 	},
	  { "loop", 		H_LOOP 		},
	  { "stream", 		H_STREAM 	},
	  { "ignore_errors",	H_IGNORE_ERRORS },
	  { "exits", 		H_DAEMON 	},
	  { "nokill", 		H_DAEMON 	},
	  { "maxaspect",	H_MAXASPECT 	},
	  { "fill",		H_FILL 		},
	  { "noisy",		H_NOISY 	},
	  { "embed",            H_EMBED         },
	  { "noembed",          H_NOEMBED       },
	  { NULL, 		0 		}
     };
     options_t *opt;

     for (opt = options; opt->name; opt++)
     {
	  if (gobble(*x, opt->name))
	  {
	       *x += strlen(opt->name);
	       commandp->flags |= opt->flag;
	       return 1;
	  }
	  if (gobble(*x, "swallow"))
	  {
	       commandp->flags |= H_SWALLOW;
	       if ((*x = gobble_winname(*x, "swallow", commandp)))
		    return 1;
	  }
	  if (gobble(*x, "event_swallow"))
	  {
	       commandp->flags |= H_EVENT_SWALLOW;
	       if ((*x = gobble_winname(*x, "event_swallow", commandp)))
		    return 1;
	  }
	  if (gobble(*x, "controls"))
	  {
	       commandp->flags |= H_CONTROLS | H_SWALLOW | H_FILL;
	       commandp->winname = strdup("mozplugger-controller");
	       *x += 8;
	       return 1;
	  }
     }
     return 0;
}

/*****************************************************************************
 * Read the configuration file into memory.
 *****************************************************************************/
static void read_config(FILE *f)
{
     command_t **commandp = NULL;
     mimetype_t **typep = NULL;
     char buffer[16384];
     char file[128];
     int have_commands = 1;

     D("read_config\n");

     handlep = &first_handle;

     while (!feof(f))
     {
	  fgets(buffer, sizeof(buffer), f);
	  D("::: %s", buffer);

	  if (buffer[0]=='#' || !buffer[0] || buffer[0]=='\n')
	       continue;

	  if (buffer[strlen(buffer)-1] == '\n')
	       buffer[strlen(buffer)-1] = 0;

	  if (isspace(buffer[0]))
	  {
	       char *x = buffer + 1;
	       while (isspace(*x)) x++;
	       if (!*x)
	       {
		    D("Empty line.\n");
		    continue;
	       }

	       /* Command */
	       D("New command\n");

	       if (!(*commandp = (command_t *)malloc(sizeof(command_t))))
		    return;
	       (*commandp)->flags = 0;
	       (*commandp)->cmd = NULL;
	       (*commandp)->winname = NULL;
	       (*commandp)->next = NULL;

	       have_commands++;

	       D("Parsing %s\n", x);

	       while (*x != ':' && *x)
	       {
		    switch(*x) {
		    case ',':
		    case ' ':
		    case '\t':
			 x++;
			 break;
		    default:
			 if (!parse_options(&x, *commandp))
			 {
			      D("Unknown directive: %s\n", x);
			      fprintf(stderr,"Unknown directive: %s\n", x);
			      x += strlen(x);
			 }
		    }
	       }
	       if (*x == ':')
	       {
		    x++;
		    while (isspace(*x)) x++;

		    if (sscanf(x, "if %s", file) != 1
			&& sscanf(x, "%s", file) != 1)
			 continue;

		    if (!find(file))
		    {
			 free((*commandp)->winname);
			 free(*commandp);
			 *commandp = NULL;
			 continue;
		    }

		    (*commandp)->cmd = strdup(x);
	       } else {
		    D("No column? (%s)\n", x);
	       }

	       if (!(*commandp)->cmd)
	       {
		    free((*commandp)->winname);
		    free(*commandp);
		    *commandp = NULL;
		    continue;
	       }

	       commandp=&((*commandp)->next);
	  } else {
	       /* Mime type */

	       if (have_commands)
	       {
		    D("New handle\n");
		    if (commandp) D("Commandp=%p\n", *commandp);

		    filter_previous_handler();

		    if (!(*handlep = (handle_t *)malloc(sizeof(handle_t))))
			 return;

		    (*handlep)->commands = NULL;
		    (*handlep)->types = NULL;
		    (*handlep)->next = NULL;
		    commandp = &((*handlep)->commands);
		    typep = &((*handlep)->types);
		    prev_handlep = handlep;
		    handlep = &((*handlep)->next);
		    have_commands = 0;
	       }

	       D("New mimetype\n");
	       if (!(*typep = (mimetype_t *)malloc(sizeof(mimetype_t))))
		    return;

	       (*typep)->next = NULL;
	       (*typep)->line = strdup(buffer);

	       if (!(*typep)->line)
	       {
		    free(*typep);
		    *typep = NULL;
		    return;
	       }
	       typep = &((*typep)->next);
	  }
     }

     filter_previous_handler();
}

/*****************************************************************************
 * Find the config file or the helper file in function of the function cb.
 *****************************************************************************/
static int find_helper_file(char *basename, int (*cb)(char *,void *data), void *data)
{
     static char fname[8192];
     char *tmp;

     D("find_helper_file '%s'\n", basename);

     if ((tmp = getenv("HOME")))
     {
	  snprintf(fname, sizeof(fname), "%s/.netscape/%s", tmp, basename);
	  if (cb(fname,data)) return 1;
	  snprintf(fname, sizeof(fname), "%s/.mozilla/%s", tmp, basename);
	  if (cb(fname,data)) return 1;
	  snprintf(fname, sizeof(fname), "%s/.opera/%s", tmp, basename);
	  if (cb(fname,data)) return 1;
     }

     if ((tmp = getenv("MOZILLA_HOME")))
     {
	  snprintf(fname, sizeof(fname), "%s/%s", tmp, basename);
	  if (cb(fname, data)) return 1;
     }

     if ((tmp = getenv("OPERA_DIR")))
     {
	  snprintf(fname, sizeof(fname), "%s/%s", tmp, basename);
	  if (cb(fname, data)) return 1;
     }

     snprintf(fname, sizeof(fname), "/etc/%s", basename);
     if (cb(fname, data)) return 1;
     snprintf(fname, sizeof(fname), "/usr/etc/%s", basename);
     if (cb(fname, data)) return 1;
     snprintf(fname, sizeof(fname), "/usr/local/mozilla/%s", basename);
     if (cb(fname, data)) return 1;
     snprintf(fname, sizeof(fname), "/usr/local/netscape/%s", basename);
     if (cb(fname, data)) return 1;
     if (cb(basename, data)) return 1;
  
     return 0;
}

static int read_config_cb(char *fname, void *data)
{
     FILE *f = fopen(fname,"r");
     if (!f) return 0;
     read_config(f);
     fclose(f);
     return 1;
}

static int find_plugger_helper_cb(char *fname, void *data)
{
     struct stat buf;
     if (stat(fname, &buf)) return 0;
     helper_fname = strdup(fname);
     return 1;
}

static int find_plugger_controller_cb(char *fname, void *data)
{
  struct stat buf;
  if(stat(fname, &buf)) return 0;
  controller_fname = strdup(fname);
  return 1;
}

/*****************************************************************************
 * Find configuration file and read it into memory.
 *****************************************************************************/
static void do_read_config(void)
{
     if (first_handle) return;

     D("do_read_config\n");
  
     if (!find_helper_file("mozpluggerrc", read_config_cb, 0))
     {
	  fprintf(stderr, "MozPlugger: Unable to find the mozpluggerrc file.\n");
	  return;
     }

     if (find("mozplugger-helper")) helper_fname = "mozplugger-helper";
     find_helper_file("mozplugger-helper", find_plugger_helper_cb, 0);
     
     if (find("mozplugger-controller")) controller_fname = "mozplugger-controller";
     find_helper_file("mozplugger-controller", find_plugger_controller_cb, 0);

     if (!helper_fname)
     {
	  fprintf(stderr, "MozPlugger: Unable to find the mozplugger helper.\n");
	  return;
     }

     D("do_read_config done\n");
}

/*****************************************************************************
 * Since href's are passed to an app as an argument, just check for ways that 
 * a shell can be tricked into executing a command.
 *****************************************************************************/
static int safeURL(char* url)
{
    int  i = 0; 
    int  len = strlen(url);
    
    if (url[0] == '/')
	 return 0;

    for(i = 0; i < len; i++)
    {
        if (url[i] == '`' || url[i] == ';')
        {
            /* Somebody's trying to do something naughty. */
            return 0;
        }
    }
    return 1;
}

/*****************************************************************************
 * Go through the commands in the config file and find one that fits our needs.
 *****************************************************************************/
static int find_command(NPP instance, int streaming)
{
     handle_t *h;
     mimetype_t *m;
     command_t *c;
     char embedded = (THIS->mode == NP_EMBED);

     D("find_command\n");

     do_read_config();

     D("find_command...\n");

     for (h = first_handle; h; h = h->next)
     {
	  D("commands for this handle at (%p)\n",h->commands);
	  for (m = h->types; m; m = m->next)
	  {
	       char *tmp1 = FEND(m->line,':');
	       char tmp2;
	       int tmp3;
	       D("Checking '%s'\n", m->line);

	       while (tmp1>m->line && isspace(tmp1[-1]))
		    tmp1--;

	       tmp2 = *tmp1;
	       *tmp1 = 0;
	       D("Checking '%s' ?= '%s'\n", m->line, THIS->mimetype);
	       tmp3 = strcasecmp(THIS->mimetype, m->line);
	       *tmp1 = tmp2;
	       if (!tmp3)
	       {
		    D("Match found!\n");
		    break;
	       } else {
		    D("Not same.\n");
	       }
	  }

	  if (m)
	  {
	       for (c = h->commands; c; c = c->next)
	       {
		    D("Checking command: %s\n", c->cmd);
		    if (embedded && (c->flags & H_NOEMBED))
			 continue;
		    if ((!embedded) && (c->flags & H_EMBED))
			 continue;
		    if ((c->flags & H_LOOP) && (THIS->repeats != MAXINT))
			 continue;
		    if ((!!streaming) != (!!(c->flags & H_STREAM)))
			 continue;
		    D("Match found!\n");
		    THIS->flags = c->flags;
		    THIS->command = c->cmd;
		    THIS->winname = c->winname;
		    return 1;
	       }
	  }
     }
     D("No match found\n");
     return 0;
}

/*****************************************************************************
 * Construct a MIME Description string for netscape so that mozilla shall know
 * when to call us back.
 *****************************************************************************/
char *NPP_GetMIMEDescription(void)
{
     char *x,*y;
     handle_t *h;
     mimetype_t *m;
     int size_required;

     D("GetMIMEDescription\n");

     do_read_config();

     size_required = 0;
     for (h = first_handle; h; h = h->next)
	  for (m = h->types; m; m = m->next)
	       size_required += strlen(m->line)+1;

     D("Size required=%d\n", size_required);
     if (!(x = (char *)malloc(size_required+1)))
	  return 0;

     D("Malloc did not fail\n");

     y = x;

     for (h = first_handle; h; h = h->next)
     {
	  D("Foo: %p\n", h->commands);
	  for (m = h->types; m; m = m->next)
	  {
	       memcpy(y,m->line,strlen(m->line));
	       y+=strlen(m->line);
	       *(y++)=';';
	  }
     }
     if (x != y) y--;
     *(y++) = 0;

     D("Getmimedescription done: %s\n", x);

     return x;
}

/*****************************************************************************
 * Let mozilla know things about mozplugger.
 *****************************************************************************/
NPError NPP_GetValue(void *instance, NPPVariable variable, void *value)
{
     NPError err = NPERR_NO_ERROR;

     D("Getvalue %d\n", variable);
  
     switch (variable)
     {
     case NPPVpluginNameString:
	  D("GET Plugin name\n");
	  *((char **)value) = "MozPlugger "
	       VERSION
	       ;
	  break;

     case NPPVpluginDescriptionString:
	  D("GET Plugin description\n");
	  *((char **)value) =
	       "MozPlugger version "
	       VERSION
	       ", written by "
	       "<a href=http://fredrik.hubbe.net/>Fredrik H&uuml;binette</a> "
	       "<a href=mailto:hubbe@hubbe.net>&lt;hubbe@hubbe.net&gt</a> "
	       "and Louis Bavoil "
	       "<a href=mailto:louis@bavoil.net>&lt;louis@bavoil.net&gt</a>.<br>"
	       "For documentation on how to configure mozplugger, "
	       "check the man page. (type <tt>man&nbsp;mozplugger</tt>)"
	       ;
	  break;

     default:
	  err = NPERR_GENERIC_ERROR;
     }
     return err;
}

/*****************************************************************************
 * Convert a string to an integer.
 * The string can be true, false, yes or no.
 *****************************************************************************/
int my_atoi(char *s, int my_true, int my_false)
{
     switch (s[0])
     {
     case 't': case 'T': case 'y': case 'Y':
	  return my_true;
     case 'f': case 'F': case 'n': case 'N':
	  return my_false;
     case '0': case '1': case '2': case '3': case '4':
     case '5': case '6': case '7': case '8': case '9':
	  return atoi(s);
     }
     return -1;
}

/*****************************************************************************
 * Initiate another instance of mozplugger, it is important to know
 * that there might be several instances going at one time.
 *****************************************************************************/
NPError NPP_New(NPMIMEType pluginType,
		 NPP instance,
		 uint16 mode,
		 int16 argc,
		 char* argn[],
		 char* argv[],
		 NPSavedData* saved)
{
     int e;
     int src_attr = -1;

     D("NEW (%s)\n", pluginType);

     if (!instance)
     {
	  D("Invalid instance pointer\n");
	  return NPERR_INVALID_INSTANCE_ERROR;
     }

     if (!pluginType)
     {
	  D("Invalid plugin type\n");
	  return NPERR_INVALID_INSTANCE_ERROR;
     }

     THIS = NPN_MemAlloc(sizeof(data_t));
     if (!THIS) return NPERR_OUT_OF_MEMORY_ERROR;
     memset((char *)THIS, 0, sizeof(*THIS));

     THIS->windata.window = 0;
     THIS->display = NULL;
     THIS->displayname = NULL;
     THIS->pid = -1;
     THIS->repeats = 1;
     THIS->autostart = 1;
     THIS->mode = mode;

     if (!(THIS->mimetype = strdup(pluginType)))
	  return NPERR_OUT_OF_MEMORY_ERROR;

     THIS->num_arguments = argc;
     if (!(THIS->args = (argument_t *)NPN_MemAlloc(sizeof(argument_t) * argc)))
	  return NPERR_OUT_OF_MEMORY_ERROR;
     
     for (e = 0; e < argc; e++)
     {
	  if (!strcasecmp("loop", argn[e]))
	  {
	       THIS->repeats = my_atoi(argv[e], MAXINT, 1);
	  }
	  else if (!strcasecmp("autostart", argn[e]))
	  {
	       THIS->autostart = !!my_atoi(argv[e], 1, 0);
	  }
	  /* get the index of the src attribute */
	  else if (!strcasecmp("src", argn[e]))
	  {
	       src_attr = e;
	  }
	  /* copy the option to put it into the environement later */
	  D("VAR_%s=%s\n", argn[e], argv[e]);
	  if (!(THIS->args[e].var = (char *)malloc(strlen(argn[e]) + 5)))
	       return NPERR_OUT_OF_MEMORY_ERROR;
	  memcpy(THIS->args[e].var, "VAR_", 4);
	  memcpy(THIS->args[e].var+4, argn[e], strlen(argn[e])+1);
	  THIS->args[e].value = argv[e] ? strdup(argv[e]) : NULL;
     }

     /* Special case for quicktime. If there's an href attribute, we 
	want that instead of src but we HAVE to have a src first. */
     for (e = 0; e < argc; e++)
     {
          if (!strcasecmp("href", argn[e]) && src_attr > -1)
          {
	       if (!(THIS->href = strdup(argv[e])))
		    return NPERR_OUT_OF_MEMORY_ERROR;
          }
     }

     D("New finished\n");

     return NPERR_NO_ERROR;
}

/*****************************************************************************
 * Free data, kill processes, it is time for this instance to die.
 *****************************************************************************/
NPError NPP_Destroy(NPP instance, NPSavedData** save)
{
     int e;

     D("Destroy\n");

     if (!instance)
	  return NPERR_INVALID_INSTANCE_ERROR;

     if (THIS)
     {
	  /* Kill the mozplugger process and his sons */
 	  if (THIS->pid > 0)
	       my_kill(-THIS->pid);

	  for (e = 0; e < THIS->num_arguments; e++)
	  {
	       free((char *)THIS->args[e].var);
	       free((char *)THIS->args[e].value);
	  }
	  NPN_MemFree((char *)THIS->args);

	  free(THIS->mimetype);
	  free(THIS->href);

	  NPN_MemFree(THIS);
	  THIS = NULL;
     }

     D("Destroy finished\n");
  
     return NPERR_NO_ERROR;
}

/*****************************************************************************
 * Open a new stream.
 * Each instance can only handle one stream at a time.
 *****************************************************************************/
NPError NPP_NewStream(NPP instance,
		      NPMIMEType type,
		      NPStream *stream, 
		      NPBool seekable,
		      uint16 *stype)
{
     D("NewStream\n");

     if (instance == NULL)
	  return NPERR_INVALID_INSTANCE_ERROR;

     if (THIS->pid != -1)
	  return NPERR_INVALID_INSTANCE_ERROR;

     /* This is a stupid special case and should be coded into
      * mozpluggerc instead... */
     if (!strncasecmp("image/", type, 6) ||
	 !strncasecmp("x-image/", type, 6))
	  THIS->repeats = 1;

     /*  Replace the stream's URL with the URL in THIS->href if it
      *  exists.  Since we're going to be replacing <embed>'s src with
      *  href, we want to make sure that the URL is not going to try
      *  anything dirty. */
     if (THIS->href != NULL && safeURL(THIS->href))
     {
         D("Replacing SRC with HREF... \n");
         stream->url = THIS->href;
     }

     D("Mime type %s\n", type);
     D("Url is %s (seekable=%d)\n", stream->url, seekable);
     
     if (!find_command(instance, 1) && !find_command(instance, 0))
     {
	  NPN_Status(instance, "No appropriate application found !");
	  return NPERR_GENERIC_ERROR;
     }

     if ((THIS->flags & H_STREAM)
	 && strncasecmp(stream->url, "file:", 5)
	 && strncasecmp(stream->url, "imap:", 5)
	 && strncasecmp(stream->url, "mailbox:", 8))
     {
	  *stype = NP_NORMAL;
	  tempfile = 0;

	  if ((THIS->pid = my_fork(instance)) == -1)
	  {
	       NPN_Status(instance, "Streamer: My_Fork failed!");
	       return NPERR_GENERIC_ERROR;
	  }
    
	  if (!THIS->pid)
	  {
	       D("CHILD RUNNING run() [1]\n");
	       run(instance, stream->url);
	       _exit(EX_UNAVAILABLE);
	  }
     } else {
	  *stype = NP_ASFILEONLY;
	  tempfile = 1;
     }
     return NPERR_NO_ERROR;
}

/*****************************************************************************
 * Return 0 to avoid the browser to download data.
 *****************************************************************************/
int32 NPP_WriteReady(NPP instance, NPStream *stream)
{
     D("WriteReady\n");

     if (!instance) return 0;
     if (!tempfile) return 0;
     return 0x7fffffff;
}

/*****************************************************************************
 * Called after NPP_NewStream if *stype = NP_ASFILEONLY.
 *****************************************************************************/
void NPP_StreamAsFile(NPP instance,
		      NPStream *stream,
		      const char* fname)
{
     D("StreamAsFile\n");

     if (!instance || !tempfile || !fname)
	  return;

     if (THIS->pid != -1)
	  return;

     if ((THIS->pid = my_fork(instance)) == -1)
	  return;
    
     if (!THIS->pid)
     {
	  if (!find_command(instance, 1) && !find_command(instance, 0))
	       _exit(EX_UNAVAILABLE);

	  D("CHILD RUNNING run() [2]\n");
	  run(instance, fname);
	  _exit(EX_UNAVAILABLE);
     }
}

/*****************************************************************************
 * Find and resize the embedded window.
 *****************************************************************************/
int resize_window(int flags, Display *display, Window *window, int w, int h)
{
     Window *useme = NULL;
     Window root, parent;
     Window *children = NULL;
     unsigned int num_children;

     if (((flags & H_SWALLOW) || (flags & H_EVENT_SWALLOW)) &&
	 ((flags & H_FILL) || (flags & H_MAXASPECT)))
     {
	  /* tell the child of the stored tree to resize, sadly we don't know
	     the handle of that window, since the helper handles it for us */
	  if (!XQueryTree(display, *window, &root, &parent, &children, &num_children))
	  {
	       D("Querytree failed!!! using current window\n");
	       useme = window;
	  }
	  if (children == NULL) {
	       D("No children!!! using current window\n");
	       useme = window;
	  } else if (num_children > 1) {
	       D("Multiple Children, Not Resizing");
	       XFree(children);
	       return 1;
	  } else {
	       /* use the first child */
	       useme = &children[0];
	  }

	  if (flags & H_FILL) {
	       fill_window(display, *useme, w, h);
	  } else if (flags & H_MAXASPECT) {
	       max_aspect_window(display, *useme, w, h);
	  }

	  XFree(children);
     }

     return 0;
}

/*****************************************************************************
 * Called when initializing or resizing the window.
 *****************************************************************************/
NPError NPP_SetWindow(NPP instance, NPWindow* window)
{
     D("SetWindow\n");

     if (!instance)
	  return NPERR_INVALID_INSTANCE_ERROR;
  
     if (!window)
	  return NPERR_NO_ERROR;
  
     if (!window->window)
	  return NPERR_NO_ERROR;

     if (!window->ws_info)
	  return NPERR_NO_ERROR;

     THIS->display = ((NPSetWindowCallbackStruct *)window->ws_info)->display;
     THIS->displayname = XDisplayName(DisplayString(THIS->display));
     THIS->windata = *window;
     
     if (resize_window(THIS->flags, THIS->display, (Window *)window, window->width, window->height))
	  return NPERR_NO_ERROR;

     if (THIS->pid != -1)
     {
	  D("Sending SIGWINCH to pid (%d)\n", THIS->pid);
	  kill(THIS->pid, SIGWINCH);
     }

     usleep(4000);

     return NPERR_NO_ERROR;
}

/*****************************************************************************
 * Empty functions.
 *****************************************************************************/
NPError NPP_DestroyStream(NPP instance, NPStream *stream, NPError reason)
{
     D("DestroyStream\n");
     return NPERR_NO_ERROR;
}

NPError NPP_Initialize(void)
{
     D("Initialize\n");
     return NPERR_NO_ERROR;
}

jref NPP_GetJavaClass()
{
     D("GetJavaClass\n");
     return NULL;
}

void NPP_Shutdown(void)
{
     D("Shutdown\n");
}

void NPP_Print(NPP instance, NPPrint* printInfo)
{
     D("Print\n");
}

int32 NPP_Write(NPP instance,
		NPStream *stream,
		int32 offset,
		int32 len,
		void *buf)
{
     D("Write\n");
     return len;
}
