# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import time
import signal
import atexit
import logging
from optparse import OptionParser
from os.path import basename, join, abspath, exists
from gettext import gettext as _

from gevent.wsgi import WSGIServer

import active_document as ad
import restful_document as rd
enforce = ad.util.enforce
util = ad.util

from restful_document import env, printf


_LOGFILE_FORMAT = '%(asctime)s %(levelname)s %(name)s: %(message)s'
_application_name = 'sugar-network-server'


def config(name, description, version, homepage, no_exit=False):
    """Configure server.

    Function will override default parameters using configure files and
    command-line arguments.

    :param name:
        id string to use for cases like configure file names
    :param description:
        server description string
    :param version:
        server version
    :param homepage:
        home page for the server project
    :returns:
        not processed arguments

    """
    global _application_name
    _application_name = name

    parser = OptionParser(
            usage='%prog [OPTIONS] [COMMAND]',
            description=description,
            add_help_option=False)
    parser.print_version = lambda: sys.stdout.write('%s\n' % version)

    parser.add_option('-h', '--help',
            help=_('show this help message and exit'),
            action='store_true')
    parser.add_option('-V', '--version',
            help=_('show version number and exit'),
            action='version')

    util.Option.seek('active-document', ad.env)
    util.Option.seek('main', env)
    util.Option.bind(parser, [
            '/etc/%s.conf' % name,
            '~/.config/%s/config' % name,
            ])

    options, args = parser.parse_args()

    if not no_exit and not args and not options.help:
        prog = basename(sys.argv[0])
        print 'Usage: %s [OPTIONS] [COMMAND]' % prog
        print '       %s -h|--help' % prog
        print
        print description
        print _HELP % homepage
        exit(0)

    if options.help:
        parser.print_help()
        print _HELP % homepage
        if not no_exit:
            exit(0)

    util.Option.merge(options)
    return args


def application(documents):
    """Create WSGI application.

    :param documents:
        list of document classes, inherited from
        `restful_document.Document`, to serve
    :returns:
        object that can be used as WSGI `application()` function

    """
    _setup_logging()
    if env.master.value:
        node = ad.Master(documents)
    else:
        node = ad.Node(documents)
    atexit.register(node.close)
    return rd.Router(node)


def fork(documents):
    """Run server in forked process.

    :param documents:
        list of document classes, inherited from
        `restful_document.Document`, to serve
    :returns:
        server process pid

    """
    child_pid = os.fork()
    if child_pid:
        # Let server to start listening port
        time.sleep(1)
        return child_pid

    _setup_logging()
    _forward_stdout()

    node = ad.Master(documents)
    httpd = WSGIServer((env.host.value, env.port.value), rd.Router(node))

    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        pass
    finally:
        httpd.stop()
        node.close()

    # pylint: disable-msg=W0212
    os._exit(0)


def serve_forever(args, documents, **ssl_args):
    """Process commands to control WSGI server.

    :param args:
        list of commands to process
    :param documents:
        list of document classes, inherited from
        `restful_document.Document`, to serve
    :param ssl_args:
        arguments to pass to `ssl.wrap_socket` to enable SSL connections

    """
    _Server(args, documents, ssl_args)


class _Server(object):

    def __init__(self, args, documents, ssl_args):
        self._args = args
        self._documents = documents
        self._ssl_args = ssl_args

        if env.keyfile.value:
            self._ssl_args['keyfile'] = env.keyfile.value
        if env.certfile.value:
            self._ssl_args['certfile'] = env.certfile.value

        command = self._args.pop(0)

        if not env.debug.value:
            logging_level = logging.WARNING
        elif env.debug.value == 1:
            logging_level = logging.INFO
        else:
            logging_level = logging.DEBUG
        logging_format = _LOGFILE_FORMAT
        if env.foreground.value or command not in ['start']:
            logging_format = '-- %s' % logging_format
        logging.basicConfig(level=logging_level, format=logging_format)

        try:
            enforce(hasattr(self, '_cmd_' + command),
                    _('Unknown command "%s"') % command)
            exit(getattr(self, '_cmd_' + command)() or 0)
        except Exception:
            printf.exception(_('Aborted %s due to error'), _application_name)
            exit(1)
        finally:
            printf.flush_hints()

    def _cmd_config(self):
        if self._args:
            opt = self._args.pop(0)
            enforce(opt in util.Option.items, _('Unknown option "%s"'), opt)
            exit(0 if bool(util.Option.items[opt].value) else 1)
        else:
            print '\n'.join(util.Option.export())

    def _cmd_start(self):
        pidfile, pid = self._check_for_instance()
        if pid:
            logging.warning(_('%s is already run with pid %s'),
                    _application_name, pid)
            return 1
        if env.foreground.value:
            self._launch()
        else:
            if not exists(env.logdir.value):
                os.makedirs(env.logdir.value)
            enforce(os.access(env.logdir.value, os.W_OK),
                    _('No write access to %s'), env.logdir.value)
            if not exists(env.rundir.value):
                os.makedirs(env.rundir.value)
            enforce(os.access(env.rundir.value, os.W_OK),
                    _('No write access to %s'), env.rundir.value)
            _forward_stdout()
            self._daemonize(pidfile)
        return 0

    def _cmd_stop(self):
        __, pid = self._check_for_instance()
        if pid:
            os.kill(pid, signal.SIGTERM)
            return 0
        else:
            logging.warning(_('%s is not run'), _application_name)
            return 1

    def _cmd_status(self):
        __, pid = self._check_for_instance()
        if pid:
            printf.info(_('%s started'), _application_name)
            return 0
        else:
            printf.info(_('%s stopped'), _application_name)
            return 1

    def _cmd_reload(self):
        __, pid = self._check_for_instance()
        if pid:
            os.kill(pid, signal.SIGHUP)
            logging.info(_('Reload %s process'), _application_name)

    def _launch(self):
        logging.info(_('Start %s on %s:%s'),
                _application_name, env.host.value, env.port.value)

        if env.master.value:
            node = ad.Master(self._documents)
        else:
            node = ad.Node(self._documents)

        httpd = WSGIServer((env.host.value, env.port.value),
                rd.Router(node), **self._ssl_args)

        def sigterm_cb(signum, frame):
            logging.info(_('Got signal %s, will stop %s'),
                    signum, _application_name)
            httpd.stop()

        def sighup_cb(signum, frame):
            logging.info(_('Reload %s on SIGHUP signal'), _application_name)
            _forward_stdout()

        signal.signal(signal.SIGINT, sigterm_cb)
        signal.signal(signal.SIGTERM, sigterm_cb)
        signal.signal(signal.SIGHUP, sighup_cb)

        try:
            httpd.serve_forever()
        finally:
            httpd.stop()
            node.close()

    def _check_for_instance(self):
        pid = None
        pidfile = join(env.rundir.value, '%s.pid' % _application_name)
        if exists(pidfile):
            try:
                pid = int(file(pidfile).read().strip())
                os.getpgid(pid)
            except (ValueError, OSError):
                pid = None
        return pidfile, pid

    def _daemonize(self, pid_path):
        pid_path = abspath(pid_path)

        if os.fork() > 0:
            # Exit parent of the first child
            return

        # Decouple from parent environment
        os.chdir(os.sep)
        os.setsid()

        if os.fork() > 0:
            # Exit from second parent
            # pylint: disable-msg=W0212
            os._exit(0)

        # Redirect standard file descriptors
        if not sys.stdin.closed:
            stdin = file('/dev/null')
            os.dup2(stdin.fileno(), sys.stdin.fileno())

        pidfile = file(pid_path, 'w')
        pidfile.write(str(os.getpid()))
        pidfile.close()
        atexit.register(lambda: os.remove(pid_path))

        try:
            self._launch()
            status = 0
        except Exception:
            logging.exception(_('Abort sugar-server due to error'))
            status = 1

        exit(status)


def _setup_logging():
    if not env.debug.value:
        logging_level = logging.WARNING
    elif env.debug.value == 1:
        logging_level = logging.INFO
    else:
        logging_level = logging.DEBUG
    logging.basicConfig(level=logging_level, format=_LOGFILE_FORMAT)


def _forward_stdout():
    if not exists(env.logdir.value):
        os.makedirs(env.logdir.value)
    log_path = abspath(join(env.logdir.value, '%s.log' % _application_name))
    sys.stdout.flush()
    sys.stderr.flush()
    logfile = file(log_path, 'a+')
    os.dup2(logfile.fileno(), sys.stdout.fileno())
    os.dup2(logfile.fileno(), sys.stderr.fileno())
    logfile.close()


_HELP = """
Commands:
  start                 start in daemon mode
  stop                  stop daemon
  status                check for launched daemon
  reload                reopen log files in daemon mode
  config                output current configuration

See %s."""
