#!/usr/bin/env python
# sugar-lint: disable

import json
import time
import hashlib

from __init__ import tests
from tests import Resource

import active_document as ad
import restful_document as rd


class Vote(ad.AggregatorProperty):

    @property
    def value(self):
        return 'vote'


class Document(rd.Document):

    @ad.active_property(slot=1, prefix='A', full_text=True)
    def term(self, value):
        return value

    @ad.active_property(ad.StoredProperty)
    def stored(self, value):
        return value

    @ad.active_property(ad.BlobProperty)
    def blob(self, value):
        return value

    @ad.active_property(ad.CounterProperty, slot=2)
    def counter(self, value):
        return value

    @ad.active_property(Vote, counter='counter')
    def vote(self, value):
        return value


class DocumentTest(tests.Test):

    def test_Walkthrough(self):
        self.httpd(8000, [rd.User, Document])
        rest = Resource('http://localhost:8000')

        props = {
            'nickname': 'foo',
            'pubkey': tests.VALID_DSS_PUBKEY,
            'color': '',
            'machine_sn': 'machine_sn',
            'machine_uuid': 'machine_uuid',
            }
        rest.post('/user', payload=json.dumps(props), headers=tests.creds())

        guid_1 = json.loads(
                rest.post('/document', headers=tests.creds(),
                    payload=json.dumps({
                        'term': 'term',
                        'stored': 'stored',
                        })
                    ).body_string()
                )['guid']

        reply = json.loads(rest.get('/document/' + guid_1, headers=tests.creds()).body_string())
        del reply['ctime']
        del reply['mtime']
        self.assertEqual(
                {
                    'stored': 'stored',
                    'vote': False,
                    'term': 'term',
                    'counter': 0,
                    'guid': guid_1,
                    'layers': ['public'],
                    },
                reply)

        guid_2 = json.loads(
                rest.post('/document', headers=tests.creds(),
                    payload=json.dumps({
                        'term': 'term2',
                        'stored': 'stored2',
                        })
                    ).body_string()
                )['guid']

        reply = json.loads(rest.get('/document/' + guid_2, headers=tests.creds()).body_string())
        del reply['ctime']
        del reply['mtime']
        self.assertEqual(
                {
                    'stored': 'stored2',
                    'vote': False,
                    'term': 'term2',
                    'counter': 0,
                    'guid': guid_2,
                    'layers': ['public'],
                    },
                reply)

        reply = json.loads(rest.get('/document', reply='guid,stored,term,vote,counter', headers=tests.creds()).body_string())
        self.assertEqual(2, reply['total'])
        self.assertEqual(
                sorted([
                    {'guid': guid_1, 'stored': 'stored', 'term': 'term', 'vote': False, 'counter': 0},
                    {'guid': guid_2, 'stored': 'stored2', 'term': 'term2', 'vote': False, 'counter': 0},
                    ]),
                sorted(reply['result']))

        rest.put('/document/' + guid_2, headers=tests.creds(),
                payload=json.dumps({
                    'vote': True,
                    'stored': 'stored3',
                    'term': 'term3',
                    })
                )

        # Let server process commit and change `counter` property
        time.sleep(3)

        reply = json.loads(rest.get('/document/' + guid_2, headers=tests.creds()).body_string())
        del reply['ctime']
        del reply['mtime']
        self.assertEqual(
                {
                    'stored': 'stored3',
                    'vote': True,
                    'term': 'term3',
                    'counter': 1,
                    'guid': guid_2,
                    'layers': ['public'],
                    },
                reply)

        self.assertEqual(
                {'total': 2,
                    'result': sorted([
                        {'guid': guid_1, 'stored': 'stored', 'term': 'term', 'vote': False, 'counter': 0},
                        {'guid': guid_2, 'stored': 'stored3', 'term': 'term3', 'vote': True, 'counter': 1},
                        ])},
                json.loads(rest.get('/document', reply='guid,stored,term,vote,counter', headers=tests.creds()).body_string()))

        rest.delete('/document/' + guid_1, headers=tests.creds())

        # Let server process commit and change `counter` property
        time.sleep(3)

        self.assertEqual(
                {'total': 1,
                    'result': sorted([
                        {'guid': guid_2, 'stored': 'stored3', 'term': 'term3', 'vote': True, 'counter': 1},
                        ])},
                json.loads(rest.get('/document', reply='guid,stored,term,vote,counter', headers=tests.creds()).body_string()))

        self.assertEqual(
                'term3',
                json.loads(rest.get('/document/' + guid_2 + '/term', headers=tests.creds()).body_string()))
        rest.put('/document/' + guid_2 + '/term', payload=json.dumps('term4'), headers=tests.creds())
        self.assertEqual(
                'term4',
                json.loads(rest.get('/document/' + guid_2 + '/term', headers=tests.creds()).body_string()))

        payload = 'blob'
        headers = tests.creds()
        headers['Content-Type'] = 'application/octet-stream'
        rest.put('/document/' + guid_2 + '/blob', payload=payload, headers=headers)
        self.assertEqual(
                payload,
                rest.get('/document/' + guid_2 + '/blob', headers=tests.creds()).body_string())
        self.assertEqual(
                {'size': len(payload), 'sha1sum': hashlib.sha1(payload).hexdigest()},
                json.loads(rest.get('/document/' + guid_2 + '/blob', headers=tests.creds(), cmd='stat').body_string()))

        rest.delete('/document/' + guid_2, headers=tests.creds())

        # Let server process commit and change `counter` property
        time.sleep(3)

        self.assertEqual(
                {'total': 0,
                    'result': sorted([])},
                json.loads(rest.get('/document', reply='guid,stored,term,vote,counter', headers=tests.creds()).body_string()))

    def test_ServerCrash(self):
        self.httpd(8000, [rd.User, Document])
        rest = Resource('http://localhost:8000')

        props = {
            'nickname': 'foo',
            'color': '',
            'machine_sn': 'machine_sn',
            'machine_uuid': 'machine_uuid',
            'pubkey': tests.VALID_DSS_PUBKEY,
            }
        rest.post('/user', payload=json.dumps(props), headers=tests.creds())

        guid_1 = json.loads(
                rest.post('/document', headers=tests.creds(),
                    payload=json.dumps({
                        'term': 'term',
                        'stored': 'stored',
                        })
                    ).body_string()
                )['guid']
        guid_2 = json.loads(
                rest.post('/document', headers=tests.creds(),
                    payload=json.dumps({
                        'term': 'term2',
                        'stored': 'stored2',
                        'vote': '1',
                        })
                    ).body_string()
                )['guid']


        reply = json.loads(rest.get('/document', reply='guid,stored,term,vote', headers=tests.creds()).body_string())
        self.assertEqual(2, reply['total'])
        self.assertEqual(
                sorted([{'guid': guid_1, 'stored': 'stored', 'term': 'term', 'vote': False},
                        {'guid': guid_2, 'stored': 'stored2', 'term': 'term2', 'vote': True},
                        ]),
                sorted(reply['result']))

        self.httpdown(8000)
        self.httpd(8001, [rd.User, Document])
        rest = Resource('http://localhost:8001')

        reply = json.loads(rest.get('/document', reply='guid,stored,term,vote,counter', headers=tests.creds()).body_string())
        self.assertEqual(2, reply['total'])
        self.assertEqual(
                sorted([{'guid': guid_1, 'stored': 'stored', 'term': 'term', 'vote': False, 'counter': 0},
                        {'guid': guid_2, 'stored': 'stored2', 'term': 'term2', 'vote': True, 'counter': 1},
                        ]),
                sorted(reply['result']))


if __name__ == '__main__':
    tests.main()
