# Copyright (C) 2012-2013 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from os.path import isabs
from gettext import dgettext
from os.path import exists

import gtk

from sugar.graphics import style
from sugar.graphics.icon import Icon

from jarabe import config
from jarabe.model import bundleregistry
from jarabe.plugins.sn import connection


_ = lambda x: dgettext('sugar-plugin-sn', x)


class ReportFrame(gtk.VBox):

    def __init__(self, window, event):
        gtk.VBox.__init__(self)
        self._event = event
        no_upload = (event.get('exception') == 'ServiceUnavailable')

        bg = gtk.EventBox()
        bg.modify_bg(gtk.STATE_NORMAL, style.COLOR_WHITE.get_gdk_color())
        self.pack_start(bg)

        canvas = gtk.VBox()
        canvas.set_border_width(style.DEFAULT_SPACING)
        canvas.props.spacing = style.DEFAULT_SPACING
        bg.add(canvas)

        label = gtk.Label()
        label.props.use_markup = True
        bundle = bundleregistry.get_registry().get_bundle(event['guid'])
        if bundle is not None:
            activity_name = bundle.get_name()
        else:
            activity_name = event['guid']
        label.props.label = _('<b>%s</b> failed to start.') % activity_name
        canvas.pack_start(label, False)

        if no_upload:
            label = gtk.Label()
            label.props.label = \
                    _('To proceed further, connect to Sugar Network.')
            canvas.pack_start(label, False)

        buttons_box = gtk.HBox()
        buttons_box.props.spacing = style.DEFAULT_SPACING
        box = gtk.Alignment(xalign=0.5)
        box.add(buttons_box)
        canvas.pack_start(box, False)

        cancel_button = gtk.Button(stock=gtk.STOCK_STOP)
        cancel_button.connect('clicked', lambda button: window.destroy())
        buttons_box.pack_start(cancel_button, False)

        if not no_upload:
            submit_button = gtk.Button(stock=gtk.STOCK_APPLY)
            submit_button.props.label = _('Submit report')
            submit_button.props.image = Icon(icon_name='transfer-to')
            submit_button.connect('clicked', lambda button: self._submit())
            buttons_box.pack_start(submit_button, False)

        self.show_all()

        window.get_child().pack_end(self, expand=True)
        window.connect('key-press-event', self.__key_press_event_cb)

    def __key_press_event_cb(self, window, event):
        if event.keyval == gtk.keysyms.Escape:
            window.destroy()
        elif event.keyval == gtk.keysyms.Return and \
                event.state & gtk.gdk.CONTROL_MASK:
            self._submit()

    def _submit(self):
        submit(self._event)
        self.get_toplevel().destroy()


def submit(event):
    props = {'context': event['guid'],
             'error': event['error'],
             'environ': _get_environ(),
             'logs': event.get('logs') or [],
             }
    if 'solution' in event:
        props['environ']['solution'] = event['solution']
        impl = event['solution'][0]
        if not isabs(impl['guid']):
            props['implementation'] = impl['guid']
        props['version'] = impl['version']
    connection().post(['report'], props, cmd='submit', spawn=True)


def _get_environ():
    import platform
    from sugar_network.toolkit import lsb_release

    return {'lsb_distributor_id': lsb_release.distributor_id(),
            'lsb_release': lsb_release.release(),
            'os': platform.linux_distribution(),
            'uname': platform.uname(),
            'python': platform.python_version_tuple(),
            'sugar': config.version,
            }
