/* implement copying and pasting in Linux virtual consoles */
/* Andrew Haylett */
/* [# Edit 18, Date 14-Sep-94, Module selection.c #] */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <termios.h>
#include <fcntl.h>
#include <signal.h>

#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <sys/kd.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <linux/keyboard.h>

#include "mouse.h"

#define DEF_MACCEL		2
#define DEF_MBAUD		1200
#define DEF_MDELTA		25
#define DEF_MDEV		"/dev/mouse"
#define DEF_MTOGGLE		0
#define DEF_MSAMPLE		100
#define DEF_MTYPE		P_MS
#define DEF_MSLACK		-1

#define DEF_XSCALE		10
#define DEF_YSCALE		12
#define DEF_INTERVAL		250 /* msec */
#define DEF_INWORD		"0-9A-Za-z_"
#define DEF_COPY_BUTTON		"l"
#define DEF_PASTE_BUTTON	"r"
#define DEF_EXTEND_BUTTON	"c-l"

#define KBD_ALT			(0x10 << KG_ALT)
#define KBD_CTRL		(0x10 << KG_CTRL)
#define KBD_SHIFT		(0x10 << KG_SHIFT)

#define CONSOLE			"/dev/tty0"
#define ALT_CONSOLE		"/dev/console"
#define PIDFILE			"/tmp/selection.pid"

typedef enum { letter = 0, word = 1, line = 2, track = 3, unset = 4 } sel_mode;
typedef struct { int x, y; } Coord;

static int maccel = DEF_MACCEL;		/* acceleration factor */
static int mbaud = DEF_MBAUD;		/* mouse device baud rate */
static int mdelta = DEF_MDELTA;		/* acceleration threshold */
static char *mdev = DEF_MDEV;		/* mouse device */
static int mtoggle = DEF_MTOGGLE;	/* modem lines to toggle */
static int msample = DEF_MSAMPLE;	/* sample rate for Logitech mice */
static mouse_type mtype = DEF_MTYPE;	/* mouse type */
static int mslack = DEF_MSLACK;		/* amount of slack for wraparound */

static char *inword = DEF_INWORD;
static int xscale = DEF_XSCALE;
static int yscale = DEF_YSCALE;
static long click_interval = DEF_INTERVAL;
static unsigned char copy_button;
static unsigned char paste_button;
static unsigned char extend_button;
static int test_mode = 0;

static unsigned char shift_state;
static unsigned char rep_shift_state;
static char *progname;
static const char *version = "1.8 (14th September 1994)";
static int cfd;
static struct winsize win;

/* maps button bits MS_ to numbers: left=0, middle=1, right=2 */
static int but2no[8] = { 0, 2, 1, 2, 0, 1, 0, 1 };

static void usage(void);
static void parse_params(const int argc, char *argv[]);
static int open_vc(const int mode);
static void set_sel(const Coord start, const Coord end, const sel_mode mode);
static void paste(void);
static long interval(const struct timeval *t1, const struct timeval *t2);
static int check_graphics_mode(void);
static unsigned char get_shift_state(void);
static int mouse_report_mode(void);
static void sig_handler(int sig);
static int kill_selection(void);
static void load_inword(const char *charset);
static unsigned char parse_button(char *arg);
static inline int cdiff(const Coord c1, const Coord c2);
static void run_test(void);

int
main(int argc, char *argv[])
{
    struct ms_event ev;
    struct timeval tv1, tv2;
    Coord curr, prev, start, end, estart, eend;
    int dir = 0;
    int report_mouse = 0;
    sel_mode mode = unset;
    char subfunc = 7;
    long kd_mode;
    FILE *fp;
    
    progname = (rindex(argv[0], '/')) ? rindex(argv[0], '/') + 1 : argv[0];

    copy_button = parse_button(DEF_COPY_BUTTON);
    paste_button = parse_button(DEF_PASTE_BUTTON);
    extend_button = parse_button(DEF_EXTEND_BUTTON);

    parse_params(argc, argv);
    if (copy_button == extend_button || extend_button == paste_button ||
	    copy_button == paste_button)
    {
	fprintf(stderr,
	    "%s: Copy, extend and paste buttons must have unique values.\n",
	    progname);
	exit(1);
    }
    load_inword(inword);

    cfd = open_vc(O_RDWR);
    ioctl(cfd, TIOCGWINSZ, &win);
    if (! win.ws_col || ! win.ws_row)
    {
    	fprintf(stderr, "%s: Zero screen dimension, assuming 80x25.\n",
	    progname);
    	win.ws_col = 80;
    	win.ws_row = 25;
    }

    if (test_mode)
    	run_test();	/* never returns */

    if (ioctl(cfd, KDGETMODE, &kd_mode) < 0 || kd_mode != KD_TEXT)
    {
	fprintf(stderr, "%s: Must be run from a virtual console.\n", progname);
	exit(1);
    }

    if (ioctl(cfd, TIOCLINUX, &subfunc) < 0)
    {
	fprintf(stderr, "%s: You must rebuild the kernel with selection support.\n",
	    progname);
	exit(1);
    }

    if (!kill_selection())
    	sleep(1);

    if (fork())
    	exit(0);
    setsid();

    if ((fp = fopen(PIDFILE, "w")) != (FILE *)NULL) {
	fprintf(fp, "%d\n", getpid());
	fclose(fp);
    }
    else
    {
	fprintf(stderr, "%s: Could not write PID file `%s', terminating.\n",
	    progname, PIDFILE);
	exit(1);
    }

    signal(SIGHUP, sig_handler);
    signal(SIGINT, sig_handler);
    signal(SIGTERM, sig_handler);

restart:
    prev.x = 0; prev.y = 0;
    if (mode != unset)
    	mode = letter;

    if (ms_init(maccel, mbaud, mdelta, mdev, mtoggle, msample, mtype, mslack,
		win.ws_col * xscale - 1, win.ws_row * yscale - 1))
	exit(1);

    gettimeofday(&tv1, (struct timezone *)NULL);

    while (get_ms_event(&ev) == 0)
    {
	curr.x = ev.ev_x / xscale + 1;
	curr.y = ev.ev_y / yscale + 1;
	if (! cdiff(curr, prev) && ev.ev_code != MS_BUTDOWN &&
		ev.ev_code != MS_BUTUP)
	    continue;
	close(cfd);
	cfd = open_vc(O_RDWR);
	prev = curr;
	if (check_graphics_mode())
	    goto restart;
	if (ev.ev_code == MS_BUTDOWN)
	{
	    report_mouse = mouse_report_mode();
	    shift_state = get_shift_state();
	    rep_shift_state = (report_mouse) ? KBD_ALT : 0;
	}
	if (report_mouse && ! shift_state)
	{
	    set_sel(curr, curr, track);
	    if (ev.ev_code == MS_BUTDOWN)
		set_sel(curr, curr, 16 + but2no[(int)ev.ev_butstate]);
	    else if (ev.ev_code == MS_BUTUP && report_mouse == 2)
		set_sel(curr, curr, 19);
	}
	else if (ev.ev_code == MS_MOVE)
	    set_sel(curr, curr, track);
	else if ((ev.ev_butstate | shift_state) ==
		(copy_button | rep_shift_state))
	{
	    if (ev.ev_code == MS_BUTDOWN)
	    {
		gettimeofday(&tv2, (struct timezone *)NULL);
		if (interval(&tv1, &tv2) < click_interval && mode == letter)
		    mode = word;
		else if (interval(&tv1, &tv2) < click_interval && mode == word)
		    mode = line;
		else
		    mode = letter;
		tv1 = tv2;
		start = end = curr;
	    }
	    else
		end = curr;
	    set_sel(start, end, mode);
	    set_sel(curr, curr, track);
	}
	else if ((ev.ev_butstate | shift_state) ==
		(extend_button | rep_shift_state) && mode != unset)
	{
	    if (ev.ev_code == MS_BUTDOWN)
	    {
		int d1, d2;
		if (cdiff(start, end) > 0)
		{
		    Coord t;
		    t = start; start = end; end = t;
		}
		estart = start;
		eend = end;
		d1 = cdiff(curr, estart);
		d2 = cdiff(curr, eend);
		if (d1 >= 0 && d2 <= 0)
		    dir = (d1 < (-d2)) ? 0 : 1;
	    }
	    if (cdiff(curr, estart) < 0)
		dir = 0;
	    else if (cdiff(curr, eend) > 0)
		dir = 1;
	    if (dir == 0)
	    {
		start = curr; end = eend;
	    }
	    else
	    {
		start = estart; end = curr;
	    }
	    set_sel(start, end, mode);
	    set_sel(curr, curr, track);
	}
	else if ((ev.ev_butstate | shift_state) ==
		(paste_button | rep_shift_state)
		&& ev.ev_code == MS_BUTDOWN && mode != unset)
	    paste();
    }
    exit(1);
}

static void
usage()
{
    fprintf(stderr, "\
Usage: %s [-a accel] [-b baud-rate] [-c [a-][c-][s-]l|m|r] [-d delta]
       [-D] [-e [a-][c-][s-]l|m|r] [-i interval] [-k] [-l|L charset]
       [-m mouse-device] [-o modem-line] [-p [a-][c-][s-]l|m|r] [-s sample-rate]
       [-t mouse-type] [-v] [-w slack] [-x scale] [-y scale]
    -a accel             sets the acceleration (default %d)
    -b baud-rate         sets the baud rate (default %d)
    -c [a-][c-][s-]l|m|r sets the copy button (default `%s')
    -d delta             sets the delta value (default %d)
    -D                   enter test/debugging mode (default off)
    -e [a-][c-][s-]l|m|r sets the extend button (default `%s')
    -i interval          click interval for word/line sel. (default %d ms)
    -k                   kill running selection process and exit
    -l|L charset         defines character set for word selection
                          (default `%s'). -L exits after loading.
    -m mouse-device      sets mouse device (default `%s')
    -o modem-line        toggle modem line on initialization
                         modem-line = dtr | rts | both
    -p [a-][c-][s-]l|m|r sets the paste button (default `%s')
    -s sample-rate       sets the sample rate (default %d)
    -t mouse-type        sets mouse type (default `ms')
                         Microsoft = `ms', Mouse Systems Corp = `msc',
                         MM Series = `mm', Logitech = `logi', BusMouse = `bm',
                         MSC 3-bytes = `sun', PS/2 = `ps2'
    -v                   print version number and exit
    -w slack             turns on wrap-around and specifies slack (default off)
    -x scale             scale factor for horizontal mouse movement (default %d)
    -y scale             scale factor for vertical mouse movement (default %d)
",
	progname, DEF_MACCEL, DEF_MBAUD, DEF_COPY_BUTTON, DEF_MDELTA,
	DEF_EXTEND_BUTTON, DEF_INTERVAL, DEF_INWORD, DEF_MDEV,
	DEF_PASTE_BUTTON, DEF_MSAMPLE, DEF_XSCALE, DEF_YSCALE);
    exit(1);
}

extern int optind;
extern char *optarg;

static void
parse_params(const int argc, char *argv[])
{
    int opt;

    while ((opt = getopt(argc, argv, "a:b:c:d:De:i:kl:L:m:o:p:s:t:vw:x:y:")) != -1)
    {
	switch (opt)
	{
	    case 'a':
		maccel = atoi(optarg);
		if (maccel < 2)
		{
		    fprintf(stderr,
			"%s: specify value greater than 1 for -a.\n", progname);
		    exit(1);
		}
		break;
	    case 'b':
		mbaud = atoi(optarg);
		break;
	    case 'c':
	        copy_button = parse_button(optarg);
		break;
	    case 'd':
		mdelta = atoi(optarg);
		if (mdelta < 2)
		{
		    fprintf(stderr,
			"%s: specify value greater than 1 for -d.\n", progname);
		    exit(1);
		}
		break;
	    case 'D':
	    	test_mode = 1;
	    	break;
	    case 'e':
	        extend_button = parse_button(optarg);
		break;
	    case 'i':
		click_interval = atoi(optarg);
		if (click_interval < 1)
		{
		    fprintf(stderr,
			"%s: specify value greater than 0 for -i.\n", progname);
		    exit(1);
		}
		break;
	    case 'k':
		if (kill_selection())
		{
		    fprintf(stderr, "%s: no process to kill.\n", progname);
		    exit(1);
		}
		else
		    exit(0);
		break;
	    case 'l': /* load look-up-table */
		inword = optarg;
		break;
	    case 'L': /* load look-up-table and exit */
		load_inword(optarg);
		exit(0);
		break;
	    case 'm':
		mdev = optarg;
		break;
	    case 'o':
	    	if (! strcmp(optarg, "dtr"))
	    	    mtoggle = TIOCM_DTR;
	    	else if (! strcmp(optarg, "rts"))
	    	    mtoggle = TIOCM_RTS;
	    	else if (! strcmp(optarg, "both"))
	    	    mtoggle = (TIOCM_DTR | TIOCM_RTS);
		else
		{
		    fprintf(stderr, "%s: Invalid value for -o.\n", progname);
		    usage();
		}
		break;
	    case 'p':
	        paste_button = parse_button(optarg);
		break;
	    case 's':
		msample = atoi(optarg);
		break;
	    case 't':
		if (! strcmp(optarg, "ms"))
		    mtype = P_MS;
		else if (! strcmp(optarg, "sun"))
		    mtype = P_SUN;
		else if (! strcmp(optarg, "msc"))
		    mtype = P_MSC;
		else if (! strcmp(optarg, "mm"))
		    mtype = P_MM;
		else if (! strcmp(optarg, "logi"))
		    mtype = P_LOGI;
		else if (! strcmp(optarg, "bm"))
		    mtype = P_BM;
		else if (! strcmp(optarg, "ps2"))
		    mtype = P_PS2;
		else
		{
		    fprintf(stderr, "%s: Invalid type for -t.\n", progname);
		    usage();
		}
		break;
	    case 'v':
		printf("%s version %s\n", progname, version);
		exit(0);
		break;
	    case 'w':
	    	mslack = atoi(optarg);
	    	if (mslack < 0)
		{
		    fprintf(stderr,
			"%s: specify non-negative value for -w.\n", progname);
		    exit(1);
		}
	    	break;
	    case 'x':
		xscale = atoi(optarg);
		if (xscale < 1)
		{
		    fprintf(stderr,
			"%s: specify value greater than 0 for -x.\n", progname);
		    exit(1);
		}
		break;
	    case 'y':
		yscale = atoi(optarg);
		if (yscale < 1)
		{
		    fprintf(stderr,
			"%s: specify value greater than 0 for -y.\n", progname);
		    exit(1);
		}
		break;
	    default:
		usage();
		break;
	}
    }
}

/* We have to keep opening and closing the VC because (a) /dev/tty0 (current
    VC) is fixed after the open(), rather than being re-evaluated at each
    write(), and (b) because we seem to lose our grip on /dev/tty? after
    someone logs in if this is run from /etc/rc. */

static int
open_vc(const int mode)
{
    int fd;
    static char *console = CONSOLE;

    if ((fd = open(console, mode)) < 0)
    {
	console = ALT_CONSOLE;
    	if ((fd = open(console, mode)) < 0)
    	{
	    perror("selection: could not open /dev/tty0 or /dev/console");
	    exit(1);
	}
    }
    return fd;
}

/* mark selected text on screen. */
static void
set_sel(const Coord start, const Coord end, const sel_mode mode)
{
    unsigned char buf[sizeof(char) + 5 * sizeof(short)];
    unsigned short *arg = (unsigned short *)(buf + 1);

    buf[0] = 2;	/* subfunction */

    arg[0] = start.x;
    arg[1] = start.y;
    arg[2] = end.x;
    arg[3] = end.y;
    arg[4] = mode;

    if (ioctl(cfd, TIOCLINUX, buf) < 0)
    {
	perror("selection: ioctl(TIOCLINUX, 2)");
	exit(1);
    }
}

/* paste contents of selection buffer into console. */
static void
paste(void)
{
    char subfunc = 3;

    if (ioctl(cfd, TIOCLINUX, &subfunc) < 0)
    {
	perror("selection: ioctl(TIOCLINUX, 3)");
	exit(1);
    }
}

/* evaluate interval between times. */
static long
interval(const struct timeval *t1, const struct timeval *t2)
{
    return (t2->tv_sec  - t1->tv_sec)  * 1000
         + (t2->tv_usec - t1->tv_usec) / 1000;
}

/* Check whether console is in graphics mode; if so, wait until it isn't. */
static int
check_graphics_mode(void)
{
    int ch = 0;
    long kd_mode;

    do
    {
	if (ioctl(cfd, KDGETMODE, &kd_mode) < 0)
	{
	    perror("selection: ioctl(KDGETMODE)");
	    exit(1);
	}
	if (kd_mode != KD_TEXT)
	{
	    ++ch;
	    sleep(2);
	    close(cfd);
	    cfd = open_vc(O_RDWR);
	}
    } while (kd_mode != KD_TEXT);
    return (ch > 0);
}

static unsigned char
get_shift_state(void)
{
    char subfunc = 6;

    if (ioctl(cfd, TIOCLINUX, &subfunc) < 0)
    {
	perror("selection: ioctl(TIOCLINUX, 6)");
	exit(1);
    } else
	return subfunc << 4;
}

static int
mouse_report_mode(void)
{
    char subfunc = 7;

    if (ioctl(cfd, TIOCLINUX, &subfunc) < 0)
    {
	perror("selection: ioctl(TIOCLINUX, 7)");
	exit(1);
    } else
	return (int)subfunc;
}

static void
sig_handler(int sig)
{
    unlink(PIDFILE);
    exit(0);
}

static int
kill_selection(void)
{
    FILE *fp;
    int pid;

    if ((fp = fopen(PIDFILE, "r")) != (FILE *)NULL)
    {
	if (fscanf(fp, "%d", &pid) == 1)
	{
	    kill(pid, SIGHUP);
	    return 0;
	}
	fclose(fp);
    }
    return 1;
}

static void
load_inword(const char *charset)
{
    int i, c, fd;
    static unsigned long long_array[5]={
	0x05050505, /* ugly, but preserves alignment */
	0x00000000, /* control chars     */
	0x00000000, /* digits            */
	0x00000000, /* uppercase and '_' */
	0x00000000  /* lowercase         */
    };

#define inwordLut (long_array + 1)

    inwordLut[0] = inwordLut[1] = inwordLut[2] = inwordLut[3] = 0;
    for (i = 0; (c = charset[i]); i++)
    {
	if (c == '\\')    /* use the next, if any */
	{
	    if (charset[i + 1] == '\0')
		break;
	    c = charset[++i];
	}
	else if (c == '-' && i != 0 && charset[i + 1] != '\0') /* range */
	{
	    for (c = charset[i - 1] + 1; c < charset[i + 1]; c++)
		inwordLut[(c >> 5) & 3] |= 1 << (c & 0x1F);
	} else
	    inwordLut[(c >> 5) & 3] |= 1 << (c & 0x1F);
    }
    fd = open_vc(O_WRONLY);
    if (ioctl(fd, TIOCLINUX, &long_array) < 0)
    {
	perror("selection: ioctl(TIOCLINUX, 5)");
	exit(1);
    }
    close(fd);
}

static unsigned char
parse_button(char *arg)
{
    char button = 0, c1, c2;

    while (*arg)
    {
	c1 = *arg; c2 = *(arg + 1);
	if (isupper(c1))
	    c1 = tolower(c1);
	if (c1 == 'a' && c2 == '-')
	    button |= KBD_ALT;
	else if (c1 == 'c' && c2 == '-')
	    button |= KBD_CTRL;
	else if (c1 == 's' && c2 == '-')
	    button |= KBD_SHIFT;
	else if (c1 == 'l' && ! c2)
	    button |= MS_BUTLEFT;
	else if (c1 == 'm' && ! c2)
	    button |= MS_BUTMIDDLE;
	else if (c1 == 'r' && ! c2)
	    button |= MS_BUTRIGHT;
	else
	{
	    fprintf(stderr, "%s: Error in button specification.\n", progname);
	    usage();
	}
	if (! c2)
	    break;
	arg += 2;
    }
    if (! button)
    {
	fprintf(stderr, "%s: Error in button specification.\n", progname);
	usage();
    }
    return button;
}

static inline int
cdiff(const Coord c1, const Coord c2)
{
    return ((c1.x + c1.y * win.ws_col) - (c2.x + c2.y * win.ws_col));
}

static void
run_test(void)
{
    struct ms_event ev;
    Coord prev, curr;
    static char *codestr[] = { "NONE", "BUTUP", "BUTDOWN", "MOVE", "DRAG" };
    static char *butstr[] = {
	"                 ",
	"            RIGHT",
	"     MIDDLE      ",
	"     MIDDLE RIGHT",
	"LEFT             ",
	"LEFT        RIGHT",
	"LEFT MIDDLE      ",
	"LEFT MIDDLE RIGHT" };

    fprintf(stderr, "\033[2J");

    if (ms_init(maccel, mbaud, mdelta, mdev, mtoggle, msample, mtype, mslack,
		win.ws_col * xscale - 1, win.ws_row * yscale - 1))
	exit(1);

    while (get_ms_event(&ev) == 0)
    {
    	fprintf(stderr, "\033[;H%03d,%03d %7s %17s", ev.ev_x, ev.ev_y,
	    codestr[(int)ev.ev_code], butstr[(int)ev.ev_butstate]);
	curr.x = ev.ev_x / xscale + 1;
	curr.y = ev.ev_y / yscale + 1;
	if (ev.ev_code == MS_MOVE)
	    fprintf(stderr, "\033[%d;%dH ", prev.y, prev.x);
	fprintf(stderr, "\033[%d;%dH%c", curr.y, curr.x, '0' + ev.ev_butstate);
	prev = curr;
    }
}
