/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.services;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.util.EntityUtils;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.api.metamodel.runtime.xml.XmlWriter;
import org.ovirt.engine.sdk4.Error;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.HostBuilder;
import org.ovirt.engine.sdk4.internal.HttpConnection;
import org.ovirt.engine.sdk4.internal.xml.XmlGlusterHookReader;
import org.ovirt.engine.sdk4.internal.xml.XmlHostWriter;
import org.ovirt.engine.sdk4.services.GlusterHookService;
import org.ovirt.engine.sdk4.types.Action;
import org.ovirt.engine.sdk4.types.GlusterHook;
import org.ovirt.engine.sdk4.types.Host;

public class GlusterHookServiceImpl extends ServiceImpl implements GlusterHookService {
    
    public GlusterHookServiceImpl(HttpConnection connection, String path) {
        super(connection, path);
    }
    
    private class DisableRequestImpl implements DisableRequest {
        private Boolean async;
        public DisableRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        public DisableResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/disable");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                xmlWriter.writeStartElement("action");
                if (async != null) {
                    xmlWriter.writeBoolean("async", async);
                }
                xmlWriter.writeEndElement();
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new DisableResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class DisableResponseImpl implements DisableResponse {
    }
    
    public DisableRequest disable() {
        return new DisableRequestImpl();
    }
    
    private class EnableRequestImpl implements EnableRequest {
        private Boolean async;
        public EnableRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        public EnableResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/enable");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                xmlWriter.writeStartElement("action");
                if (async != null) {
                    xmlWriter.writeBoolean("async", async);
                }
                xmlWriter.writeEndElement();
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new EnableResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class EnableResponseImpl implements EnableResponse {
    }
    
    public EnableRequest enable() {
        return new EnableRequestImpl();
    }
    
    private class GetRequestImpl implements GetRequest {
        public GetResponse send() {
            HttpGet request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpGet(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new GetResponseImpl(XmlGlusterHookReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class GetResponseImpl implements GetResponse {
        private GlusterHook hook;
        
        public GetResponseImpl(GlusterHook hook) {
            this.hook = hook;
        }
        
        public GlusterHook hook() {
            return hook;
        }
        
    }
    
    public GetRequest get() {
        return new GetRequestImpl();
    }
    
    private class RemoveRequestImpl implements RemoveRequest {
        private Boolean async;
        public RemoveRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        public RemoveResponse send() {
            HttpDelete request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                if (async != null) {
                    uriBuilder.addParameter("async", XmlWriter.renderBoolean(async));
                }
                request = new HttpDelete(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                EntityUtils.consumeQuietly(response.getEntity());
                return new RemoveResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class RemoveResponseImpl implements RemoveResponse {
    }
    
    public RemoveRequest remove() {
        return new RemoveRequestImpl();
    }
    
    private class ResolveRequestImpl implements ResolveRequest {
        private Boolean async;
        public ResolveRequest async(Boolean async) {
            this.async = async;
            return this;
        }
        
        private Host host;
        public ResolveRequest host(Host host) {
            this.host = host;
            return this;
        }
        
        public ResolveRequest host(HostBuilder host) {
            this.host = host.build();
            return this;
        }
        
        private String resolutionType;
        public ResolveRequest resolutionType(String resolutionType) {
            this.resolutionType = resolutionType;
            return this;
        }
        
        public ResolveResponse send() {
            HttpPost request = new HttpPost(getConnection().getUrl() + getPath() + "/resolve");
            try (
              ByteArrayOutputStream output = new ByteArrayOutputStream();
              XmlWriter xmlWriter = new XmlWriter(output, true)
            ) {
                xmlWriter.writeStartElement("action");
                if (host != null) {
                    XmlHostWriter.writeOne(host, xmlWriter);
                }
                if (resolutionType != null) {
                    xmlWriter.writeString("resolution_type", resolutionType);
                }
                if (async != null) {
                    xmlWriter.writeBoolean("async", async);
                }
                xmlWriter.writeEndElement();
                xmlWriter.flush();
                request.setEntity(new ByteArrayEntity(output.toByteArray()));
            }
            catch (IOException ex) {
                throw new Error("Failed to write request", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (response.getStatusLine().getStatusCode() == 200) {
                Action action = checkAction(response);
                EntityUtils.consumeQuietly(response.getEntity());
                return new ResolveResponseImpl();
            }
            else {
                checkFault(response);
                return null;
            }
        }
        
    }
    
    private class ResolveResponseImpl implements ResolveResponse {
    }
    
    public ResolveRequest resolve() {
        return new ResolveRequestImpl();
    }
    
    public Service service(String path) {
        if (path == null) {
            return this;
        }
        
        throw new Error("The path " + path + " doesn't correspond to any service");
    }
    
    public String toString() {
        return "GlusterHookService:" + getPath();
    }
    
}

