/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.xml;

import java.lang.String;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.ovirt.api.metamodel.runtime.util.ArrayListWithHref;
import org.ovirt.api.metamodel.runtime.util.ListWithHref;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.internal.containers.ActionContainer;
import org.ovirt.engine.sdk4.types.Action;

public class XmlActionReader {
    
    public static Action readOne(XmlReader reader) {
        // Do nothing if there aren't more tags:
        if (!reader.forward()) {
            return null;
        }
        
        // Create the object:
        ActionContainer object = new ActionContainer();
        
        // Process the attributes:
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String name = reader.getAttributeLocalName(i);
            String image = reader.getAttributeValue(i);
            switch (name) {
                case "href":
                object.href(image);
                break;
                case "id":
                object.id(image);
                break;
                default:
                break;
            }
        }
        
        // Process the inner elements:
        reader.next();
        while (reader.forward()) {
            String name = reader.getLocalName();
            switch (name) {
                case "async":
                object.async(reader.readBoolean());
                break;
                case "bricks":
                object.bricks(XmlGlusterBrickReader.readMany(reader));
                break;
                case "certificates":
                object.certificates(XmlCertificateReader.readMany(reader));
                break;
                case "check_connectivity":
                object.checkConnectivity(reader.readBoolean());
                break;
                case "clone":
                object.clone_(reader.readBoolean());
                break;
                case "cluster":
                object.cluster(XmlClusterReader.readOne(reader));
                break;
                case "collapse_snapshots":
                object.collapseSnapshots(reader.readBoolean());
                break;
                case "comment":
                object.comment(reader.readString());
                break;
                case "connectivity_timeout":
                object.connectivityTimeout(reader.readInteger());
                break;
                case "data_center":
                object.dataCenter(XmlDataCenterReader.readOne(reader));
                break;
                case "deploy_hosted_engine":
                object.deployHostedEngine(reader.readBoolean());
                break;
                case "description":
                object.description(reader.readString());
                break;
                case "details":
                object.details(XmlGlusterVolumeProfileDetailsReader.readOne(reader));
                break;
                case "discard_snapshots":
                object.discardSnapshots(reader.readBoolean());
                break;
                case "disk":
                object.disk(XmlDiskReader.readOne(reader));
                break;
                case "disks":
                object.disks(XmlDiskReader.readMany(reader));
                break;
                case "exclusive":
                object.exclusive(reader.readBoolean());
                break;
                case "exclussive":
                object.exclussive(reader.readBoolean());
                break;
                case "fault":
                object.fault(XmlFaultReader.readOne(reader));
                break;
                case "fence_type":
                object.fenceType(reader.readString());
                break;
                case "filter":
                object.filter(reader.readBoolean());
                break;
                case "fix_layout":
                object.fixLayout(reader.readBoolean());
                break;
                case "force":
                object.force(reader.readBoolean());
                break;
                case "grace_period":
                object.gracePeriod(XmlGracePeriodReader.readOne(reader));
                break;
                case "host":
                object.host(XmlHostReader.readOne(reader));
                break;
                case "image":
                object.image(reader.readString());
                break;
                case "import_as_template":
                object.importAsTemplate(reader.readBoolean());
                break;
                case "is_attached":
                object.isAttached(reader.readBoolean());
                break;
                case "iscsi":
                object.iscsi(XmlIscsiDetailsReader.readOne(reader));
                break;
                case "iscsi_targets":
                object.iscsiTargets(reader.readStrings());
                break;
                case "job":
                object.job(XmlJobReader.readOne(reader));
                break;
                case "logical_units":
                object.logicalUnits(XmlLogicalUnitReader.readMany(reader));
                break;
                case "maintenance_enabled":
                object.maintenanceEnabled(reader.readBoolean());
                break;
                case "modified_bonds":
                object.modifiedBonds(XmlHostNicReader.readMany(reader));
                break;
                case "modified_labels":
                object.modifiedLabels(XmlNetworkLabelReader.readMany(reader));
                break;
                case "modified_network_attachments":
                object.modifiedNetworkAttachments(XmlNetworkAttachmentReader.readMany(reader));
                break;
                case "name":
                object.name(reader.readString());
                break;
                case "option":
                object.option(XmlOptionReader.readOne(reader));
                break;
                case "pause":
                object.pause(reader.readBoolean());
                break;
                case "power_management":
                object.powerManagement(XmlPowerManagementReader.readOne(reader));
                break;
                case "proxy_ticket":
                object.proxyTicket(XmlProxyTicketReader.readOne(reader));
                break;
                case "reason":
                object.reason(reader.readString());
                break;
                case "removed_bonds":
                object.removedBonds(XmlHostNicReader.readMany(reader));
                break;
                case "removed_labels":
                object.removedLabels(XmlNetworkLabelReader.readMany(reader));
                break;
                case "removed_network_attachments":
                object.removedNetworkAttachments(XmlNetworkAttachmentReader.readMany(reader));
                break;
                case "resolution_type":
                object.resolutionType(reader.readString());
                break;
                case "restore_memory":
                object.restoreMemory(reader.readBoolean());
                break;
                case "root_password":
                object.rootPassword(reader.readString());
                break;
                case "snapshot":
                object.snapshot(XmlSnapshotReader.readOne(reader));
                break;
                case "ssh":
                object.ssh(XmlSshReader.readOne(reader));
                break;
                case "status":
                object.status(reader.readString());
                break;
                case "stop_gluster_service":
                object.stopGlusterService(reader.readBoolean());
                break;
                case "storage_domain":
                object.storageDomain(XmlStorageDomainReader.readOne(reader));
                break;
                case "storage_domains":
                object.storageDomains(XmlStorageDomainReader.readMany(reader));
                break;
                case "succeeded":
                object.succeeded(reader.readBoolean());
                break;
                case "synchronized_network_attachments":
                object.synchronizedNetworkAttachments(XmlNetworkAttachmentReader.readMany(reader));
                break;
                case "template":
                object.template(XmlTemplateReader.readOne(reader));
                break;
                case "ticket":
                object.ticket(XmlTicketReader.readOne(reader));
                break;
                case "undeploy_hosted_engine":
                object.undeployHostedEngine(reader.readBoolean());
                break;
                case "use_cloud_init":
                object.useCloudInit(reader.readBoolean());
                break;
                case "use_sysprep":
                object.useSysprep(reader.readBoolean());
                break;
                case "virtual_functions_configuration":
                object.virtualFunctionsConfiguration(XmlHostNicVirtualFunctionsConfigurationReader.readOne(reader));
                break;
                case "vm":
                object.vm(XmlVmReader.readOne(reader));
                break;
                case "link":
                readLink(reader, object);
                break;
                default:
                reader.skip();
                break;
            }
        }
        
        // Discard the end tag:
        reader.next();
        
        return object;
    }
    
    public static Iterator<Action> iterateMany(final XmlReader reader) {
        return new Iterator<Action>() {
            private boolean first = true;
            
            @Override
            public void remove() {
                // Empty on purpose
            }
            
            @Override
            public boolean hasNext() {
                if (first) {
                    if (!reader.forward()) {
                        return false;
                    }
                    reader.next();
                    first = false;
                }
                if (!reader.forward()) {
                    reader.next();
                    return false;
                }
                return true;
            }
            
            @Override
            public Action next() {
                Action next = readOne(reader);
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }
    
    public static List<Action> readMany(XmlReader reader) {
        List<Action> list = new ArrayList<>();
        Iterator<Action> iterator = iterateMany(reader);
        while (iterator.hasNext()) {
            list.add(iterator.next());
        }
        return list;
    }
    
    private static void readLink(XmlReader reader, ActionContainer object) {
        // Process the attributes:
        String rel = null;
        String href = null;
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String attrName = reader.getAttributeLocalName(i);
            String attrVal = reader.getAttributeValue(i);
            switch (attrName) {
                case "href":
                href = attrVal;
                break;
                case "rel":
                rel = attrVal;
                break;
                default:
                reader.skip();
                break;
            }
        }
        reader.skip();
    }
}

